;+
; Hydra/DDEIS TPP (Ten Parameter Patched Ions) Extraction Routine
; FILE: /home/ppq/idl_tools_public/release/extract_tpp.pro
; WRITTEN: Version 1.0, 06/11/98, Pamela A. Puhl-Quinn
; This routine extracts the TPP ion key parameters from CDF files
;   DEFAULT DIRECTORY: /opt/hydra/moment_data0/tppion/
;   DEFAULT NAME: YYYYMMDD_hyd_extra_v?.??.cdf
;
; USAGE:
;   extract_tpp,960529,0200,6,tpp
;
; MANDATORY ARGUMENTS:
;	1st argument(DATE):        YYMMDD or YYYYMMDD
;	2nd argument(START TIME):  'HH:MM:SS','HH:MM',HHMMSS,HHMM,HH.HHH,H
;	3rd argument(# HOURS):     # of hours you want extracted
;       4th argument:              Name of returned data structure
;
; OPTIONAL KEYWORDS:
;  indir='cdf_directory_path'    INPUT
;  extract_code_version = ecv    OUTPUT, vers. # of this extraction rtn
;  tpp_data_version = mdv        OUTPUT, vers. # of the tpp data extracted
;---------------------------------------------------------------------------
;-

PRO extract_tpp,date_in,time_in,hours_in,tpp,indir=indir,extract_code_version=extract_code_version, tpp_data_version=tpp_data_version

   IF (n_params() EQ 0) THEN BEGIN
      doc_library, 'extract_tpp'
      return
   ENDIF

; Define the version number for this extraction routine:
   extract_code_version = 1.0

; Convert the date and time string using Jeremy's tool----------------------
   hydra_time_io,date_in,time_in,iyear,imonth,iday,ihour,iminute,isecond
   date_str =  strmid(date_in,0,8)  
   hour = string(ihour,'(i2.2)')
   min = string(iminute,'(i2.2)')
   second = string(isecond,'(i2.2)')
   start_time = ihour*3600. + iminute*60. + isecond
   end_time =  start_time + hours_in*3600.

   myname= 'extract_tpp.pro'

   IF (keyword_set(indir)) THEN BEGIN
      
      f = hydra_findfile( date_str, /kp_extra, path=indir,$
                          version=version)
      tpp_data_version = version
      
   ENDIF ELSE BEGIN

      f = hydra_findfile( date_str, /kp_extra, version=version)
      tpp_data_version = version

   ENDELSE

   if (f NE '') then begin
      print, myname+': opening '+f+'...'
      cdf = cdf_open(f)
   endif else begin
      return
   ENDELSE

; Extract the variables

   cdf_control, cdf, variable='time', get_var_info=r
   IF r.maxrec EQ -1 THEN BEGIN
      print, myname+': No data found in CDF'
      stop
   ENDIF

   cdf_varget, cdf, 'time', time, rec_count=r.maxrec+1
   rec_start= ( where( time ge start_time ) )(0)
   if rec_start eq -1 then begin
      print, myname+': No data found after start_time, resetting '
      rec_start = 0
   endif
   rec_count= n_elements( where( time(rec_start:*) lt end_time $
                                 and time(rec_start:*) gt start_time, $
                                 count ))
   
   if ( count ) eq 0 then begin
      print, myname+': No data found in time interval specified'
      return
   endif      
   
   data_values = { tpp_dum, $
                   time:0.0, $
                   density:0.0, $
                   u:fltarr(3), $
                   tparl:0.0, $
                   tperp:0.0, $
                   gyrotropy:0.0, $
                   rchi2:0.0, $
                   rstudy_den:fltarr(6), $
                   iretu:0 $
                 }   
   
   tpp = replicate( data_values, rec_count )
   
   cdf_varget, cdf, 'time',time, $
    rec_start=rec_start, rec_count=rec_count

   cdf_varget, cdf, 'h_den',density, $
    rec_start=rec_start, rec_count=rec_count

   cdf_varget, cdf, 'h_u',u, $
    rec_start=rec_start, rec_count=rec_count

   cdf_varget, cdf, 'h_tparl',tparl, $
    rec_start=rec_start, rec_count=rec_count

   cdf_varget, cdf, 'h_tperp',tperp, $
    rec_start=rec_start, rec_count=rec_count

   cdf_varget, cdf, 'h_gyrotropy',gyrotropy, $
    rec_start=rec_start, rec_count=rec_count

   cdf_varget, cdf, 'h_rchi2',rchi2, $
    rec_start=rec_start, rec_count=rec_count

   cdf_varget, cdf, 'rstudy_den_stats',rstudy_den, $
    rec_start=rec_start, rec_count=rec_count

   cdf_varget, cdf, 'h_iretu',iretu, $
    rec_start=rec_start, rec_count=rec_count

   cdf_close,cdf
   
; One-dimensional arrays
   tpp.time = reform(time)
   tpp.density = reform(density)
   tpp.tparl = reform(tparl)
   tpp.tperp = reform(tperp)
   tpp.gyrotropy = reform(gyrotropy)
   tpp.rchi2 = reform(rchi2)
   tpp.iretu = reform(iretu)

; Arrays of more than one dimension
   tpp.u =  u
   tpp.rstudy_den = rstudy_den

   return 

END
