; Given normalized coordinates, return the coordinates on the
; current plot's axes.  The motivation for this procedure was to
; make a way that annotations could be printed within a plot having
; unknown axes.  We might be missing the simple and obvious way to 
; do this, and if you know it please call us.  
;
; Written by: Jeremy Faden, University of Iowa. 
;             jbf@space-theory.physics.uiowa.edu

function norm_axis, n_coord
  return=fltarr(2)
  return(0)=!x.crange(0)+(!x.crange(1)-!x.crange(0))*n_coord(0)
  return(1)=!y.crange(0)+(!y.crange(1)-!y.crange(0))*n_coord(1)
  
  return, return
end
;=====================================================================
;+
; NAME: faverage.pro
;
; AUTHOR: Pamela A. Puhl-Quinn, ppq@space-theory.physics.uiowa.edu
;
; HISTORY:
; 1st version: 	8/30/96 Written, Pamela Puhl-Quinn
; Recompiled, 10/24/96, ppq
; Recompiled, 12/22/96, ppq  b1_42
; Recompiled, 1/6/97, ppq b1_43
; Recompiled, 3/3/97, ppq b1_43
; Recompiled, 3/7/97, ppq b1_44
; Recompiled, 3/26/97, ppq b1_45
; Level 1 read standard, 4/7/97, ppq
; Combined f_average_plots and f_pitch, 4/8/97, ppq
; Recompiled, 7/28/97, ppq b1_50 - level0 is still usable
; Rewrite using the new calibration variables, level0 not
;    usable,9/8/97, ppq
; Fixed error in faverage_l1.pro involving sorting the arrays, also
;    confirmed the meaning of the one-cnt levels, 10/20/97, ppq
;    (NOTE: one-cnt level is calculated with v_obs not v_deb!!)
; Assigned versions, faverage and faverage_l1 versions must match,
;    added noshift option to ALL plots (not just fplots_en), added usage
;    help when no arguments are given, 10/21/97,ppq
; Assign version for the tar/gz files on the Web, 4/6/98:  v3.00
;
; USAGE:                             
;    faverage, 960713,1024,10,/fplots_en,pa_range=[0,30]
;
; MANDATORY ARGUMENTS:
;		1st argument:  YYMMDD or YYYYMMDD
;		2nd argument: 'HH:MM:SS','HH:MM',HHMMSS,HHMM,HH.HHH,H
;		3rd argument:  # of minutes
;
; MANDATORY KEYWORDS (only one at a time):
;   /fplots_en		; log<f> vs. energy, nine plots per page
;   /cplots_en		; log<C> vs. energy, nine plots per page
;   /defplots_en        ; Differential energy flux vs. energy, 9 plots/page
;   /dfplots_en         ; Differential flux vs. energy, 9 plots/page
;
; OPTIONAL KEYWORDS:
;
; Data filters
;   pa_range=[min,max]  ; Pitch-angle filter (in degrees)
;		(default is all pitch angles)
;   /alias_tog		; Split the distribution into 1/2 block chunks
;			; but plot both halves TOGether on the same graph
;		(default is to plot the whole block of data)
;   /alias_sep		; Split the distribution into 1/2 block chunks
;			; and plot each half SEPerately
;		(default is to plot the whole block of data)
;
; Output devices
;   /display		; display on screen (default)
;   /print		; create a post-script file AND print at UIowa
;   /ps			; create a post-script file, idl.ps
;   /gif		; create a gif file, idl.gif
;   nwindow=number	; designate an IDL window number other than 0
;
; Species
;   /ions		; plot ions 
;   /electrons		; plot electrons (default)
;
; Energy range
;   e_range=[min,max] 	; Energy range to be plotted in eV
;		(default is whole energy range)
;   /high		; Energy range from 0 to 1000eV, linear scale
;   /noshift		; Don't shift the measurements by the SC
;                         potential, denote location of SC potential
;                         with a dashed line
;
; Input directory path for the level1 data
;   indir='input_directory_path_string'
;
; NOTES:
; For the /*plot*_* plots, viewing the next page of graphs 
; is done by hitting any key
;
; KEEPING THE DATA SET:
; If you want to use the data set for other purposes, you can easily make
; an IDL save set by using the keyword /keep_data.  The file created is
; 'idlsave.dat' and can be restored within your own procedure by typing:
;		restore, 'idlsave.dat'
; The variable saved is l1_data, as described below.
;
; l1_data is an array of structures defined as such: 
; l1_data = replicate(data,nblocks) where nblocks is the number of blocks
; that were extracted, and data is a structure of the form: 
;   data = {data, $
;	species:fix(0),$		; -1 for ele, 1 for ion
;	mode:fix(0),$			; Block mode
;	sc_potential:float(0.),$	; As reported by EFI, in eV
;	sc_potential_stat:long(0),$	; 0=unknown, 1=EFI, 2=EFI+2eV
;	min_index:long(0),$		; IDL index of first positive
;					; energy level, whole block
;	min_index1:long(0),$		; IDL index of first positive
;					; energy level, 1st half of blk
;	min_index2:long(0),$		; IDL index of first positive
;					; energy level, 2nd half of blk
;	npts:long(0),$			; # of energy levels, whole blk
;	npts_ta:long(0),$		; # of energy levels, 1/2 blk
;	time:double(0.),$		; Block time in seconds
;	count:lonarr(110),$		; # of measmnts comprising faverage(i)
;	count1:lonarr(110),$		; # of measmnts comprising faverage1(i)
;	count2:lonarr(110),$		; # of measmnts comprising faverage2(i)
;	count_all:lonarr(110),$		; Total # measmnts in energy(i) shell
;	count1_all:lonarr(110),$	; Total # measmnts in energy1(i) shell
;	count2_all:lonarr(110),$	; Total # measmnts in energy2(i) shell
;	faverage:dblarr(110),$		; Spin-averaged dist. fnc.,whole block
;	faverage1:dblarr(110),$		; Spin-averaged dist. fnc., 1st half
;	faverage2:dblarr(110),$		; Spin-averaged dist. fnc., 2nd half
;	caverage:dblarr(110),$		; Spin-averaged corr. cnts, whole
;	caverage1:dblarr(110),$		; Spin-averaged corr. cnts, 1st half
;	caverage2:dblarr(110),$		; Spin-averaged corr. cnts, 2nd half
;	energy:dblarr(110),$		; S/C corrected energy, whole block
;	energy1:dblarr(110),$		; S/C corrected energy, 1st half
;	energy2:dblarr(110),$		; S/C corrected energy, 2nd half
;	onef:dblarr(110),$		; One-count level for faverage
;	onef1:dblarr(110),$		; One-count level for faverage1
;	onef2:dblarr(110),$		; One-count level for faverage2
;	onec:dblarr(110),$		; One-count level for caverage
;	onec1:dblarr(110),$		; One-count level for caverage1
;	onec2:dblarr(110)}		; One-count level for caverage2
;
; So, for example, the block times are stored in l1_data(*).time, the 
; <f> array for the first time is stored in l1_data(0).faverage, the energy
; array associated with this <f> is stored in l1_data(0).energy, etc. 
; You don't need to define this structure explicitly in your own IDL
; procedure.  Just restore it and use it as you like.
;
;-
;=====================================================================
pro faverage, date_in, time_in, nmin,$
		pa_range=pa_range,e_range=e_range,$
		nwindow=nwindow,$ 
		display=display, $
               	ions=ions,electrons=electrons,print=print,ps=ps,$
		fplots_en=fplots_en,cplots_en=cplots_en, $
              defplots_en=defplots_en, dfplots_en=dfplots_en, $
		high=high,gif=gif,$
		alias_tog=alias_tog,alias_sep=alias_sep,$
		keep_data=keep_data,noshift=noshift,$
		indir=indir

common faverage_blocks, cdate,ctime,cnmin,cpa_range,cspecies,cindir,$
		l1_data,mode_info

; Usage help
  IF (n_params() EQ 0) THEN BEGIN
     doc_library, 'faverage'
     return
  ENDIF

; Set the version number for this software, which also must be set in
; faverage_l1.pro in order for them to work together.
  version =  3.0

; Setting/Error handling of the species keywords
if (keyword_set(ions)) then begin
   species='ions'
   mass = 1.67e-24
endif else begin
   species='elec'
   mass = 9.1e-28
endelse

; Setting/Error handling of the pitch_angle keywords
if (keyword_set(pa_range)) then begin
   pa_min = strtrim(pa_range(0),2)
   pa_max = strtrim(pa_range(1),2)
endif else begin
   pa_min = '0'
   pa_max = '180'
   pa_range = [0,180]
endelse

; Setting/Error handling of the print/display/ps keywords
if (keyword_set(print) and keyword_set(display)) then begin
   message, 'Both display and print keywords set; exiting',/cont 
   goto, end_flag
endif $
else if (keyword_set(ps) and keyword_set(display)) then begin
   message, 'Both display and ps keywords set; exiting',/cont
   goto, end_flag
endif $
else if (keyword_set(ps) and keyword_set(print)) then begin
   message, 'Both ps and print keywords set; exiting',/cont
   goto, end_flag
endif 
if (keyword_set(print)) then print=1 else print = 0 
if (keyword_set(ps)) then ps=1 else ps=0
if (keyword_set(gif)) then gif=1 else gif=0

if (keyword_set(indir)) then indir_in=indir else indir_in='nada'

if (keyword_set(nwindow)) then lwindow = nwindow else lwindow = 0

; Convert the date and time string using Jeremy's tool----------------------
hydra_time_io,date_in,time_in,fyear,fmonth,fday,fhour,fminute,fsecond
date =  strmid(date_in,2,6)  
hour = string(fhour,'(i2.2)')
min = string(fminute,'(i2.2)')
second = string(fsecond,'(i2.2)')
timef =  hour+':'+min+':'+second
;--------------------------------------

   time = hour + min + second
   filename = date+'.'+time+'.dat'
   datestr=strmid(date,0,2)+'/'+strmid(date,2,2)+'/'+strmid(date,4,2)
   nmin = strtrim(nmin,1)

   message, 'date = '+date, /cont
   message, 'time = '+timef, /cont
   message, 'nmin = '+nmin, /cont
   message, 'species = '+species, /cont
   message, 'pitch angles: '+pa_min+'-'+pa_max,/cont

; Check to see if l1_data already exists in the buffer for these specs, 
; and if not, then the data has to be read

   if (keyword_set(cdate)) then begin  

      if (cdate eq date and ctime eq timef and cnmin eq nmin $
          and cspecies eq species and cpa_range(0) eq pa_range(0) and $
          cpa_range(1) eq pa_range(1) and cindir eq $
	  indir_in and not keyword_set(level0)) then begin
         read=0			; data already in buffer
         message, 'Data already in buffer...' ,/cont	
      endif else IF (NOT keyword_set(level0)) THEN begin
         read=1			; data needs to be read
         message, 'Reading level 1 data...',/cont
      ENDIF ELSE BEGIN
         read = 1
         message, 'Reading level 0 data...', /cont
      ENDELSE

   endif else begin
      read=1                    ; data needs to be read
      message, 'Reading level 1 data...', /cont
   endelse

;**********************READ DATA FROM LEVEL 1***********************

if (read eq 1) then begin

   faverage_l1,date,timef,nmin,pa_range,species,indir_in,l1_data,mode_info,version

	; Set the value of cdate in the buffer in order to access the
	; buffered data on subsequent calls to this procedure without having
	; to read the data in from a file every time
	message, 'Set buffer variables...', /cont
	cdate = date
	ctime = timef
	cnmin = nmin
	cpa_range = intarr(2)
	cpa_range = pa_range
	cspecies = species
	cindir = indir_in
endif

; Did we encounter more than one block mode?
; This will affect the sizes of the arrays below

j_energy=l1_data(0).npts
j_energy_ta = l1_data(0).npts_ta
j_block=n_elements(l1_data)

; Declare array dimension/types
minimum_index = intarr(j_block)
minimum_index1 = intarr(j_block)
minimum_index2 = intarr(j_block)
sc_potential = fltarr(j_block)
sc_pot_status = intarr(j_block)
mode = intarr(j_block)
y_con   = dblarr(j_block)
x_con   = dblarr(j_energy,j_block)
z_con   = dblarr(j_energy,j_block)
x_con_2 = dblarr(j_energy,j_block)
z_con_2 = dblarr(j_energy,j_block)
one_cnt_f = dblarr(j_energy,j_block)
one_cnt_c = dblarr(j_energy,j_block)
f_1 = dblarr(j_energy_ta,j_block)
f_2 = dblarr(j_energy_ta,j_block)
c_1 = dblarr(j_energy_ta,j_block)
c_2 = dblarr(j_energy_ta,j_block)
en_1 = dblarr(j_energy_ta,j_block)
en_2 = dblarr(j_energy_ta,j_block)
onef1 = dblarr(j_energy_ta,j_block)
onef2 = dblarr(j_energy_ta,j_block)
onec1 = dblarr(j_energy_ta,j_block)
onec2 = dblarr(j_energy_ta,j_block)

; Define the arrays
minimum_index = l1_data.min_index
minimum_index1 = l1_data.min_index1
minimum_index2 = l1_data.min_index2
sc_potential = l1_data.sc_potential
sc_pot_status = l1_data.sc_potential_stat
mode = l1_data.mode

; Time
y_con=l1_data.time/3600.

; Energy
x_con = l1_data.energy(0:j_energy-1,*)
en_1 = l1_data.energy1(0:j_energy_ta-1,*)
en_2 = l1_data.energy2(0:j_energy_ta-1,*)

; Distribution function
z_con = l1_data.faverage(0:j_energy-1,*)
one_cnt_f = l1_data.onef(0:j_energy-1,*)
f_1 = l1_data.faverage1(0:j_energy_ta-1,*)
f_2 = l1_data.faverage2(0:j_energy_ta-1,*)
onef1 = l1_data.onef1(0:j_energy_ta-1,*)
onef2 = l1_data.onef2(0:j_energy_ta-1,*)

; Counts
z_con_2 = l1_data.caverage(0:j_energy-1,*)
one_cnt_c = l1_data.onec(0:j_energy-1,*)
c_1 = l1_data.caverage1(0:j_energy_ta-1,*)
c_2 = l1_data.caverage2(0:j_energy_ta-1,*)
onec1 = l1_data.onec1(0:j_energy_ta-1,*)
onec2 = l1_data.onec2(0:j_energy_ta-1,*)

if (keyword_set(defplots_en)) then begin ; Diff. energy flux
    z_con_2 = z_con_2*187970.
    one_cnt_c = one_cnt_c*187970.
    c_1 = c_1*187970.
    c_2 = c_2*187970.
    onec1 = onec1*187970.
    onec2 = onec2*187970.
endif else if (keyword_set(dfplots_en)) then begin ; Diff. flux
    z_con_2 = z_con_2*187970./(x_con/1000.)
    one_cnt_c = one_cnt_c*187970./(x_con/1000.)
    c_1 = c_1*187970./(en_1/1000.)
    c_2 = c_2*187970./(en_2/1000.)
    onec1 = onec1*187970./(en_1/1000.)
    onec2 = onec2*187970./(en_2/1000.)
endif

; Set floor and max values for both f and counts, and set zero count values
if (species eq 'elec') then begin
; Distribution function
    floor_f = 1.d-35
    max_f = 1.d-24
    zero_f  = 1.d-36
; Counts
    floor_c = 1.d-2
    max_c = 1.d+4
    zero_c  = 1.d-3
endif else begin
; Distribution function
    floor_f = 1.d-30
    max_f = 1.d-17
    zero_f  = 1.d-31
; Counts
    floor_c = 1.d-2
    max_c = 1.d+3
    zero_c  = 1.d-3
endelse

if (keyword_set(defplots_en)) then begin
    floor_c = floor_c*187970.
    max_c = max_c*187970.
    zero_c = zero_c*187970.
endif else if (keyword_set(dfplots_en)) then begin
    min_energy = min(x_con(where(x_con ge 0.))/1000.)
    max_energy = max(x_con(where(x_con ge 0.))/1000.)
    floor_c = floor_c*187970./max_energy
    max_c = max_c*187970./min_energy
    zero_c = zero_c*187970./max_energy
endif

; Handle zero counts (set equal to zero_f instead because of log graphs)

zeros = where(z_con eq 0)
if (zeros(0) ne -1) then begin
	z_con(zeros) = zero_f
	z_con_2(zeros) = zero_c
endif

zeros = where(f_1 eq 0)
if (zeros(0) ne -1) then begin
	f_1(zeros) = zero_f
	c_1(zeros) = zero_c
endif

zeros = where(f_2 eq 0)
if (zeros(0) ne -1) then begin
	f_2(zeros) = zero_f
	c_2(zeros) = zero_c
endif 

; Convert energy (x_con) to speed (x_con_2)
ge_zero = where(x_con ge 0)
lt_zero = where(x_con lt 0)
x_con_2(ge_zero)= sqrt(2.*x_con(ge_zero)*1.6d-12/mass)*1.e-5
if (lt_zero(0) ne -1) then x_con_2(lt_zero)=-1.   ;Values not plotted


; Determine labels for the graphs
   sc_stat_str = strarr(j_block)
   sc_pot_str = strarr(j_block)
   x_title_str_en = strarr(j_block)
   x_title_str_sp = strarr(j_block)
   for i=0,j_block-1 do begin
	if (sc_pot_status(i) eq 0) then begin
		sc_stat_str(i) = '(Unknown)'
		x_title_str_en(i) = 'Observed Energy'
                x_title_str_sp(i) = 'Observed Speed'
		sc_pot_str(i) = ' '
	endif else if (sc_pot_status(i) eq 1) then begin
		sc_stat_str(i) = '(EFI exact)'
		if (keyword_set(noshift)) then begin
 		   x_title_str_en(i) = 'Observed Energy'
                   x_title_str_sp(i) = 'Observed Energy'
		endif else begin
 		   x_title_str_en(i) = 'Debye Sphere Energy'
                   x_title_str_sp(i) = 'Debye Sphere Speed'
		endelse
	sc_pot_str(i) = string(sc_potential(i),format='(f5.1)')+'V'
	endif else if (sc_pot_status(i) eq 2) then begin
		sc_stat_str(i) = '(EFI + 2V)'
		if (keyword_set(noshift)) then begin
 		   x_title_str_en(i) = 'Observed Energy'
                   x_title_str_sp(i) = 'Observed Energy'
		endif else begin
 		   x_title_str_en(i) = 'Debye Sphere Energy'
                   x_title_str_sp(i) = 'Debye Sphere Speed'
		endelse
	sc_pot_str(i) = string(sc_potential(i),format='(f5.1)')+'V'
	endif
   endfor

   if (keyword_set(fplots_en)) then ytitle='!5<f> [cm!u-6!ns!u3!n]' else if $
     (keyword_set(cplots_en)) then ytitle='!5<C>' else if $
     (keyword_set(defplots_en)) then ytitle='!5(cm!e2!n-s-sr-!4D!xE/E)!e-1!n' $
     else if (keyword_set(dfplots_en)) then ytitle='!5(cm!e2!n-s-sr-!4D!xE/E-keV)!e-1!n' else message, 'Must specify either fplots_en, cplots_en, defplots_en, or dfplots_en'

; Conform with level0
j_block = j_block-1

;*********************************END READ*************************
end_read:

slab = ' Version 1, l1lib_ver= '+strtrim(mode_info.footprint.l1lib_version,2) $
	+ ' ,l1_ver= '+ strtrim(mode_info.footprint.l1_version,2) $
	+ ' ,hydlib_ver= '+ strtrim(mode_info.footprint.hydralib_version,2)


if (keyword_set(keep_data)) then begin
	save, l1_data
	message, 'Saving data set: idlsave.dat', /cont
endif

; Set character size for display (which changes for print/ps)
   outs_size = 1
   plt_size = 1.75

if (print EQ 1 or ps EQ 1) then begin
   set_ps, orient='L'
   outs_size=.6
   plt_size = 1.25
endif

!p.background = 234
!p.color = 0
!p.font = -1

if (keyword_set(defplots_en) or keyword_set(dfplots_en)) then cplots_en=1
   
;======================PLOTTING STATMENTS===================
if (keyword_set(fplots_en) and not keyword_set(alias_sep) and $
	not keyword_set(alias_tog)) then begin
   print, '	'
   message, 'plotting <f> time-series...',/cont
   if (print eq 0 and ps eq 0) then $
     window,lwindow,xsize=950,ysize=850,title='Press any key to see next page'

   set_page, nh=3, nv=3

   if (species eq 'ions') then charge=1 else charge=-1

   en_temp = fltarr(j_energy,j_block+1)
   scpot = fltarr(j_block+1)
   ; Determine which energy you want to use (shifted or not by the SC pot)
   if (keyword_set(noshift)) then begin
	for ii = 0, j_block do begin
	   if (sc_pot_status(ii) ne 0) then begin
	      en_temp(*,ii) = x_con(*,ii) - charge*sc_potential(ii)
	   endif else begin
	      en_temp(*,ii) = x_con(*,ii)
	   endelse
	endfor
   endif else begin
	en_temp = x_con
   endelse

   ; Set x-axis range and unit, and the y-axis range
   if (keyword_set(high)) then begin
      max_x = 1000.
      min_x = 0.  
      x_unit_str = ' [eV]'
      scpot = sc_potential
   endif else if (keyword_set(e_range)) then begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = e_range(1)*1.e-3		;keV
      min_x = e_range(0)*1.e-3		;keV
      x_unit_str = ' [keV]'
   endif else begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = max(en_temp(*,*))
      min_x = min(en_temp(where(en_temp(*,*) gt 0)))
      x_unit_str = ' [keV]'
   endelse
   min_y = floor_f
   max_y =max_f

;  Create a plotting keyword structure

if (keyword_set(high)) then begin
      kw = {ylog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,ytickformat:'(e8.1)',$
        ytitle:ytitle,charsize:plt_size,$
	psym:-2,symsize:.6}
endif else begin
      kw = {ylog:1,xlog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,ytickformat:'(e8.1)',$
        ytitle:ytitle,charsize:plt_size,$
	psym:-2,symsize:.6}
endelse


;  Dimension the xyouts annotation position vectors
	outs1 = fltarr(2)
	outs2 = fltarr(2)
	outs3 = fltarr(2)
	outs4 = fltarr(2)
	outs5 = fltarr(2)
	outs_eol = float(0.)

   for i=0,j_block do begin
      pos_pos=where(en_temp(*,i) gt 0 and en_temp(*,i) le max_x)
      plot, 	en_temp(pos_pos,i),z_con(pos_pos,i),$
      		_extra = kw,$	
      		xtitle = x_title_str_en(i) + x_unit_str

      oplot,en_temp(pos_pos,i),one_cnt_f(pos_pos,i),line=1

;      delta =  0.d0
;      count =  l1_data.count(0:j_energy-1,*)
;      delta = double(count(pos_pos,i)) * mode_info.factor
;      e_old = en_temp(pos_pos,i)*1.e+3 + sc_potential(i)
;      v_old = sqrt(2*e_old*1.6d-12/9.1d-28)
;      v4 =  v_old^4
;      onef_me = 1./(delta*v4)      
;      print, 'time: ',y_con(i)*3600.d0
;      print, 'level: ',pos_pos(0)
;      print, 'N: ',double(count(pos_pos(0),i))
;      print, 'factor: ',mode_info.factor
;      print, 'shifted energy: ',en_temp(pos_pos(0),i)*1.e+3
;      print, 'obs energy: ', e_old(0)
;      print, 'sc_potential: ',sc_potential(i)
;      print, 'sc_potential_status: ',sc_pot_status(i)
;      print, 'obs velocity: ',v_old(0),v4(0)
;      print, 'onef: ',onef_me(0)
;      oplot,en_temp(pos_pos,i),onef_me,line=0


      IF (keyword_set(noshift)) THEN BEGIN 
         oplot,[scpot(i),scpot(i)],[min_y,max_y],line=3
      ENDIF

;  Form the time string
      hour_block=long(y_con(i))
      minute_block=long((y_con(i)-long(y_con(i)))*60.)
      second_block=long(((y_con(i)-long(y_con(i)))*60.   $
		-minute_block)*60.)

      hour_string=strtrim(hour_block,1)
        if (strlen(hour_string) eq 1) then  $
			hour_string='0'+hour_string
      minute_string=strtrim(minute_block,1)
        if (strlen(minute_string) eq 1) then $
			minute_string='0'+minute_string
      second_string=strtrim(second_block,1)
        if (strlen(second_string) eq 1) then $
			second_string='0'+second_string

      timestring=hour_string+':'+minute_string+':'+second_string

;  Annotation:
;  Set the normalized position vectors for the xyouts annotation
;  (This only needs to be done once, but must be done after the
;  first plot call so that !x.crange and !y.crange are correct)

      if (i eq 0) then begin
	   if (keyword_set(high)) then begin
	      outs1 = norm_axis([.5,.9])
			outs1(1) = 10.^outs1(1)
	      outs2 = norm_axis([.5,.85])
			outs2(1) = 10.^outs2(1)
	      outs3 = norm_axis([.5,.8])
			outs3(1) = 10.^outs3(1)
	      outs4 = norm_axis([.5,.75])
			outs4(1) = 10.^outs4(1)
	      outs5 = norm_axis([.5,.7])
	           	outs_eol = .6*max_x
			outs5(1) = 10.^outs5(1)
	   endif else begin
	      outs1 = norm_axis([.5,.9])
			outs1 = 10.^outs1
	      outs2 = norm_axis([.5,.85])
			outs2 = 10.^outs2
	      outs3 = norm_axis([.5,.8])
			outs3 = 10.^outs3
	      outs4 = norm_axis([.5,.75])
			outs4 = 10.^outs4
	      outs5 = norm_axis([.5,.7])
	                outs_eol = 10.^(outs5(0)+.6)
			outs5 = 10.^outs5
	   endelse
      endif 
      xyouts, outs1(0), outs1(1), $
		datestr+' , '+species,charsize=outs_size
      xyouts, outs2(0), outs2(1), $
		'UT='+timestring, $
		charsize=outs_size
      xyouts, outs3(0), outs3(1), $
		'!7U!5!dsc!n='+sc_pot_str(i)+'  '+sc_stat_str(i),$
		charsize=outs_size

      oplot, [outs4(0),outs_eol],[outs4(1),outs4(1)], line=1
      xyouts, outs_eol, outs4(1), $
		'1 count in shell',charsize=outs_size

      xyouts, outs5(0), outs5(1), $
		'*** Whole block',charsize=outs_size

      if (!p.multi(0) eq 0 and print eq 0 or i eq j_block) then begin
	label_page,side_label=slab
	message, 'Press any key to see next page',/cont
        res = get_kbrd(1)
      endif

   endfor

; Change en_temp back to eV in case used elsewhere
if (not keyword_set(high)) then $
  en_temp(*,*) = en_temp(*,*)*1.d+3

endif
;===========================================================
if (keyword_set(alias_tog) and keyword_set(fplots_en)) then begin
   message, 'plotting <f> time-series...', /cont
   if (print eq 0 and ps eq 0) then $
     window,lwindow,xsize=950,ysize=850,title='Press any key to see next page'

   set_page, nh=3, nv=3

   if (species eq 'ions') then charge=1 else charge=-1

   en_temp = fltarr(j_energy,j_block+1)
   en_temp1 = fltarr(j_energy,j_block+1)
   en_temp2 = fltarr(j_energy,j_block+1)
   scpot = fltarr(j_block+1)
   ; Determine which energy you want to use (shifted or not by the SC pot)
   if (keyword_set(noshift)) then begin
	for ii = 0, j_block do begin
	   if (sc_pot_status(ii) ne 0) then begin
	      en_temp(*,ii) = x_con(*,ii) - charge*sc_potential(ii)
	      en_temp1(*,ii) = en_1(*,ii) - charge*sc_potential(ii)
	      en_temp2(*,ii) = en_2(*,ii) - charge*sc_potential(ii)
	   endif else begin
	      en_temp(*,ii) = x_con(*,ii)
	      en_temp1(*,ii) = en_1(*,ii)
	      en_temp2(*,ii) = en_2(*,ii)
	   endelse
	endfor
   endif else begin
	en_temp = x_con
        en_temp1 = en_1
        en_temp2 = en_2
   ENDELSE

   ; Set x-axis range and unit, and the y-axis range
   if (keyword_set(high)) then begin
      max_x = 1000.
      min_x = 0.  
      x_unit_str = ' [eV]'
      scpot = sc_potential
   endif else if (keyword_set(e_range)) then begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      en_temp1(*,*) = en_temp1(*,*)*1.d-3     ;eV to keV
      en_temp2(*,*) = en_temp2(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = e_range(1)*1.e-3		;keV
      min_x = e_range(0)*1.e-3		;keV
      x_unit_str = ' [keV]'
   endif else begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      en_temp1(*,*) = en_temp1(*,*)*1.d-3     ;eV to keV
      en_temp2(*,*) = en_temp2(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = max(en_temp(*,*))
      min_x = min(en_temp(where(en_temp(*,*) gt 0)))
      x_unit_str = ' [keV]'
   endelse

   min_y = floor_f
   max_y =max_f

;  Create a plotting keyword structure
   if (keyword_set(high)) then begin
      kw = {ylog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,ytickformat:'(e8.1)',line:3,$
        ytitle:ytitle,charsize:plt_size,$
	psym:-2,symsize:.6}
   endif else begin
      kw = {ylog:1,xlog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,ytickformat:'(e8.1)',line:3,$
        ytitle:ytitle,charsize:plt_size,$
	psym:-2,symsize:.6}
   endelse

;  Dimension the xyouts annotation position vectors
	outs1 = fltarr(2)
	outs2 = fltarr(2)
	outs3 = fltarr(2)
	outs4 = fltarr(2)
	outs5 = fltarr(2)
	outs6 = fltarr(2)
	outs_eol = float(0.)
  
   for i=0,j_block do begin
      pos_pos_1=where(en_temp1(*,i) gt 0 and en_temp1(*,i) le max_x)
      plot, 	en_temp1(pos_pos_1,i),f_1(pos_pos_1,i),$
		_extra = kw, $
      		xtitle = x_title_str_en(i) + x_unit_str

      pos_pos_2=where(en_temp2(*,i) gt 0 and en_temp2(*,i) le max_x)
      oplot,en_temp2(pos_pos_2,i),f_2(pos_pos_2,i),psym=-5,symsize=.6

      pos_pos_big = where(en_temp(*,i) gt 0 and en_temp(*,i) le max_x)
      oplot,en_temp(pos_pos_big,i),one_cnt_f(pos_pos_big,i),line=1

      IF (keyword_set(noshift)) THEN BEGIN 
         oplot,[scpot(i),scpot(i)],[min_y,max_y],line=3
      ENDIF

;  Form time string
      hour_block=long(y_con(i))
      minute_block=long((y_con(i)-long(y_con(i)))*60.)
      second_block=long(((y_con(i)-long(y_con(i)))*60.   $
		-minute_block)*60.)
      hour_string=strtrim(hour_block,1)
        if (strlen(hour_string) eq 1) then  $
			hour_string='0'+hour_string
      minute_string=strtrim(minute_block,1)
        if (strlen(minute_string) eq 1) then $
			minute_string='0'+minute_string
      second_string=strtrim(second_block,1)
        if (strlen(second_string) eq 1) then $
			second_string='0'+second_string

      timestring=hour_string+':'+minute_string+':'+second_string

;  Annotation:
;  Set the normalized position vectors for the xyouts annotation
;  (This only needs to be done once, but must be done after the
;  first plot call so that !x.crange and !y.crange are correct)

      if (i eq 0) then begin
	   if (keyword_set(high)) then begin
	      outs1 = norm_axis([.5,.9])
			outs1(1) = 10.^outs1(1)
	      outs2 = norm_axis([.5,.85])
			outs2(1) = 10.^outs2(1)
	      outs3 = norm_axis([.5,.8])
			outs3(1) = 10.^outs3(1)
	      outs4 = norm_axis([.5,.75])
			outs4(1) = 10.^outs4(1)
	      outs5 = norm_axis([.5,.7])
			outs5(1) = 10.^outs5(1)
	      outs6 = norm_axis([.5,.65])
			outs6(1) = 10.^outs6(1)
	      outs_eol = .6*max_x
	   endif else begin
	      outs1 = norm_axis([.5,.9])
			outs1 = 10.^outs1
	      outs2 = norm_axis([.5,.85])
			outs2 = 10.^outs2
	      outs3 = norm_axis([.5,.8])
			outs3 = 10.^outs3
	      outs4 = norm_axis([.5,.75])
			outs4 = 10.^outs4
	      outs5 = norm_axis([.5,.7])
	                outs_eol = 10.^(outs5(0)+.6)
			outs5 = 10.^outs5
	      outs6 = norm_axis([.5,.65])
			outs6 = 10.^outs6
	   endelse
      endif

      xyouts, outs1(0), outs1(1), $
		datestr+' , '+species,charsize=outs_size
      xyouts, outs2(0), outs2(1), $
		'UT='+timestring, $
		charsize=outs_size
      xyouts, outs3(0), outs3(1), $
		'!7U!5!dsc!n='+sc_pot_str(i)+'  '+sc_stat_str(i),$
		charsize=outs_size
      oplot, [outs4(0),outs_eol],[outs4(1),outs4(1)], line=1
      xyouts, outs_eol, outs4(1), $
		'1 count in shell',charsize=outs_size

      xyouts, outs5(0), outs5(1), $
		'***** 1st half of block',charsize=outs_size

      xyouts, outs6(0), outs6(1), $
		'!7DDDDD!6 2nd half of block',charsize=outs_size

      if (!p.multi(0) eq 0 and print eq 0 or i eq j_block) then begin
	label_page,side_label=slab
	message, 'Press any key to see next page', /cont
        res = get_kbrd(1)
      endif

   endfor

endif
;===========================================================
if (keyword_set(alias_sep) and keyword_set(fplots_en)) then begin
   message, 'plotting <f> time-series...',/cont
   if (print eq 0 and ps eq 0) then $
     window,lwindow,xsize=950,ysize=850,title='Press any key to see next page'

   set_page, nh=3, nv=3

   if (species eq 'ions') then charge=1 else charge=-1

   en_temp = fltarr(j_energy,j_block+1)
   en_temp1 = fltarr(j_energy,j_block+1)
   en_temp2 = fltarr(j_energy,j_block+1)
   scpot = fltarr(j_block+1)
   ; Determine which energy you want to use (shifted or not by the SC pot)
   if (keyword_set(noshift)) then begin
	for ii = 0, j_block do begin
	   if (sc_pot_status(ii) ne 0) then begin
	      en_temp(*,ii) = x_con(*,ii) - charge*sc_potential(ii)
	      en_temp1(*,ii) = en_1(*,ii) - charge*sc_potential(ii)
	      en_temp2(*,ii) = en_2(*,ii) - charge*sc_potential(ii)
	   endif else begin
	      en_temp(*,ii) = x_con(*,ii)
	      en_temp1(*,ii) = en_1(*,ii)
	      en_temp2(*,ii) = en_2(*,ii)
	   endelse
	endfor
   endif else begin
	en_temp = x_con
        en_temp1 = en_1
        en_temp2 = en_2
   ENDELSE

   ; Set x-axis range and unit, and the y-axis range
   if (keyword_set(high)) then begin
      max_x = 1000.
      min_x = 0.  
      x_unit_str = ' [eV]'
      scpot = sc_potential
   endif else if (keyword_set(e_range)) then begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      en_temp1(*,*) = en_temp1(*,*)*1.d-3     ;eV to keV
      en_temp2(*,*) = en_temp2(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = e_range(1)*1.e-3		;keV
      min_x = e_range(0)*1.e-3		;keV
      x_unit_str = ' [keV]'
   endif else begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      en_temp1(*,*) = en_temp1(*,*)*1.d-3     ;eV to keV
      en_temp2(*,*) = en_temp2(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = max(en_temp(*,*))
      min_x = min(en_temp(where(en_temp(*,*) gt 0)))
      x_unit_str = ' [keV]'
   endelse

   min_y = floor_f
   max_y =max_f

;  Create a plotting keyword structure
   if (keyword_set(high)) then begin
      kw = {ylog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,ytickformat:'(e8.1)',$
        ytitle:ytitle,charsize:plt_size}
   endif else begin
      kw = {ylog:1,xlog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,ytickformat:'(e8.1)',$
        ytitle:ytitle,charsize:plt_size}
   endelse

;  Dimension the xyouts annotation position vectors
	outs1 = fltarr(2)
	outs2 = fltarr(2)
	outs3 = fltarr(2)
	outs4 = fltarr(2)
	outs5 = fltarr(2)
	outs6 = fltarr(2)
	outs_eol = float(0.)

   for i=0,j_block do begin

;====1st half plot=========
      pos_pos_1=where(en_temp1(*,i) gt 0 and en_temp1(*,i) le max_x)
      plot, 	en_temp1(pos_pos_1,i),f_1(pos_pos_1,i),$
		psym=-2,symsize=.6,$
		_extra=kw,$
      		xtitle = x_title_str_en(i) + x_unit_str	

      oplot,en_temp1(pos_pos_1,i),onef1(pos_pos_1,i),line=1

      IF (keyword_set(noshift)) THEN BEGIN 
         oplot,[scpot(i),scpot(i)],[min_y,max_y],line=3
      ENDIF

;  Form the time string
      hour_block=long(y_con(i))
      minute_block=long((y_con(i)-long(y_con(i)))*60.)
      second_block=long(((y_con(i)-long(y_con(i)))*60.   $
		-minute_block)*60.)
      hour_string=strtrim(hour_block,1)
        if (strlen(hour_string) eq 1) then  $
			hour_string='0'+hour_string
      minute_string=strtrim(minute_block,1)
        if (strlen(minute_string) eq 1) then $
			minute_string='0'+minute_string
      second_string=strtrim(second_block,1)
        if (strlen(second_string) eq 1) then $
			second_string='0'+second_string

      timestring=hour_string+':'+minute_string+':'+second_string

;  Annotation:
;  Set the normalized position vectors for the xyouts annotation
;  (This only needs to be done once, but must be done after the
;  first plot call so that !x.crange and !y.crange are correct)

      if (i eq 0) then begin
	   if (keyword_set(high)) then begin
	      outs1 = norm_axis([.5,.9])
			outs1(1) = 10.^outs1(1)
	      outs2 = norm_axis([.5,.85])
			outs2(1) = 10.^outs2(1)
	      outs3 = norm_axis([.5,.8])
			outs3(1) = 10.^outs3(1)
	      outs4 = norm_axis([.5,.75])
			outs4(1) = 10.^outs4(1)
	      outs5 = norm_axis([.5,.7])
			outs5(1) = 10.^outs5(1)
	      outs6 = norm_axis([.5,.65])
			outs6(1) = 10.^outs6(1)
	      outs_eol = .6*max_x
	   endif else begin
	      outs1 = norm_axis([.5,.9])
			outs1 = 10.^outs1
	      outs2 = norm_axis([.5,.85])
			outs2 = 10.^outs2
	      outs3 = norm_axis([.5,.8])
			outs3 = 10.^outs3
	      outs4 = norm_axis([.5,.75])
			outs4 = 10.^outs4
	      outs5 = norm_axis([.5,.7])
	                outs_eol = 10.^(outs5(0)+.6)
			outs5 = 10.^outs5
	      outs6 = norm_axis([.5,.65])
			outs6 = 10.^outs6
	   endelse
      endif

      xyouts, outs1(0), outs1(1), $
		datestr+' , '+species,charsize=outs_size
      xyouts, outs2(0), outs2(1), $
		'UT='+timestring, $
		charsize=outs_size
      xyouts, outs3(0), outs3(1), $
		'!7U!5!dsc!n='+sc_pot_str(i)+'  '+sc_stat_str(i),$
		charsize=outs_size
      oplot, [outs4(0),outs_eol],[outs4(1),outs4(1)], line=1
      xyouts, outs_eol, outs4(1), $
		'1 count in shell',charsize=outs_size

      xyouts, outs5(0), outs5(1), $
		'***** 1st half of block',charsize=outs_size

      if (!p.multi(0) eq 0 and print eq 0) then begin
	label_page,side_label=slab
	message, 'Press any key to see next page',/cont
        res = get_kbrd(1)
      endif

;======2nd half plot=========
      pos_pos_2=where(en_temp2(*,i) gt 0 and en_temp2(*,i) le max_x)
      plot, 	en_temp2(pos_pos_2,i),f_2(pos_pos_2,i),$
		psym=-5,symsize=.6,$
		_extra=kw,$
      		xtitle = x_title_str_en(i) + x_unit_str

      oplot,en_temp2(pos_pos_2,i),onef2(pos_pos_2,i),line=1

      IF (keyword_set(noshift)) THEN BEGIN 
         oplot,[scpot(i),scpot(i)],[min_y,max_y],line=3
      ENDIF

      xyouts, outs1(0), outs1(1), $
		datestr+' , '+species,charsize=outs_size
      xyouts, outs2(0), outs2(1), $
		'UT='+timestring, $
		charsize=outs_size
      xyouts, outs3(0), outs3(1), $
		'!7U!5!dsc!n='+sc_pot_str(i)+'  '+sc_stat_str(i),$
		charsize=outs_size
      oplot, [outs4(0),outs_eol],[outs4(1),outs4(1)], line=1
      xyouts, outs_eol, outs4(1), $
		'1 count in shell',charsize=outs_size

      xyouts, outs5(0), outs5(1), $
		'!7DDDDD!6 2nd half of block',charsize=outs_size

      if (!p.multi(0) eq 0 and print eq 0 or i eq j_block) then begin
	label_page,side_label=slab
	message, 'Press any key to see next page',/cont
        res = get_kbrd(1)
      endif

   endfor

endif
;===========================================================
if (keyword_set(alias_sep) and keyword_set(cplots_en)) then begin
   message, 'plotting <C> time-series...',/cont
   if (print eq 0 and ps eq 0) then $
     window,lwindow,xsize=950,ysize=850,title='Press any key to see next page'

   set_page, nh=3, nv=3

   if (species eq 'ions') then charge=1 else charge=-1

   en_temp = fltarr(j_energy,j_block+1)
   en_temp1 = fltarr(j_energy,j_block+1)
   en_temp2 = fltarr(j_energy,j_block+1)
   scpot = fltarr(j_block+1)
   ; Determine which energy you want to use (shifted or not by the SC pot)
   if (keyword_set(noshift)) then begin
	for ii = 0, j_block do begin
	   if (sc_pot_status(ii) ne 0) then begin
	      en_temp(*,ii) = x_con(*,ii) - charge*sc_potential(ii)
	      en_temp1(*,ii) = en_1(*,ii) - charge*sc_potential(ii)
	      en_temp2(*,ii) = en_2(*,ii) - charge*sc_potential(ii)
	   endif else begin
	      en_temp(*,ii) = x_con(*,ii)
	      en_temp1(*,ii) = en_1(*,ii)
	      en_temp2(*,ii) = en_2(*,ii)
	   endelse
	endfor
   endif else begin
	en_temp = x_con
        en_temp1 = en_1
        en_temp2 = en_2
   ENDELSE

   ; Set x-axis range and unit, and the y-axis range
   if (keyword_set(high)) then begin
      max_x = 1000.
      min_x = 0.  
      x_unit_str = ' [eV]'
      scpot = sc_potential
   endif else if (keyword_set(e_range)) then begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      en_temp1(*,*) = en_temp1(*,*)*1.d-3     ;eV to keV
      en_temp2(*,*) = en_temp2(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = e_range(1)*1.e-3		;keV
      min_x = e_range(0)*1.e-3		;keV
      x_unit_str = ' [keV]'
   endif else begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      en_temp1(*,*) = en_temp1(*,*)*1.d-3     ;eV to keV
      en_temp2(*,*) = en_temp2(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = max(en_temp(*,*))
      min_x = min(en_temp(where(en_temp(*,*) gt 0)))
      x_unit_str = ' [keV]'
   endelse

   min_y = floor_c
   max_y =max_c

;  Create a plotting keyword structure
   if (keyword_set(high)) then begin
      kw = {ylog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,$
        ytitle:ytitle,charsize:plt_size}
   endif else begin
      kw = {ylog:1,xlog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,$
        ytitle:ytitle,charsize:plt_size}
   endelse

;  Dimension the xyouts annotation position vectors
	outs1 = fltarr(2)
	outs2 = fltarr(2)
	outs3 = fltarr(2)
	outs4 = fltarr(2)
	outs5 = fltarr(2)
	outs6 = fltarr(2)
	outs_eol = float(0.)

   for i=0,j_block do begin

;====1st half plot=========
      pos_pos_1=where(en_temp1(*,i) gt 0 and en_temp1(*,i) le max_x)
      plot, 	en_temp1(pos_pos_1,i),c_1(pos_pos_1,i), $
		psym=-2,symsize=.6,$
		_extra=kw,$
		xtitle = x_title_str_en(i) + x_unit_str

      oplot,en_temp1(pos_pos_1,i),onec1(pos_pos_1,i),line=1

      IF (keyword_set(noshift)) THEN BEGIN 
         oplot,[scpot(i),scpot(i)],[min_y,max_y],line=3
      ENDIF

;  Form time string
      hour_block=long(y_con(i))
      minute_block=long((y_con(i)-long(y_con(i)))*60.)
      second_block=long(((y_con(i)-long(y_con(i)))*60.   $
		-minute_block)*60.)
      hour_string=strtrim(hour_block,1)
        if (strlen(hour_string) eq 1) then  $
			hour_string='0'+hour_string
      minute_string=strtrim(minute_block,1)
        if (strlen(minute_string) eq 1) then $
			minute_string='0'+minute_string
      second_string=strtrim(second_block,1)
        if (strlen(second_string) eq 1) then $
			second_string='0'+second_string

      timestring=hour_string+':'+minute_string+':'+second_string

;  Annotation:
;  Set the normalized position vectors for the xyouts annotation
;  (This only needs to be done once, but must be done after the
;  first plot call so that !x.crange and !y.crange are correct)

      if (i eq 0) then begin
	   if (keyword_set(high)) then begin
	      outs1 = norm_axis([.5,.9])
			outs1(1) = 10.^outs1(1)
	      outs2 = norm_axis([.5,.85])
			outs2(1) = 10.^outs2(1)
	      outs3 = norm_axis([.5,.8])
			outs3(1) = 10.^outs3(1)
	      outs4 = norm_axis([.5,.75])
			outs4(1) = 10.^outs4(1)
	      outs5 = norm_axis([.5,.7])
			outs5(1) = 10.^outs5(1)
	      outs6 = norm_axis([.5,.65])
			outs6(1) = 10.^outs6(1)
	      outs_eol = .6*max_x
	   endif else begin
	      outs1 = norm_axis([.5,.9])
			outs1 = 10.^outs1
	      outs2 = norm_axis([.5,.85])
			outs2 = 10.^outs2
	      outs3 = norm_axis([.5,.8])
			outs3 = 10.^outs3
	      outs4 = norm_axis([.5,.75])
			outs4 = 10.^outs4
	      outs5 = norm_axis([.5,.7])
	                outs_eol = 10.^(outs5(0)+.6)
			outs5 = 10.^outs5
	      outs6 = norm_axis([.5,.65])
			outs6 = 10.^outs6
	   endelse
      endif

      xyouts, outs1(0), outs1(1), $
		datestr+' , '+species,charsize=outs_size
      xyouts, outs2(0), outs2(1), $
		'UT='+timestring, $
		charsize=outs_size
      xyouts, outs3(0), outs3(1), $
		'!7U!5!dsc!n='+sc_pot_str(i)+'  '+sc_stat_str(i),$
		charsize=outs_size
      oplot, [outs4(0),outs_eol],[outs4(1),outs4(1)], line=1
      xyouts, outs_eol, outs4(1), $
		'1 count in shell',charsize=outs_size

      xyouts, outs5(0), outs5(1), $
		'***** 1st half of block',charsize=outs_size

      if (!p.multi(0) eq 0 and print eq 0) then begin
	label_page,side_label=slab
	message, 'Press any key to see next page',/cont
        res = get_kbrd(1)
      endif

;======2nd half plot=========
      pos_pos_2=where(en_temp2(*,i) gt 0 and en_temp2(*,i) le max_x)
      plot, 	en_temp2(pos_pos_2,i),c_2(pos_pos_2,i),$
		psym=-5,symsize=.6,$
		_extra=kw,$
		xtitle = x_title_str_en(i) + x_unit_str

      oplot,en_temp2(pos_pos_2,i),onec2(pos_pos_2,i),line=1

      IF (keyword_set(noshift)) THEN BEGIN 
         oplot,[scpot(i),scpot(i)],[min_y,max_y],line=3
      ENDIF

      xyouts, outs1(0), outs1(1), $
		datestr+' , '+species,charsize=outs_size
      xyouts, outs2(0), outs2(1), $
		'UT='+timestring, $
		charsize=outs_size
      xyouts, outs3(0), outs3(1), $
		'!7U!5!dsc!n='+sc_pot_str(i)+'  '+sc_stat_str(i),$
		charsize=outs_size
      oplot, [outs4(0),outs_eol],[outs4(1),outs4(1)], line=1
      xyouts, outs_eol, outs4(1), $
		'1 count in shell',charsize=outs_size

      xyouts, outs5(0), outs5(1), $
		'!7DDDDD!6 2nd half of block',charsize=outs_size

      if (!p.multi(0) eq 0 and print eq 0 or i eq j_block) then begin
	label_page,side_label=slab
	message, 'Press any key to see next page',/cont
        res = get_kbrd(1)
      endif

   endfor

endif
;===========================================================
if (keyword_set(alias_tog) and keyword_set(cplots_en)) then begin
   message, 'plotting <C> time-series...',/cont
   if (print eq 0 and ps eq 0) then $
     window,lwindow,xsize=950,ysize=850,title='Press any key to see next page'

   set_page, nh=3, nv=3

   if (species eq 'ions') then charge=1 else charge=-1

   en_temp = fltarr(j_energy,j_block+1)
   en_temp1 = fltarr(j_energy,j_block+1)
   en_temp2 = fltarr(j_energy,j_block+1)
   scpot = fltarr(j_block+1)
   ; Determine which energy you want to use (shifted or not by the SC pot)
   if (keyword_set(noshift)) then begin
	for ii = 0, j_block do begin
	   if (sc_pot_status(ii) ne 0) then begin
	      en_temp(*,ii) = x_con(*,ii) - charge*sc_potential(ii)
	      en_temp1(*,ii) = en_1(*,ii) - charge*sc_potential(ii)
	      en_temp2(*,ii) = en_2(*,ii) - charge*sc_potential(ii)
	   endif else begin
	      en_temp(*,ii) = x_con(*,ii)
	      en_temp1(*,ii) = en_1(*,ii)
	      en_temp2(*,ii) = en_2(*,ii)
	   endelse
	endfor
   endif else begin
	en_temp = x_con
        en_temp1 = en_1
        en_temp2 = en_2
   ENDELSE

   ; Set x-axis range and unit, and the y-axis range
   if (keyword_set(high)) then begin
      max_x = 1000.
      min_x = 0.  
      x_unit_str = ' [eV]'
      scpot = sc_potential
   endif else if (keyword_set(e_range)) then begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      en_temp1(*,*) = en_temp1(*,*)*1.d-3     ;eV to keV
      en_temp2(*,*) = en_temp2(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = e_range(1)*1.e-3		;keV
      min_x = e_range(0)*1.e-3		;keV
      x_unit_str = ' [keV]'
   endif else begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      en_temp1(*,*) = en_temp1(*,*)*1.d-3     ;eV to keV
      en_temp2(*,*) = en_temp2(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = max(en_temp(*,*))
      min_x = min(en_temp(where(en_temp(*,*) gt 0)))
      x_unit_str = ' [keV]'
   endelse

   min_y = floor_c
   max_y =max_c

;  Create a plotting keyword structure
   if (keyword_set(high)) then begin
      kw = {ylog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,$
        ytitle:ytitle,charsize:plt_size}
   endif else begin
      kw = {ylog:1,xlog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,$
        ytitle:ytitle,charsize:plt_size}
   endelse

;  Dimension the xyouts annotation position vectors
	outs1 = fltarr(2)
	outs2 = fltarr(2)
	outs3 = fltarr(2)
	outs4 = fltarr(2)
	outs5 = fltarr(2)
	outs6 = fltarr(2)
	outs_eol = float(0.)

   for i=0,j_block do begin

      pos_pos_1=where(en_temp1(*,i) gt 0 and en_temp1(*,i) le max_x)
      plot, 	en_temp1(pos_pos_1,i),c_1(pos_pos_1,i), $
		psym=-2,symsize=.6,$
		_extra=kw,$
      		xtitle = x_title_str_en(i) + x_unit_str

      pos_pos_2=where(en_temp2(*,i) gt 0 and en_temp2(*,i) le max_x)
      oplot,en_temp2(pos_pos_2,i),c_2(pos_pos_2,i),psym=-5,symsize=.6

      pos_pos_big = where(en_temp(*,i) gt 0 and en_temp(*,i) le max_x)
      oplot,en_temp(pos_pos_big,i),one_cnt_c(pos_pos_big,i),line=1

      IF (keyword_set(noshift)) THEN BEGIN 
         oplot,[scpot(i),scpot(i)],[min_y,max_y],line=3
      ENDIF

;  Form time string
      hour_block=long(y_con(i))
      minute_block=long((y_con(i)-long(y_con(i)))*60.)
      second_block=long(((y_con(i)-long(y_con(i)))*60.   $
		-minute_block)*60.)
      hour_string=strtrim(hour_block,1)
        if (strlen(hour_string) eq 1) then  $
			hour_string='0'+hour_string
      minute_string=strtrim(minute_block,1)
        if (strlen(minute_string) eq 1) then $
			minute_string='0'+minute_string
      second_string=strtrim(second_block,1)
        if (strlen(second_string) eq 1) then $
			second_string='0'+second_string

      timestring=hour_string+':'+minute_string+':'+second_string

;  Annotation:
;  Set the normalized position vectors for the xyouts annotation
;  (This only needs to be done once, but must be done after the
;  first plot call so that !x.crange and !y.crange are correct)

      if (i eq 0) then begin
	   if (keyword_set(high)) then begin
	      outs1 = norm_axis([.5,.9])
			outs1(1) = 10.^outs1(1)
	      outs2 = norm_axis([.5,.85])
			outs2(1) = 10.^outs2(1)
	      outs3 = norm_axis([.5,.8])
			outs3(1) = 10.^outs3(1)
	      outs4 = norm_axis([.5,.75])
			outs4(1) = 10.^outs4(1)
	      outs5 = norm_axis([.5,.7])
			outs5(1) = 10.^outs5(1)
	      outs6 = norm_axis([.5,.65])
			outs6(1) = 10.^outs6(1)
	      outs_eol = .6*max_x
	   endif else begin
	      outs1 = norm_axis([.5,.9])
			outs1 = 10.^outs1
	      outs2 = norm_axis([.5,.85])
			outs2 = 10.^outs2
	      outs3 = norm_axis([.5,.8])
			outs3 = 10.^outs3
	      outs4 = norm_axis([.5,.75])
			outs4 = 10.^outs4
	      outs5 = norm_axis([.5,.7])
	                outs_eol = 10.^(outs5(0)+.6)
			outs5 = 10.^outs5
	      outs6 = norm_axis([.5,.65])
			outs6 = 10.^outs6
	   endelse
      endif
      xyouts, outs1(0), outs1(1), $
		datestr+' , '+species,charsize=outs_size
      xyouts, outs2(0), outs2(1), $
		'UT='+timestring, $
		charsize=outs_size
      xyouts, outs3(0), outs3(1), $
		'!7U!5!dsc!n='+sc_pot_str(i)+'  '+sc_stat_str(i),$
		charsize=outs_size
      oplot, [outs4(0),outs_eol],[outs4(1),outs4(1)], line=1
      xyouts, outs_eol, outs4(1), $
		'1 count in shell',charsize=outs_size

      xyouts, outs5(0), outs5(1), $
		'***** 1st half of block',charsize=outs_size

      xyouts, outs6(0), outs6(1), $
		'!7DDDDD!6 2nd half of block',charsize=outs_size

      if (!p.multi(0) eq 0 and print eq 0 or i eq j_block) then begin
	label_page,side_label=slab
	message, 'Press any key to see next page',/cont
        res = get_kbrd(1)
      endif

   endfor

endif

;===========================================================
if (keyword_set(cplots_en) and not keyword_set(alias_sep) and not $
	keyword_set(alias_tog)) then begin
   message, 'plotting <C> time series...',/cont

   if (print eq 0 and ps eq 0) then $
   window,lwindow,xsize=950,ysize=850,title='Press any key to see next'

   set_page, nh=3, nv=3

   if (species eq 'ions') then charge=1 else charge=-1

   en_temp = fltarr(j_energy,j_block+1)
   scpot = fltarr(j_block+1)
   ; Determine which energy you want to use (shifted or not by the SC pot)
   if (keyword_set(noshift)) then begin
	for ii = 0, j_block do begin
	   if (sc_pot_status(ii) ne 0) then begin
	      en_temp(*,ii) = x_con(*,ii) - charge*sc_potential(ii)
	   endif else begin
	      en_temp(*,ii) = x_con(*,ii)
	   endelse
	endfor
   endif else begin
	en_temp = x_con
   ENDELSE

   ; Set x-axis range and unit, and the y-axis range
   if (keyword_set(high)) then begin
      max_x = 1000.
      min_x = 0.  
      x_unit_str = ' [eV]'
      scpot = sc_potential
   endif else if (keyword_set(e_range)) then begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = e_range(1)*1.e-3		;keV
      min_x = e_range(0)*1.e-3		;keV
      x_unit_str = ' [keV]'
   endif else begin
      en_temp(*,*) = en_temp(*,*)*1.d-3     ;eV to keV
      scpot = sc_potential*1.d-3
      max_x = max(en_temp(*,*))
      min_x = min(en_temp(where(en_temp(*,*) gt 0)))
      x_unit_str = ' [keV]'
   endelse
   min_y = floor_c
   max_y =max_c

;  Create a plotting keyword structure
   if (keyword_set(high)) then begin
      kw = {ylog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,$
        ytitle:ytitle,charsize:plt_size,psym:-1,symsize:.6}
   endif else begin
      kw = {ylog:1,xlog:1,xrange:[min_x,max_x],yrange:[min_y,max_y], $
	ystyle:1,xstyle:1,$
        ytitle:ytitle,charsize:plt_size,psym:-1,symsize:.6}
   endelse

;  Dimension the xyouts annotation position vectors
	outs1 = fltarr(2)
	outs2 = fltarr(2)
	outs3 = fltarr(2)
	outs4 = fltarr(2)
	outs5 = fltarr(2)
	outs_eol = float(0.)

   for i=0,j_block do begin

      pos_pos=where(en_temp(*,i) gt 0 and en_temp(*,i) le max_x)
      plot, 	en_temp(pos_pos,i),z_con_2(pos_pos,i),$
		_extra = kw,$
      		xtitle = x_title_str_en(i)+ x_unit_str

      oplot,en_temp(pos_pos,i),one_cnt_c(pos_pos,i),line=1

      IF (keyword_set(noshift)) THEN BEGIN 
         oplot,[scpot(i),scpot(i)],[min_y,max_y],line=3
      ENDIF

;  Form time string
      hour_block=long(y_con(i))
      minute_block=long((y_con(i)-long(y_con(i)))*60.)
      second_block=long(((y_con(i)-long(y_con(i)))*60.   $
		-minute_block)*60.)

      hour_string=strtrim(hour_block,1)
        if (strlen(hour_string) eq 1) then  $
			hour_string='0'+hour_string
      minute_string=strtrim(minute_block,1)
        if (strlen(minute_string) eq 1) then $
			minute_string='0'+minute_string
      second_string=strtrim(second_block,1)
        if (strlen(second_string) eq 1) then $
			second_string='0'+second_string

      timestring=hour_string+':'+minute_string+':'+second_string

;  Annotation:
;  Set the normalized position vectors for the xyouts annotation
;  (This only needs to be done once, but must be done after the
;  first plot call so that !x.crange and !y.crange are correct)

      if (i eq 0) then begin
	   if (keyword_set(high)) then begin
	      outs1 = norm_axis([.5,.9])
			outs1(1) = 10.^outs1(1)
	      outs2 = norm_axis([.5,.85])
			outs2(1) = 10.^outs2(1)
	      outs3 = norm_axis([.5,.8])
			outs3(1) = 10.^outs3(1)
	      outs4 = norm_axis([.5,.75])
			outs4(1) = 10.^outs4(1)
	      outs5 = norm_axis([.5,.7])
	           	outs_eol = .6*max_x
			outs5(1) = 10.^outs5(1)
	   endif else begin
	      outs1 = norm_axis([.5,.9])
			outs1 = 10.^outs1
	      outs2 = norm_axis([.5,.85])
			outs2 = 10.^outs2
	      outs3 = norm_axis([.5,.8])
			outs3 = 10.^outs3
	      outs4 = norm_axis([.5,.75])
			outs4 = 10.^outs4
	      outs5 = norm_axis([.5,.7])
	                outs_eol = 10.^(outs5(0)+.6)
			outs5 = 10.^outs5
	   endelse
      endif

      xyouts, outs1(0), outs1(1), $
		datestr+' , '+species,charsize=outs_size
      xyouts, outs2(0), outs2(1), $
		'UT='+timestring, $
		charsize=outs_size
      xyouts, outs3(0), outs3(1), $
		'!7U!5!dsc!n='+sc_pot_str(i)+'  '+sc_stat_str(i),$
		charsize=outs_size

      oplot, [outs4(0),outs_eol],[outs4(1),outs4(1)], line=1
      xyouts, outs_eol, outs4(1), $
		'1 count in shell',charsize=outs_size

      xyouts, outs5(0), outs5(1), $
		'+++++ Whole block',charsize=outs_size

      if (!p.multi(0) eq 0 and print eq 0 or i eq j_block) then begin
	label_page,side_label=slab
	message, 'Press any key to see next page',/cont
        res = get_kbrd(1)
      endif


   endfor

endif
;============================================================


if (print EQ 1) then begin
   end_of_prog, /print
endif else if (gif eq 1) then begin
   message, 'idl.gif created',/cont
   write_gif,'idl.gif', TVRD()
   end_of_prog
endif else begin
   end_of_prog
endelse

end_flag:

END

