;=====================================================================
;+
; NAME: ppa_spectra.pro
;
; WRITTEN:	Pamela A. Puhl-Quinn, 4/02/97, v1.00
;               Uses L1 files, 3/31/98, v2.00, ppq
;               Post-script option added, 8/7/98, ppq
;               Grey-out where validb_flag=0, 10/9/98, ppq
;
; USAGE:                             
;                 --------Mandatory----- 	----Optional----------
;    ppa_spectra,970111,1024,5,	                cps_range=[.1,100]
;						raw_range=[1,500]
;						vt_range = [.01,1000]
;						nwindow=5
; MANDATORY ARGUMENTS:
;		1st argument:  YYMMDD or YYYYMMDD
;		2nd argument: 'HH:MM:SS','HH:MM',HHMMSS,HHMM,HH.HHH,H
;		3rd argument:  # of minutes
;
; OPTIONAL KEYWORDS:
;   raw_range=[min,max] 	; choose range for raw counts
;   vt_range=[min,max]	        ; choose range for vt. viewing time
;   cps_range=[min,max] 	; choose range for counts/sec
;   nwindow=num			; choose window number for display
;
; This tool will not cross over day boundaries, you must load the next
; day.
;-
;====================================================================

pro ppa_spectra, date_in, time_in, nmin , $
                 nwindow=nwindow, $
                 raw_range=raw_range, $
                 cps_range=cps_range, $
                 vt_range=vt_range
   
   common ppa_spectra,cdate,ctime,cnmin, $
    ppa_info, ppa_data

; Usage help
  IF (n_params() EQ 0) THEN BEGIN
     doc_library, 'ppa_spectra'
     stop
  ENDIF

; Convert the date and time into something useful==================
   hydra_time_io,date_in,time_in,fyear,fmonth,fday,fhour,fminute,fsecond
   date =  strmid(date_in,0,8)
   time = string(fhour,'(i2.2)')+':'+string(fminute,'(i2.2)')+':'+string(fsecond,'(i2.2)')

;=================================================
   
; Check to see if the filenames for 'date' already exists in the buffer, 
; and if not, then the filenames have to be read
   
   if (keyword_set(cdate)) then begin  
      if (cdate eq date and ctime eq time and cnmin eq nmin) then begin
         read=0			; data already in buffer 		
      endif else begin
         read=1			; data needs to be read
      endelse
   endif else begin
      read=1                    ; data needs to be read
   endelse
   
   if (keyword_set(nwindow)) then lwindow=nwindow else lwindow=0
   
   if (keyword_set(raw_range)) then begin
      if (raw_range(0) le 0 or raw_range(1) le 0) then begin
         message,'raw_range must be greater than zero'
      endif else begin
         real_raw_min = raw_range(0)
         real_raw_max = raw_range(1)
      endelse
   endif else begin
      real_raw_min = 1
      real_raw_max = 5000
   endelse
   
   if (keyword_set(vt_range)) then begin
      if (vt_range(0) le 0 or vt_range(1) le 0) then begin
         message,'vt_range must be greater than zero'
      endif else begin
         real_vt_min = vt_range(0)
         real_vt_max = vt_range(1)
      endelse
   endif else begin
      real_vt_min = .01
      real_vt_max = 1000.
   endelse
   
   if (keyword_set(cps_range)) then begin
      if (cps_range(0) le 0 or cps_range(1) le 0) then begin
         message,'cps_range must be greater than zero'
      endif else begin
         real_cps_min = cps_range(0)
         real_cps_max = cps_range(1)
      endelse
   endif else begin
      real_cps_min = .001
      real_cps_max = 5000.
   endelse
   
   IF (read) THEN BEGIN 
      
      init_ppa_l1, date, ppa_info, status=status, error_msg=error_msg
      IF (status EQ 0) THEN message, error_msg
      
      time1 = time_in           ; Hours
      time2 = time1 + nmin/60.  ; Hours

      message,'time1:  '+string(time1,'(f5.2)')+' hr',/cont
      message,'time2:  '+string(time2,'(f5.2)')+' hr',/cont
      
      recnum = find_ppa_recnum([time1,time2], ppa_info, status=status, $
                               error_msg=error_msg)
      IF (status EQ 0) THEN message, error_msg

;      recnum = indgen(10) + 48
      
      read_ppa_l1, recnum, ppa_data, ppa_info
      
; Set the value of cdate in the buffer in order to access the
; buffered data on subsequent calls to this procedure without having
; to read the data in from a file every time
      cdate = date
      ctime = time
      cnmin = nmin
      
   endif
   
;===========Spectrograms=======================================	
   
   IF (ppa_info.mode EQ 1) THEN BEGIN 
      nsweep = 4
      nlevels = long(ppa_info.nen/float(nsweep)) ; Should be 9
      nhbins = ppa_info.nbin    ; Should be 120
   ENDIF ELSE IF (ppa_info.mode EQ 40) THEN BEGIN
      nsweep = 2
      nlevels = long(ppa_info.nen/float(nsweep)) ; Should be 8
      nhbins = ppa_info.nbin    ; Should be 160
   ENDIF ELSE BEGIN 
      message, 'Not able to handle mode other than 1 or 40'
   ENDELSE
      
   ntime = n_elements(ppa_data.time)
   for i_block = 0, ntime-1 do begin
      
      for isweep = 0,nsweep-1 do BEGIN
         ihour = long(ppa_data(i_block).time/3600.)
         imin = long((ppa_data(i_block).time/3600.-ihour)*60.)
         isec =long(((ppa_data(i_block).time/3600.-ihour)*60. -imin)*60.)
         shour = string(ihour,'(i2.2)')
         smin = string(imin,'(i2.2)')
         ssec = string(isec,'(i2.2)')
         stime = shour+':'+smin+':'+ssec
         sblock_time = date +'_'+stime+' Sweep '+ string(isweep+1,'(i1)')
         
         e_inds = indgen(nlevels) + isweep*nlevels
         red_energy=ppa_info.energy(e_inds)
         validb_flag = reform(ppa_data(i_block).validb_flag(e_inds))

         red_alpha=ppa_info.alpha
         red_width=ppa_info.width
         cps_2d = fltarr(nhbins,nlevels)
         vt_2d = reform(ppa_data(i_block).viewtime(0:nhbins-1,e_inds))
         raw_2d = reform(ppa_data(i_block).counts(0:nhbins-1,e_inds))
         
         vt_zeros = where (vt_2d eq 0)
         vt_ne_zero = where (vt_2d ne 0)
         cps_2d(vt_ne_zero) = float(raw_2d(vt_ne_zero)) $
             /(vt_2d(vt_ne_zero))
         if (vt_zeros(0) ne -1) then cps_2d(vt_zeros) = -999

; Assign grey (-999) to those energy levels where validb_flag is 0
         FOR ilev=0,nlevels-1 DO BEGIN
            IF (NOT validb_flag(ilev)) THEN BEGIN
               cps_2d(*,ilev) = -999
               vt_2d(*,ilev) = -999
               raw_2d(*,ilev) = -999
            ENDIF 
         ENDFOR
         
;===========================END OF NEW STUFF========================
         
; Sort the arrays so that the energy is monotonically increasing
         
         red_energy_sorted = fltarr(nlevels)
         cps_2d_sorted = fltarr(nhbins,nlevels)
         vt_2d_sorted = fltarr(nhbins,nlevels)
         raw_2d_sorted = lonarr(nhbins,nlevels)
         
         sorted_indices=sort(red_energy)
         
         red_energy_sorted(*)=red_energy(sorted_indices)
         cps_2d_sorted(*,*)=cps_2d(*,sorted_indices)
         vt_2d_sorted(*,*) = vt_2d(*,sorted_indices)
         raw_2d_sorted(*,*) = raw_2d(*,sorted_indices)
         
; Expand red_alpha and cps_2d_sorted from nhbins to 256; energy stays the same
         energy_spectra = fltarr(nlevels)
         alpha_spectra = fltarr(256)
         cps_spectra = fltarr(256,nlevels)
         vt_spectra = fltarr(256,nlevels)
         raw_spectra = lonarr(256,nlevels)
         
         energy_spectra = red_energy_sorted
         
         kk = -1                ; Expanded array counter
         for jj = 0,nhbins-1 do begin ; nhbins counter
            for i = 1,red_width(jj) do begin
               kk = kk + 1
               alpha_spectra(kk) = red_alpha(jj)
               cps_spectra(kk,*) = cps_2d_sorted(jj,*)
               vt_spectra(kk,*) = vt_2d_sorted(jj,*)
               raw_spectra(kk,*) = raw_2d_sorted(jj,*)
            endfor
         endfor
         
         
; Finally, bump the cps and vt rate up so that when it is 'byte'-ized, we
; will not truncate all of the values to 0 or 1 if they happen to be small
         
         bump_cps = 5000.
         bump_vt = 1000.
         bump_raw = 1.          ; No need to bump the raw counts
         cps_spectra = cps_spectra*bump_cps
         vt_spectra = vt_spectra*bump_vt
         raw_spectra = raw_spectra*bump_raw
         
;==========================Spectrogram================================
         
         pi=acos(-1)
;======File independent procedures=============
         if (i_block eq 0 and isweep eq 0) then begin
            
            winxsz = 750.
            winysz = 750.
            window,lwindow,xsize=winxsz,ysize=winysz,title='PPA Spectrograms ('+    $
             strtrim(lwindow,2)+')'
; Want a 100 element color table where
;            0:96  = VIB G YOR (rainbow)
;              97  = white
;              98  = grey
;              99  = black
            
            n_rainbow_plus_white = 98 ; Number of colors defining rainbow + white, 0:97
            
            _extra = {ncolors:100}
            loadct_hydra, 18, _extra=_extra
            multi, 1.03         ;Want to define two colors of my own on the end
            top=  97            ;top is going to be white
            grey= top+1         ;grey is used for when cps = -999 (i.e. vt=0)
            black = top+2       ;black is used for when cps = 0
            tvlct, red, gre, blu, /get
            red(top)= 255 & gre(top)=255 & blu(top)=255
            red(grey)=140 & gre(grey)=140 & blu(grey)=140
            red(black)=0 & gre(black)=0 & blu(black)=0
; Redefine the first two blacks to be blue
            for i=0,1 do begin
               red(i)=red(2)
               gre(i)=gre(2)
               blu(i)=blu(2)
            endfor
            tvlct, red, gre, blu
         ENDIF
         
;=============================================
         
         !p.multi=[0,1,1,0,0]
         !x.style=1
         !y.style=1
         !p.charsize=.9
         !p.background = top
         !p.font = -1
         !p.color = black
         !p.ticklen = -.01
         
;======Dummy axes for the spectrograms===========
         x=[1,2,3]
         y=[1,2,3]
         
         xmin = 0.
         xmax = 180.
         
         ymin = energy_spectra(0)
         ymax = 10^((alog10(energy_spectra(nlevels-1)) - $
                     alog10(energy_spectra(nlevels-2))) + alog10(energy_spectra(nlevels-1)))
         
         tickvec = fltarr(nlevels)
         for i = 0,nlevels-2 do begin
            tickvec(i) = 10^((alog10(energy_spectra(i+1)) - $
                              alog10(energy_spectra(i)))/2. + alog10(energy_spectra(i)))
         endfor
         tickvec(nlevels-1) = 10^((alog10(energy_spectra(nlevels-1)) - $
                           alog10(energy_spectra(nlevels-2)))/2. + alog10(energy_spectra(nlevels-1)))
         
         tickname = string(energy_spectra,'(i4)')
         
;========cps spectragram=============
         plot, x, y, /nodata,/ylog,position=[.07,.68,.85,.95],color=black,$
          yrange=[ymin,ymax],$
          xrange=[xmin,xmax],$
          yticks = nlevels-1,$
          ytickv = tickvec,$
          ytickname = tickname,xticks=18,xtickv=findgen(19)*10.,xtickname=strarr(19)+' '
; Please note that the only reason this is a representative scaling for
; the energy levels is because the levels increase approximately
; logarithmically.  If they didn't, you'd have to do something more
; sophisticated.
         
;!p.multi = !p.multi + [1,0,0,0,0]
         
; Define the data and device ranges established by the plot above;
; Remember that !y.crange is just the power of 10 when /ylog is
; used!!!
         
         xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
         lldata = [xdata(0), ydata(0)]
         sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
         
         ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
         ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
         sizedev=(ur-ll)(0:1)

         sizedev_spec = sizedev
         ll_spec1 = ll
         
         gt_zero = where(cps_spectra gt 0)
         lt_zero = where(cps_spectra lt 0) ; When vt = 0 (cps=-999), grey
         zeros = where(cps_spectra eq 0) ; When raw_dist = 0, black
         
; Define a new array as the log of cps_spectra, retain both the 0 and
; -999 values for consistency
         cps_log = fltarr(256,nlevels)
         cps_log(gt_zero) = alog10(cps_spectra(gt_zero))
         if (zeros(0) ne -1) then cps_log(zeros) = 0
         if (lt_zero(0) ne -1) then cps_log(lt_zero) = -999
         
; Define the byte-scaled color array Zcolor to perform the 'tv' on
; Define the minimum:	real_min = real_cps_min
;			scaled_min = real_cps_min*bump_cps
;			log10(scaled_min)=alog10(real_cps_min*bump_cps)
         
         minZcolor = alog10(real_cps_min*bump_cps)
         
; Define the maximum:	real_max = real_cps_max
;			scaled_max = real_cps_max*bump_cps
;			log10(scaled_max)=alog10(real_cps_max*bump_cps)
         
         maxZcolor = alog10(real_cps_max*bump_cps)
         
         Zcolor = bytarr(256,nlevels)
         Zcolor(gt_zero) = bytscl(cps_log(gt_zero),max=maxZcolor, $
                                  min=minZcolor,top=top)
         if (lt_zero(0) ne -1) then Zcolor(lt_zero) = grey
         if (zeros(0) ne -1) then Zcolor(zeros) = black
         
;         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0), ll(1)
         
         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0)/winxsz,ll(1)/winysz,/normal
         
         plot, x,y,/ylog,/nodata, /noerase,position=[.07,.68,.85,.95],$
          yrange=[ymin,ymax],$
          xrange=[xmin,xmax],xticks=18,xtickv=findgen(19)*10., xtickname=strarr(19)+' ',$
          yticks = nlevels-1, $
          ytickv = tickvec,$
          ytickname=tickname,$
          ytitle='Energy, [eV]',$
          title='PPA '+ sblock_time + ' ' + $
          ' Counts/sec ( = Raw/Viewing Time; Grey where validb_flag= 0)', color=black
         
;=================================================
         
; Now for the color bar (uuugh!)
         
; Create a 2-d array which depicts the color bar data
; There are n_rainbow_plus_white colors, ranging from navy (0) to white (97)
         
         Zcolor_bar = bytarr(10,n_rainbow_plus_white)
         for i = 0,9 do begin
            Zcolor_bar(i,*) = bindgen(n_rainbow_plus_white)
         endfor
         
; Create a plot for the axes
         plot, 	x, y ,/ylog,/nodata,position=[.93,.68,.97,.95],/noerase, $
          xrange = [0,6],xstyle=4,$
          yrange=[real_cps_min,real_cps_max],ystyle=1,ytickformat = '(f8.3)',$
          color=black
         
         xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
         lldata = [xdata(0), ydata(0)]
         sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
         
         ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
         ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
         sizedev=(ur-ll)(0:1)
         

         sizedev_cb = sizedev
         ll_cb1 = ll

;         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0), ll(1)
         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0)/winxsz, ll(1)/winysz,/normal         
         plot, 	x, y ,/ylog,/nodata,position=[.93,.68,.97,.95],/noerase, $
          xrange = [0,6], xstyle=4,$
          yrange=[real_cps_min,real_cps_max], ystyle=1, ytickformat='(f8.3)', $
          title = 'Counts/sec',color=black
         
;========vt spectragram=============
         
         plot, x, y, /nodata,/noerase,/ylog,position=[.07,.06,.85,.33],color=black,$
          yrange=[ymin,ymax],$
          xrange=[xmin,xmax],$
          yticks = nlevels-1,$
          ytickv = tickvec,$
          ytickname = tickname,xticks=18,xtickv=findgen(19)*10.
         
; Define the data and device ranges established by the plot above;
; Remember that !y.crange is just the power of 10 when /ylog is
; used!!!
         
         xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
         lldata = [xdata(0), ydata(0)]
         sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
         
         ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
         ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
         sizedev=(ur-ll)(0:1)

         ll_spec2 = ll
         
         gt_zero = where(vt_spectra gt 0)
         lt_zero = where(vt_spectra LT 0) ; Where validb_flag invalid
         zeros = where(vt_spectra eq 0) ; When vt = 0, black
         
; Define a new array as the log of vt_spectra
         vt_log = fltarr(256,nlevels)
         vt_log(gt_zero) = alog10(vt_spectra(gt_zero))
         if (zeros(0) ne -1) then vt_log(zeros) = 0
         IF (lt_zero(0) NE -1) THEN vt_log(lt_zero) = -999
         
; Define the byte-scaled color array Zcolor to perform the 'tv' on
; Define the minimum:	real_min = real_vt_min
;			scaled_min = real_vt_min*bump_vt
;			log10(scaled_min)=alog10(real_vt_min*bump_vt)
         
         minZcolor = alog10(real_vt_min*bump_vt)
         
; Define the maximum:	real_max = real_vt_max
;			scaled_max = real_vt_max*bump_vt
;			log10(scaled_max)=alog10(real_vt_max*bump_vt)
         
         maxZcolor = alog10(real_vt_max*bump_vt)
         
         Zcolor = bytarr(256,nlevels)
         Zcolor(gt_zero) = bytscl(vt_log(gt_zero),max=maxZcolor, $
                                  min=minZcolor,top=top)
         IF (lt_zero(0) NE -1) THEN Zcolor(lt_zero) = grey
         if (zeros(0) ne -1) then Zcolor(zeros) = black
         
;         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0), ll(1)
         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0)/winxsz, ll(1)/winysz,/normal         
         plot, x,y,/ylog,/nodata,/noerase,position=[.07,.06,.85,.33],$
          yrange=[ymin,ymax],$
          xrange=[xmin,xmax],xticks=18,xtickv=findgen(19)*10.,$
          yticks = nlevels-1,$
          ytickv = tickvec,$
          ytickname = tickname,$
          ytitle='Energy, [eV]',$
          xtitle='Pitch Angle [degrees]',$
          title='PPA Probable Viewing Time [s] (Grey where validb_flag=0)', color=black
         
;=================================================
; Now for the color bar (uuugh!)
         
; Create a 2-d array which depicts the color bar data
; There are n_rainbow_plus_white colors, ranging from navy (0) to
; white (97)
         
         Zcolor_bar = bytarr(10,n_rainbow_plus_white)
         for i = 0,9 do begin
            Zcolor_bar(i,*) = bindgen(n_rainbow_plus_white)
         endfor
         
; Create a plot for the axes
         plot, x, y ,/ylog,/nodata,position=[.93,.06,.97,.33],/noerase, $
          xstyle=4, xrange=[0,5],$
          yrange=[real_vt_min,real_vt_max], ytickformat='(f8.3)', ystyle=1, $
          color=black
         
         xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
         lldata = [xdata(0), ydata(0)]
         sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
         
         ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
         ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
         sizedev=(ur-ll)(0:1)

         ll_cb2 = ll
         
;         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0), ll(1)
         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0)/winxsz, ll(1)/winysz,/normal         
         plot, x, y ,/ylog,/nodata,position=[.93,.06,.97,.33],/noerase, $
          xstyle=4, xrange = [0,5], $
          yrange=[real_vt_min,real_vt_max], ystyle=1, ytickformat = '(f8.3)', $
          title = 'Time [s]',color=black
         
;========raw spectragram=============
         plot, x, y, /nodata,/noerase,/ylog,position=[.07,.37,.85,.64],color=black,$
          yrange=[ymin,ymax],$
          xrange=[xmin,xmax],$
          yticks = nlevels-1,$
          ytickv = tickvec,$
          ytickname = tickname,xticks=18,xtickv=findgen(19)*10.,xtickname=strarr(19)+' '
         
; Define the data and device ranges established by the plot above;
; Remember that !y.crange is just the power of 10 when /ylog is
; used!!!
         
         xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
         lldata = [xdata(0), ydata(0)]
         sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
         
         ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
         ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
         sizedev=(ur-ll)(0:1)

         ll_spec3 = ll
         
         gt_zero = where(raw_spectra gt 0)
         lt_zero = where(raw_spectra LT 0) ; Where validb_flag invalid
         zeros = where(raw_spectra eq 0) ; When raw_dist = 0, black
         
; Define a new array as the log of raw_spectra
         raw_log = fltarr(256,nlevels)
         raw_log(gt_zero) = alog10(raw_spectra(gt_zero))
         if (zeros(0) ne -1) then raw_log(zeros) = -1. ; To distinguish them from the log(1) values
         IF (lt_zero(0) NE -1) THEN raw_log(lt_zero) = -999
         
; Define the byte-scaled color array Zcolor to perform the 'tv' on
; Define the minimum:	real_min = real_raw_min
;			scaled_min = real_raw_min*bump_raw
;			log10(scaled_min)=alog10(real_raw_min*bump_raw)
         
         minZcolor = alog10(real_raw_min*bump_raw)
         
; Define the maximum:	real_max = real_raw_max
;			scaled_max = real_raw_max*bump_raw
;			log10(scaled_max)=alog10(real_raw_max*bump_raw)
         
         maxZcolor = alog10(real_raw_max*bump_raw)
         
         Zcolor = bytarr(256,nlevels)
         Zcolor(gt_zero) = bytscl(raw_log(gt_zero),max=maxZcolor, $
                                  min=minZcolor,top=top)
         if (zeros(0) ne -1) then Zcolor(zeros) = black
         IF (lt_zero(0) NE -1) THEN Zcolor(lt_zero) = grey
         
;         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0), ll(1)
         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0)/winxsz, ll(1)/winysz,/normal         
         plot, x,y,/ylog,/nodata, /noerase,position=[.07,.37,.85,.64],$
          yrange=[ymin,ymax],$
          xrange=[xmin,xmax],xticks=18,xtickv=findgen(19)*10.,xtickname=strarr(19)+' ',$
          yticks = nlevels-1,$
          ytickv = tickvec,$
          ytickname = tickname,$
          ytitle='Energy, [eV]',$
          title='PPA Raw Counts (Grey where validb_flag=0)', color=black
         
;=================================================
; Now for the color bar (uuugh!)
         
; Create a 2-d array which depicts the color bar data
; There are n_rainbow_plus_white colors, ranging from navy (0) to white (97)
         
         Zcolor_bar = bytarr(10,n_rainbow_plus_white)
         for i = 0,9 do begin
            Zcolor_bar(i,*) = bindgen(n_rainbow_plus_white)
         endfor
         
; Create a plot for the axes
         plot, x, y ,/ylog,/nodata,position=[.93,.37,.97,.64],/noerase, $
          xstyle=4, xrange = [0,5], $
          yrange=[real_raw_min,real_raw_max], ystyle=1, $
          color=black
         
         xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
         lldata = [xdata(0), ydata(0)]
         sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
         
         ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
         ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
         sizedev=(ur-ll)(0:1)
         

         ll_cb3 = ll

;         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0), ll(1)
         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0)/winxsz, ll(1)/winysz,/normal         
         plot, x, y ,/ylog,/nodata,position=[.93,.37,.97,.64],/noerase, $
          xstyle=4, xrange = [0,5],$
          yrange=[real_raw_min,real_raw_max], ystyle=1, $
          title = 'Raw Cnts.',color=black
         
         side_label = ' date='+date+', product_name='+ppa_info.product_name+', product_version='+ppa_info.product_version
         label_page, side_label=side_label

         ans = ' '
         read, 'Make a post-script file of this? (y or n, q to quit) ', ans
         IF ans EQ 'q' THEN GOTO, end_flag
         IF ans EQ 'Y' OR ans EQ 'y' THEN BEGIN

            ihour = long(ppa_data(i_block).time/3600.)
            imin = long((ppa_data(i_block).time/3600.-ihour)*60.)
            isec =long(((ppa_data(i_block).time/3600.-ihour)*60. -imin)*60.)
            shour = string(ihour,'(i2.2)')
            smin = string(imin,'(i2.2)')
            ssec = string(isec,'(i2.2)')
            stime = shour+':'+smin+':'+ssec
            sblock_time = date +'_'+stime+' Sweep '+ string(isweep+1,'(i1)')

            set_plot, 'ps'
            device, /landscape, /color, bits=8, file='PPA_'+date+'_'+stime+'_Sweep_'+string(isweep+1,'(i1)')+'.ps'

            e_inds = indgen(nlevels) + isweep*nlevels
            red_energy=ppa_info.energy(e_inds)
            validb_flag = reform(ppa_data(i_block).validb_flag(e_inds))
            red_alpha=ppa_info.alpha
            red_width=ppa_info.width
            cps_2d = fltarr(nhbins,nlevels)
            vt_2d = reform(ppa_data(i_block).viewtime(0:nhbins-1,e_inds))
            raw_2d = reform(ppa_data(i_block).counts(0:nhbins-1,e_inds))
            
            vt_zeros = where (vt_2d eq 0)
            vt_ne_zero = where (vt_2d ne 0)
            cps_2d(vt_ne_zero) = float(raw_2d(vt_ne_zero)) $
             /(vt_2d(vt_ne_zero))
            if (vt_zeros(0) ne -1) then cps_2d(vt_zeros) = -999

; Assign grey (-999) to those energy levels where validb_flag is 0
            FOR ilev=0,nlevels-1 DO BEGIN
               IF (NOT validb_flag(ilev)) THEN BEGIN
                  cps_2d(*,ilev) = -999
                  vt_2d(*,ilev) = -999
                  raw_2d(*,ilev) = -999
               ENDIF 
            ENDFOR
            
;===========================END OF NEW STUFF========================
            
; Sort the arrays so that the energy is monotonically increasing
            
            red_energy_sorted = fltarr(nlevels)
            cps_2d_sorted = fltarr(nhbins,nlevels)
            vt_2d_sorted = fltarr(nhbins,nlevels)
            raw_2d_sorted = lonarr(nhbins,nlevels)
            
            sorted_indices=sort(red_energy)
            
            red_energy_sorted(*)=red_energy(sorted_indices)
            cps_2d_sorted(*,*)=cps_2d(*,sorted_indices)
            vt_2d_sorted(*,*) = vt_2d(*,sorted_indices)
            raw_2d_sorted(*,*) = raw_2d(*,sorted_indices)
            
; Expand red_alpha and cps_2d_sorted from nhbins to 256; energy stays the same
            energy_spectra = fltarr(nlevels)
            alpha_spectra = fltarr(256)
            cps_spectra = fltarr(256,nlevels)
            vt_spectra = fltarr(256,nlevels)
            raw_spectra = lonarr(256,nlevels)
            
            energy_spectra = red_energy_sorted
            
            kk = -1             ; Expanded array counter
            for jj = 0,nhbins-1 do begin ; nhbins counter
               for i = 1,red_width(jj) do begin
                  kk = kk + 1
                  alpha_spectra(kk) = red_alpha(jj)
                  cps_spectra(kk,*) = cps_2d_sorted(jj,*)
                  vt_spectra(kk,*) = vt_2d_sorted(jj,*)
                  raw_spectra(kk,*) = raw_2d_sorted(jj,*)
               endfor
            endfor
            
            
; Finally, bump the cps and vt rate up so that when it is 'byte'-ized, we
; will not truncate all of the values to 0 or 1 if they happen to be small
            
            bump_cps = 5000.
            bump_vt = 1000.
            bump_raw = 1.       ; No need to bump the raw counts
            cps_spectra = cps_spectra*bump_cps
            vt_spectra = vt_spectra*bump_vt
            raw_spectra = raw_spectra*bump_raw
            
;==========================Spectrogram================================
            
            pi=acos(-1)
;======File independent procedures=============
            if (i_block eq 0 and isweep eq 0) then begin
               
               winxsz = 750.
               winysz = 750.

; Want a 100 element color table where
;            0:96  = VIB G YOR (rainbow)
;              97  = white
;              98  = grey
;              99  = black
               
               n_rainbow_plus_white = 98 ; Number of colors defining rainbow + white, 0:97
               
               _extra = {ncolors:100}
               loadct_hydra, 18, _extra=_extra
               multi, 1.03      ;Want to define two colors of my own on the end
               top=  97         ;top is going to be white
               grey= top+1      ;grey is used for when cps = -999 (i.e. vt=0)
               black = top+2    ;black is used for when cps = 0
               tvlct, red, gre, blu, /get
               red(top)= 255 & gre(top)=255 & blu(top)=255
               red(grey)=140 & gre(grey)=140 & blu(grey)=140
               red(black)=0 & gre(black)=0 & blu(black)=0
; Redefine the first two blacks to be blue
               for i=0,1 do begin
                  red(i)=red(2)
                  gre(i)=gre(2)
                  blu(i)=blu(2)
               endfor
               tvlct, red, gre, blu
            ENDIF
            
;=============================================
            
            !p.multi=[0,1,1,0,0]
            !x.style=1
            !y.style=1
            !p.charsize=.75
            !p.background = top
            !p.font = -1
            !p.color = black
            !p.ticklen = -.01
            
;======Dummy axes for the spectrograms===========
            x=[1,2,3]
            y=[1,2,3]
            
            xmin = 0.
            xmax = 180.
            
            ymin = energy_spectra(0)
            ymax = 10^((alog10(energy_spectra(nlevels-1)) - $
                        alog10(energy_spectra(nlevels-2))) + alog10(energy_spectra(nlevels-1)))
            
            tickvec = fltarr(nlevels)
            for i = 0,nlevels-2 do begin
               tickvec(i) = 10^((alog10(energy_spectra(i+1)) - $
                                 alog10(energy_spectra(i)))/2. + alog10(energy_spectra(i)))
            endfor
            tickvec(nlevels-1) = 10^((alog10(energy_spectra(nlevels-1)) - $
                                      alog10(energy_spectra(nlevels-2)))/2. + alog10(energy_spectra(nlevels-1)))
            
            tickname = string(energy_spectra,'(i4)')
            
;========cps spectragram=============
            plot, x, y, /nodata,/ylog,position=[.07,.68,.85,.95],color=black,$
             yrange=[ymin,ymax],$
             xrange=[xmin,xmax],$
             yticks = nlevels-1,$
             ytickv = tickvec,$
             ytickname = tickname,xticks=18,xtickv=findgen(19)*10.,xtickname=strarr(19)+' '
; Please note that the only reason this is a representative scaling for
; the energy levels is because the levels increase approximately
; logarithmically.  If they didn't, you'd have to do something more
; sophisticated.
            
;!p.multi = !p.multi + [1,0,0,0,0]
            
; Define the data and device ranges established by the plot above;
; Remember that !y.crange is just the power of 10 when /ylog is
; used!!!
            
            xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
            lldata = [xdata(0), ydata(0)]
            sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
            
            ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
            ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
            sizedev=(ur-ll)(0:1)

            ll = ll_spec1
            sizedev = sizedev_spec

            
            gt_zero = where(cps_spectra gt 0)
            lt_zero = where(cps_spectra lt 0) ; When vt = 0 (cps=-999), grey
            zeros = where(cps_spectra eq 0) ; When raw_dist = 0, black
            
; Define a new array as the log of cps_spectra, retain both the 0 and
; -999 values for consistency
            cps_log = fltarr(256,nlevels)
            cps_log(gt_zero) = alog10(cps_spectra(gt_zero))
            if (zeros(0) ne -1) then cps_log(zeros) = 0
            if (lt_zero(0) ne -1) then cps_log(lt_zero) = -999
            
; Define the byte-scaled color array Zcolor to perform the 'tv' on
; Define the minimum:	real_min = real_cps_min
;			scaled_min = real_cps_min*bump_cps
;			log10(scaled_min)=alog10(real_cps_min*bump_cps)
            
            minZcolor = alog10(real_cps_min*bump_cps)
            
; Define the maximum:	real_max = real_cps_max
;			scaled_max = real_cps_max*bump_cps
;			log10(scaled_max)=alog10(real_cps_max*bump_cps)
            
            maxZcolor = alog10(real_cps_max*bump_cps)
            
            Zcolor = bytarr(256,nlevels)
            Zcolor(gt_zero) = bytscl(cps_log(gt_zero),max=maxZcolor, $
                                     min=minZcolor,top=top)
            if (lt_zero(0) ne -1) then Zcolor(lt_zero) = grey
            if (zeros(0) ne -1) then Zcolor(zeros) = black
            
;         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0), ll(1)
           
            Zcolor2 = congrid(Zcolor,sizedev(0),sizedev(1))
            
;         tv, Zcolor2, ll(0)/winxsz, ll(1)/winysz,/normal
            
            tv, Zcolor, ll(0)/winxsz, ll(1)/winysz,/normal,xsize=float(sizedev(0))/winxsz,ysize=float(sizedev(1))/winysz
            
            plot, x,y,/ylog,/nodata, /noerase,position=[.07,.68,.85,.95],$
             yrange=[ymin,ymax],$
             xrange=[xmin,xmax],$
             yticks = nlevels-1, $
             ytickv = tickvec,$
             ytickname=tickname,$
             ytitle='Energy, [eV]',$
             title='PPA '+ sblock_time + ' ' + $
             ' Counts/sec ( = Raw/Viewing Time; Grey where validb_flag=0)', color=black,xticks=18,xtickv=findgen(19)*10.,xtickname=strarr(19)+' '
            
;=================================================
            
; Now for the color bar (uuugh!)
            
; Create a 2-d array which depicts the color bar data
; There are n_rainbow_plus_white colors, ranging from navy (0) to white (97)
            
            Zcolor_bar = bytarr(10,n_rainbow_plus_white)
            for i = 0,9 do begin
               Zcolor_bar(i,*) = bindgen(n_rainbow_plus_white)
            endfor
            
; Create a plot for the axes
            plot, 	x, y ,/ylog,/nodata,position=[.93,.68,.97,.95],/noerase, $
             xrange = [0,6],xstyle=4,$
             yrange=[real_cps_min,real_cps_max],ystyle=1,ytickformat = '(f8.3)',$
             color=black
            
            xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
            lldata = [xdata(0), ydata(0)]
            sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
            
            ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
            ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
            sizedev=(ur-ll)(0:1)
            

            ll = ll_cb1
            sizedev = sizedev_cb


;         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0), ll(1)
;         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0)/winxsz, ll(1)/winysz,/normal         
            tv, Zcolor_bar, ll(0)/winxsz, ll(1)/winysz,/normal,xsize=float(sizedev(0))/winxsz,ysize=float(sizedev(1))/winysz
            
            plot, 	x, y ,/ylog,/nodata,position=[.93,.68,.97,.95],/noerase, $
             xrange = [0,6], xstyle=4,$
             yrange=[real_cps_min,real_cps_max], ystyle=1, ytickformat='(f8.3)', $
             title = 'Counts/sec',color=black
            
;========vt spectragram=============
            
            plot, x, y, /nodata,/noerase,/ylog,position=[.07,.06,.85,.33],color=black,$
             yrange=[ymin,ymax],$
             xrange=[xmin,xmax],$
             yticks = nlevels-1,$
             ytickv = tickvec,$
             ytickname = tickname,xticks=18,xtickv=findgen(19)*10.
            
; Define the data and device ranges established by the plot above;
; Remember that !y.crange is just the power of 10 when /ylog is
; used!!!
            
            xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
            lldata = [xdata(0), ydata(0)]
            sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
            
            ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
            ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
            sizedev=(ur-ll)(0:1)

            ll = ll_spec2
            sizedev = sizedev_spec
            
            gt_zero = where(vt_spectra gt 0)
            lt_zero = where(vt_spectra LT 0) ; Where validb_flag invalid
            zeros = where(vt_spectra eq 0) ; When vt = 0, black
            
; Define a new array as the log of vt_spectra
            vt_log = fltarr(256,nlevels)
            vt_log(gt_zero) = alog10(vt_spectra(gt_zero))
            if (zeros(0) ne -1) then vt_log(zeros) = 0
            IF (lt_zero(0) NE -1) THEN vt_log(lt_zero) = -999
            
; Define the byte-scaled color array Zcolor to perform the 'tv' on
; Define the minimum:	real_min = real_vt_min
;			scaled_min = real_vt_min*bump_vt
;			log10(scaled_min)=alog10(real_vt_min*bump_vt)
            
            minZcolor = alog10(real_vt_min*bump_vt)
            
; Define the maximum:	real_max = real_vt_max
;			scaled_max = real_vt_max*bump_vt
;			log10(scaled_max)=alog10(real_vt_max*bump_vt)
            
            maxZcolor = alog10(real_vt_max*bump_vt)
            
            Zcolor = bytarr(256,nlevels)
            Zcolor(gt_zero) = bytscl(vt_log(gt_zero),max=maxZcolor, $
                                     min=minZcolor,top=top)
            if (zeros(0) ne -1) then Zcolor(zeros) = black
            IF (lt_zero(0) NE -1) THEN Zcolor(lt_zero) = grey
            
;         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0), ll(1)
;         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0)/winxsz,
;         ll(1)/winysz,/normal 
            tv, Zcolor, ll(0)/winxsz, ll(1)/winysz,/normal,xsize=float(sizedev(0))/winxsz,ysize=float(sizedev(1))/winysz
            
            plot, x,y,/ylog,/nodata,/noerase,position=[.07,.06,.85,.33],$
             yrange=[ymin,ymax],$
             xrange=[xmin,xmax],$
             yticks = nlevels-1,$
             ytickv = tickvec,$
             ytickname = tickname,$
             ytitle='Energy, [eV]',$
             xtitle='Pitch Angle [degrees]',$
             title='PPA Probable Viewing Time [s] (Grey when validb_flag=0)', color=black,xticks=18,xtickv=findgen(19)*10.
            
;=================================================
; Now for the color bar (uuugh!)
            
; Create a 2-d array which depicts the color bar data
; There are n_rainbow_plus_white colors, ranging from navy (0) to
; white (97)
            
            Zcolor_bar = bytarr(10,n_rainbow_plus_white)
            for i = 0,9 do begin
               Zcolor_bar(i,*) = bindgen(n_rainbow_plus_white)
            endfor
            
; Create a plot for the axes
            plot, x, y ,/ylog,/nodata,position=[.93,.06,.97,.33],/noerase, $
             xstyle=4, xrange=[0,5],$
             yrange=[real_vt_min,real_vt_max], ytickformat='(f8.3)', ystyle=1, $
             color=black
            
            xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
            lldata = [xdata(0), ydata(0)]
            sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
            
            ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
            ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
            sizedev=(ur-ll)(0:1)

            ll = ll_cb2
            sizedev = sizedev_cb
            
;         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0), ll(1)
;         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0)/winxsz,
;         ll(1)/winysz,/normal
            tv, Zcolor_bar, ll(0)/winxsz, ll(1)/winysz,/normal,xsize=float(sizedev(0))/winxsz,ysize=float(sizedev(1))/winysz
            
            plot, x, y ,/ylog,/nodata,position=[.93,.06,.97,.33],/noerase, $
             xstyle=4, xrange = [0,5], $
             yrange=[real_vt_min,real_vt_max], ystyle=1, ytickformat = '(f8.3)', $
             title = 'Time [s]',color=black
            
;========raw spectragram=============
            plot, x, y, /nodata,/noerase,/ylog,position=[.07,.37,.85,.64],color=black,$
             yrange=[ymin,ymax],$
             xrange=[xmin,xmax],$
             yticks = nlevels-1,$
             ytickv = tickvec,$
             ytickname = tickname,xticks=18,xtickv=findgen(19)*10.,xtickname=strarr(19)+' '
            
; Define the data and device ranges established by the plot above;
; Remember that !y.crange is just the power of 10 when /ylog is
; used!!!
            
            xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
            lldata = [xdata(0), ydata(0)]
            sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
            
            ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
            ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
            sizedev=(ur-ll)(0:1)

            ll = ll_spec3
            sizedev = sizedev_spec
            
            gt_zero = where(raw_spectra gt 0)
            zeros = where(raw_spectra eq 0) ; When raw_dist = 0, black
            lt_zero = where(raw_spectra LT 0) ; Where validb_flag invalid

; Define a new array as the log of raw_spectra
            raw_log = fltarr(256,nlevels)
            raw_log(gt_zero) = alog10(raw_spectra(gt_zero))
            if (zeros(0) ne -1) then raw_log(zeros) = -1. ; To distinguish them from the log(1) values
            IF (lt_zero(0) NE -1) THEN raw_log(lt_zero) = -999
            
; Define the byte-scaled color array Zcolor to perform the 'tv' on
; Define the minimum:	real_min = real_raw_min
;			scaled_min = real_raw_min*bump_raw
;			log10(scaled_min)=alog10(real_raw_min*bump_raw)
            
            minZcolor = alog10(real_raw_min*bump_raw)
            
; Define the maximum:	real_max = real_raw_max
;			scaled_max = real_raw_max*bump_raw
;			log10(scaled_max)=alog10(real_raw_max*bump_raw)
            
            maxZcolor = alog10(real_raw_max*bump_raw)
            
            Zcolor = bytarr(256,nlevels)
            Zcolor(gt_zero) = bytscl(raw_log(gt_zero),max=maxZcolor, $
                                     min=minZcolor,top=top)
            if (zeros(0) ne -1) then Zcolor(zeros) = black
            IF (lt_zero(0) NE -1) THEN Zcolor(lt_zero) = grey
            
;         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0), ll(1)
;         tv, congrid(Zcolor,sizedev(0),sizedev(1)), ll(0)/winxsz, ll(1)/winysz,/normal         
            tv, Zcolor, ll(0)/winxsz, ll(1)/winysz,/normal,xsize=float(sizedev(0))/winxsz,ysize=float(sizedev(1))/winysz
            
            plot, x,y,/ylog,/nodata, /noerase,position=[.07,.37,.85,.64],$
             yrange=[ymin,ymax],$
             xrange=[xmin,xmax],$
             yticks = nlevels-1,$
             ytickv = tickvec,$
             ytickname = tickname,$
             ytitle='Energy, [eV]',$
             title='PPA Raw Counts (Grey where validb_flag=0)', color=black,xticks=18,xtickv=findgen(19)*10.,xtickname=strarr(19)+' '
            
;=================================================
; Now for the color bar (uuugh!)
            
; Create a 2-d array which depicts the color bar data
; There are n_rainbow_plus_white colors, ranging from navy (0) to white (97)
            
            Zcolor_bar = bytarr(10,n_rainbow_plus_white)
            for i = 0,9 do begin
               Zcolor_bar(i,*) = bindgen(n_rainbow_plus_white)
            endfor
            
; Create a plot for the axes
            plot, x, y ,/ylog,/nodata,position=[.93,.37,.97,.64],/noerase, $
             xstyle=4, xrange = [0,5], $
             yrange=[real_raw_min,real_raw_max], ystyle=1, $
             color=black
            
            xdata = !x.crange & ydata=10^(!y.crange) ;ranges in data space
            lldata = [xdata(0), ydata(0)]
            sizedata = [xdata(1)-xdata(0),ydata(1)-ydata(0)]
            
            ll = fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
            ur = fix(convert_coord(xdata(1),ydata(1),/data,/to_device))
            sizedev=(ur-ll)(0:1)
         
            ll = ll_cb3
            sizedev = sizedev_cb
            
;         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0), ll(1)
;         tv, congrid(Zcolor_bar,sizedev(0),sizedev(1)), ll(0)/winxsz,
;         ll(1)/winysz,/normal
            
            tv, Zcolor_bar, ll(0)/winxsz, ll(1)/winysz,/normal,xsize=float(sizedev(0))/winxsz,ysize=float(sizedev(1))/winysz
            
            plot, x, y ,/ylog,/nodata,position=[.93,.37,.97,.64],/noerase, $
             xstyle=4, xrange = [0,5],$
             yrange=[real_raw_min,real_raw_max], ystyle=1, $
             title = 'Raw Cnts.',color=black
            
            side_label = ' date='+date+', product_name='+ppa_info.product_name+', product_version='+ppa_info.product_version
            label_page, side_label=side_label
            message, side_label, /cont
            

            device, /close
            set_plot, 'X'

         ENDIF
         
         
      endfor
      
   endfor
   
;=====================================================================
   
   end_flag:

   end_of_prog
   
end
