pro paraboloid_fit, x_in, y_in, z_in, sigma_z_in, $
                    aprime=aprime_out, sigma_aprime=sigma_aprime_out, $
                    bprime=bprime_out, sigma_bprime=sigma_bprime_out, $
                    x0=x0_out, y0=y0_out, z0=z0_out, $
                    rchi2=rchi2_out, istat=istat_out, model=model
;+
; NAME: paraboloid_fit
;
; PURPOSE: Fit 'buckshot' surface data (x_i, y_i, z_i, sigma_z_i) to an
;          elliptic paraboloid of the form:
;          
;          z = z0 + aprime*(x-x0)^2 + bprime*(y-y0)^2
;
;          which is an elliptic paraboloid with minimum of
;          z0 located at (x0,y0).
;
; CATEGORY: Fitting
;
; CALLING SEQUENCE: 
;   paraboloid_fit, x_in, y_in, z_in, sigma_z_in, $
;                   aprime=aprime_out, sigma_aprime=sigma_aprime_out, $
;                   bprime=bprime_out, sigma_bprime=sigma_bprime_out, $
;                   x0=x0_out, y0=y0_out, z0=z0_out, $
;                   rchi2=rchi2_out, istat=istat_out, model=model
; 
; INPUTS: x_in, y_in, z_in, sigma_z_in    4, 1-D arrays of the buckshot
;                                         surface data
;	
; KEYWORD PARAMETERS: /model is a diagnostic keyword
;
; OUTPUTS: aprime,sigma_aprime,bprime,sigma_bprime,x0,y0,z0   Fit coefficients
;          rchi2       Reduced chi-squared of the fit
;          istat       Status code
;                        = 0, fit successful
;                        = 1, singular matrix, fatal error
;                        = 2, small pivot, fatal error
;
; COMMON BLOCKS: None
;
; EXAMPLE:
;aprime = 1.d0/10.d0^2
;bprime = 1.d0/20.d0^2
;x0 = 50
;y0 = 25
;z0 = 5
;a = aprime*x0^2+bprime*y0^2+z0
;b = -2.d0*aprime*x0
;c = -2.d0*bprime*y0
;d = aprime
;e = bprime
;ndata = 100
;x_in = (randomu(seed,ndata)*2.d0-1.d0)*200.d0
;y_in = (randomu(seed,ndata)*2.d0-1.d0)*200.d0
;z_in = a + b*x_in + c*y_in + d*x_in^2 + e*y_in^2
;sigma_z_in = make_array(ndata,/double,value=1.d0)
;paraboloid_fit, x_in, y_in, z_in, sigma_z_in, $
;  aprime=aprime_out, sigma_aprime=sigma_aprime_out, $
;  bprime=bprime_out, sigma_bprime=sigma_bprime_out, $
;  x0=x0_out, y0=y0_out, z0=z0_out, $
;  rchi2=rchi2_out, istat=istat_out
;print, aprime, aprime_out, sigma_aprime_out
;print, bprime, bprime_out, sigma_bprime_out
;print, x0, x0_out
;print, y0, y0_out
;print, z0, z0_out
;print, rchi2_out
;print, istat_out
;
; MODIFICATION HISTORY:  Written 7/25/00, Pamela A. Puhl-Quinn
;-

if (n_params() eq 0 and not keyword_set(model)) then begin
    doc_library, 'paraboloid_fit'
    return
endif

on_error, 2
model = keyword_set(model)
if (n_params() ne 4 and not model) then message, 'Incorrect number of arguments'

if n_elements(aprime_out) ne 0 then dum = temporary(aprime_out)
if n_elements(sigma_aprime_out) ne 0 then dum = temporary(sigma_aprime_out)
if n_elements(bprime_out) ne 0 then dum = temporary(bprime_out)
if n_elements(sigma_bprime_out) ne 0 then dum = temporary(sigma_bprime_out)
if n_elements(x0_out) ne 0 then dum = temporary(x0_out)
if n_elements(y0_out) ne 0 then dum = temporary(y0_out)
if n_elements(z0_out) ne 0 then dum = temporary(z0_out)
if n_elements(rchi2_out) ne 0 then dum = temporary(rchi2_out)
if n_elements(istat_out) ne 0 then dum = temporary(istat_out)

; MODEL=============================================
if model then begin
    aprime = 1.d0/10.d0^2
    bprime = 1.d0/20.d0^2
    x0 = 50
    y0 = 25
    z0 = 5
    a = aprime*x0^2+bprime*y0^2+z0
    b = -2.d0*aprime*x0
    c = -2.d0*bprime*y0
    d = aprime
    e = bprime
    ndata = 100
    x_in = (randomu(seed,ndata)*2.d0-1.d0)*200.d0
    y_in = (randomu(seed,ndata)*2.d0-1.d0)*200.d0
    z_in = a + b*x_in + c*y_in + d*x_in^2 + e*y_in^2
    sigma_z_in = make_array(ndata,/double,value=1.d0)
endif
;==================================================

ndata = n_elements(x_in)
g = double(z_in)
sigma_g = double(sigma_z_in)
f_0 = make_array(ndata,/double,value=1.d0)
f_1 = double(x_in)
f_2 = double(y_in)
f_3 = f_1^2
f_4 = f_2^2
sigma_coefs = 1
generalized_least_squares_fit, g, sigma_g, $
  5, f_0, f_1, f_2, f_3, f_4, b=coefs, sigma_b=sigma_coefs, rchi2=rchi2_out, $
  istat=istat_out

if (istat_out ne 0) then return

aprime_out = coefs(3) & sigma_aprime_out = sigma_coefs(3)
bprime_out = coefs(4) & sigma_bprime_out = sigma_coefs(4)
x0_out = -coefs(1)/(2.d0*aprime_out)
y0_out = -coefs(2)/(2.d0*bprime_out)
z0_out = coefs(0) - aprime_out*x0_out^2 - bprime_out*y0_out^2

if model then begin
    print, aprime, aprime_out, sigma_aprime_out
    print, bprime, bprime_out, sigma_bprime_out
    print, x0, x0_out
    print, y0, y0_out
    print, z0, z0_out
    print, rchi2_out
    print, istat_out
endif

return
end

