pro read_hydra_spec_l1, date_str, start_time, end_time, $
                        options=options, $
                        up=up, data_up=data_up, $
                        dn=dn, data_dn=data_dn, $
                        perp=perp, data_perp=data_perp, $
                        all=all, data_all=data_all, $
                        calc_all=calc_all, $
                        tres=tres, $ ; specify time resolution
                        range_up=up_range, $
                        range_perp=perp_range, $
                        range_dn=dn_range, $
                        gprogress=gprogress, $
                        include_detectors= include_detectors

   common get_error, get_err_no, get_err_msg

;  load data from level-one files.
;  History:
;    Jun 97.  Code taken from spectra utility. JBF

   myname= 'read_hydra_spec_l1'

; index space definitions to avoid future confusion
;   day space      0 is first record in the day
;   izoom space    0 is the first block returned to spectra
;   mode space     0 is the first block in a new mode (or edible data chunk)
   
   forward_function spectra_l1_correct
   forward_function spectra_l1_pitch

;  handle keywords and defaults
   calc_all= keyword_set( calc_all )
   if n_elements( up_range ) eq 0 then up_range=[0,30]
   if n_elements( perp_range ) eq 0 then perp_range=[75,105]
   if n_elements( dn_range ) eq 0 then dn_range=[150,180]   
   if n_elements( tres ) eq 0 then tres=1.15
   if n_elements( options ) eq 0 then options=0L

   if n_elements( include_detectors ) eq 0 then $
     include_detectors=intarr(12)  ; all zeroes indicates no filter.

;  begin code
   time_limit= getenv('HYDRA_L1_SPEC_TIME_LIMIT')
   if time_limit eq '' then time_limit = 9999 $ ; minutes
   else time_limit= long( time_limit )

   if end_time - start_time gt time_limit*60L then begin
       get_err_msg = 'To avoid rediculously long reads, data set is !c' + $
         'limited to '+strtrim(time_limit,2)+' minutes. The environment!c' + $
         'variable HYDRA_L1_SPEC_TIME_LIMIT specifies this time limit.'
       get_err_no= 1
       return
   endif

;  The header_checksum can be use to make certain that the header and
;  the stored data are from the same load.
   header_checksum= fix( ( systime(1) ) mod 7919 ) 
   
   header= { spec_header_2, $
             $                          
             start_time:start_time, $ ; T90
             end_time:end_time, $ ; T90
             $
             data:'', $    ; a place to put the name of the data variable name
             $               
             nen:fix(0), $
             mode_energy:fltarr(111,50), $
             reload_flag:0, $
             message_flag:0, $
             pitch_range:fltarr(2), $
             time_resolution:float(tres), $
             options:options, $
             include_detectors:include_detectors, $
             $             
             spec_type_id:0, $
             spec_type_name:' ', $                 
             $
             data_version:float(0), $ ; ddcal version or survey file version
             file1:' ', $       ; input file names
             file2:' ', $
             reader_version:float(0), $
             $                     
             calibration_version:float(0), $
             rgains_version:float(0), $
             igains_version:float(0), $
             glint_mask_version:float(0), $
             background_version:float(0), $
             mfe_calibration_version:float(0), $
             sc_potential_version:float(0), $
             intersweep_version:float(0), $
             $
             header_checksum:header_checksum $
           }

   data_values1= { hydra_spec_data, $
                   time:double(0), $ ; seconds since start_time
                   mode:byte(0), $ ; telemetry mode specifies energies
                   spectrum:fltarr(111), $
                   sigma:fltarr(111), $                   
                   species:byte(0), $ ; 1=electrons 2=ions 3=both
                   sc_pot:float(0), $
                   sc_pot_qual:byte(0), $
                   psi_status:byte(0), $
                   header_checksum:header_checksum $
                 }
   
   f= hydra_findfile( date_str, /level1 )
   if get_err_no ne 0 then return
   fcal= hydra_findfile( date_str, /ddcal, version=l1version )
   if get_err_no ne 0 then return
   
   if (fcal ne '' and f ne '') then begin
       CDF= cdf_open( f )
       CAL= cdf_open( fcal )
       get_err_no=0             ; tell papco all is well
   endif else begin
       get_err_msg='Input file not found.'
       get_err_no=1
   endelse

   message, 'Reading level1 file '+f, /cont
   message, 'Reading level1 calibration file '+fcal, /cont
   header.file1= f
   header.file2= fcal

;  get "stamp" info
   Hyd_cdf_varget, CDF, 'GLINT_MASK_VERSION', gmver
   header.glint_mask_version= gmver

   Hyd_cdf_varget, CDF, 'SC_POT_VERSION', scpot_vers
   header.sc_potential_version= scpot_vers

   header.data_version= l1version
   
   b_sec= tres
   message, 'Averaging for '+strtrim(b_sec,2)+' second resolution.', /cont

   izoom_start= long(start_time/b_sec)
   izoom_end= long(end_time/b_sec)+1
   n_blocks= izoom_end-izoom_start+1

   subblocks= fix( 13.8/b_sec +0.5 )
   n_blocks= n_blocks + subblocks*1
   recstart= izoom_start/subblocks
   reccount= n_blocks/subblocks+1
   nblocks= reccount*subblocks
   
;  get total number of blocks in the file
   cdf_control, CDF, variable='BLOCK_STATUS', get_var_info=r
   nrec= r.maxrec
   
;  get starting record and number of records
   hyd_cdf_varget, CDF, 'BLOCK_TIME', block_epoch, $
     rec_start=0, rec_count=nrec
   rectime= block_epoch mod 86400000 / 1000
   rtime= where( rectime le izoom_start*b_sec )
   if rtime(0) ne -1 then recstart= rtime(n_elements(rtime)-1) else recstart=0

   if (reccount gt nrec-recstart) then begin
       reccount= nrec-recstart
       nblocks= reccount*subblocks
   endif

   if reccount le 1 then begin
       get_err_msg='No data found at or after this time'
       print, get_err_msg
       get_err_no=1
       return
   endif

   cdf_control, CDF, variable='BLOCK_STATUS', get_var_info=r
   if (reccount gt r.maxrec-recstart) then begin
       reccount= r.maxrec-recstart
       nblocks= reccount*subblocks
   endif

   data_all= replicate( data_values1, nblocks )

   if calc_all then usesangles=1 else usesangles=0
   
; pitch cache files won't be used
   l1pitch=-1

   hyd_cdf_varget, CDF, 'BLOCK_STATUS', block_status, $
     rec_start=recstart, rec_count= reccount
   block_status=reform(block_status)

; poke in flags to make spectra process 45min at a time.
   r=indgen((reccount/200)+1)*200
   block_status(r)= block_status(r) or 2

; rnew is where mode or bias levels change
   rnew= where(( block_status and 2 ) ne 0 ) + recstart
   if rnew(0) eq -1 then rnew=0 ; else $
   nnew= n_elements(rnew)

;  get the overflow block numbers
   cdf_control, CDF, get_var_info=v, variable='OV_BLK_NUM'
   if (v.maxrec ge 0) then begin
       hyd_cdf_varget, CDF,'OV_BLK_NUM',ov_blk_num,rec_start=0,rec_count=v.maxrec+1
   endif else begin
       ov_blk_num=-1
   endelse

;  GET THE TIME-STAT FILE INFO
   hyd_cdf_varget, CDF, 'BLOCK_TIME', block_epoch, $
     rec_start=recstart, rec_count= reccount

   hyd_cdf_varget, CAL, 'BLOCK_TIME', block_epoch_cal, $
     rec_start=recstart, rec_count= reccount
   
   if total( abs( block_epoch - block_epoch_cal ) ) gt 0.1 then begin
       message, 'Calibration and Raw Data files out-of-sync.', /cont
       message, 'Consult Jeremy Faden, jbf@hydra.physics.uiowa.edu', /cont
       message, 'before using.  (.continue will proceed as before.)', /cont
       stop
   endif

   block_time0= fltarr(reccount)
   block_time0(*)= ( block_epoch(*) mod (86400*1000L) ) / 1000

   block_time0_= rebin( block_time0, nblocks, /sample )
   r=indgen(reccount)*subblocks
   for isub=1,subblocks-1 do $
     block_time0_(r+isub)= block_time0_(r+isub) + isub*b_sec

;  make itime array
   itimeoff= long(block_time0_(0)/b_sec)
   itime= long(block_time0_/b_sec)-itimeoff

   r= where( itime lt nblocks)  ; might pull off too many
   itime0= long(block_time0/b_sec)-itimeoff  ; store for loop later

   itime=itime(r)


   icont=indgen(n_elements(itime)) ; continuous indeces

;  now fill the arrays
   data_all(itime).time= (temporary(block_time0_))(icont)
   
;  fill invalid blocks with valid time.
   rt= where( data_all(*).time eq 0.0 )
   bsec= tres
   for i=0L,n_elements(rt)-1 do begin
       if rt(i) gt 0 then data_all(rt(i)).time= $
         data_all(rt(i)-1).time+bsec
   endfor

   hyd_cdf_varget, cal, 'BLOCK_MODE', block_model1, $
     rec_start=recstart, rec_count= reccount
   block_var= rebin(block_model1,nblocks, /sample)
   block_mode0= intarr(nblocks)
   data_all(itime).mode= block_var(icont)

   hyd_cdf_varget, cal, 'SC_POT', block_var, $
     rec_start=recstart, rec_count= reccount
   r= where(block_var eq -1e-31)
   if r(0) ne -1 then block_var(r)=1e31 ; fill
   block_var= rebin( reform( block_var ), nblocks, /sample )
   block_potential0= make_array( nblocks, /float, value=1e31 )
   block_potential0(itime)= block_var(icont)
   data_all(itime).sc_pot= block_var(icont)
   
   hyd_cdf_varget, cal, 'POT_QUALITY', pot_quality, $
     rec_start=recstart, rec_count= reccount
   data_all(itime).sc_pot_qual= pot_quality(icont)

   hyd_cdf_varget, cal, 'SWEEP_POT_MAX', hr_sc_pot, $
     rec_start=recstart, rec_count= reccount

;  How close can the s/c potential be before we decide to throw out
;  the point?  ( sc_pot / pot_fraction ) > observed_energy !
   hyd_cdf_varget, cal, 'POT_FRACTION', pot_fraction

;  Copy the data into structures for each angle range.
   if calc_all then begin
       data_up= data_all
       data_dn= data_all
       data_perp= data_all
   endif

;  CHECK TO SEE IF MFE HIGH RES BFIELD DATA EXISTS IN L1 FILE
   cdf_control, CDF, var='B_HIGH_RAW', get_var_info=r
   if n_elements( r ) gt 0 then begin
       if r.maxrec gt -1 then begin
           highres=1
       endif else begin
           highres=0
       endelse
   endif

;  GET ANGLES INFORMATION
   hyd_cdf_varget, CDF, 'PHASE_START', phase_start, $
     rec_start=recstart, rec_count=reccount
   phase_start= reform(phase_start)

;  GET OVERFLOW BLOCKS
   if (ov_blk_num)(0) ne -1 then begin
       hyd_cdf_varget, CDF, 'OV_INDEX', ov_index
       hyd_cdf_varget, CDF, 'OV_BLK_NUM', rec_count=ov_index, ov_block_num
   endif else begin
       ov_block_num=-1
   endelse

;  AVERAGE THE DATA
   if n_elements( gprogress ) gt 1 then begin ; progress graphic 
       if !d.name ne 'PS' then begin
           plot, [0.05,0.95], [.5,.5], $
             xrange=[0,1], xstyle=5, $
             yrange=[0,1], ystyle=5, $
             color=gprogress(4), $
             position=gprogress(0:3), /normal
           gprogstr= 'Reading ' + strtrim( reccount,2 ) + ' data blocks:'
           xyouts, 0.05, .6, gprogstr, color=gprogress(4)
       endif
   endif
   
   nen= 111
   mode_length= intarr(51)

   aligned_pr= up_range * !dtor
   opposed_pr= dn_range * !dtor
   perp_pr= perp_range * !dtor
   
   for imode= 0, nnew-1 do begin

       rnewi= rnew(imode)
       recstartoff= rnewi-recstart ; goes from mode space to izoom space

       modei= block_model1(recstartoff)

       hyd_cdf_varget1, CDF, 'BLOCK_INDEX', rec_start=rnewi, offset
       offset= offset-1

       hyd_cdf_varget, CAL, 'LENGTH', length, rec_start= modei-1
       
       hyd_cdf_varget, CAL, 'TIME_OFFSET', time_offset, rec_start=modei

       hyd_cdf_varget, CAL, 'CAL_ENERGY_INDEX', energy_index, rec_start= modei-1

       rele= where( energy_index(*,0) lt 0 )
       rion= where( energy_index(*,0) gt 0 )

       mode_length( modei ) = length

       hyd_cdf_varget, CAL, 'CAL_ENERGY_SEQ_BOX', energy_seq, rec_start= modei-1

       hyd_cdf_varget, CDF, 'ELE_BIAS', elebiasi, rec_start= rnewi
       hyd_cdf_varget, CDF, 'ION_BIAS', ionbiasi, rec_start= rnewi

       counts_field= string(modei,format='(i3.3)')+'_e_'+ $
         string(elebiasi(0),format='(i2.2)')+'_'+ $
         string(elebiasi(1),format='(i2.2)')+'_i_'+ $
         string(ionbiasi(0),format='(i2.2)')+'_'+ $
         string(ionbiasi(1),format='(i2.2)')

       if imode eq nnew-1 then nrecmode=recstart+reccount-rnewi else $
         nrecmode=rnew(imode+1)-rnewi

;      more stamp accounting
       Hyd_cdf_varget, CAL, 'ELE_RGAIN_VERSION_'+counts_field, rgains_vers
       Hyd_cdf_varget, CAL, 'ELE_IGAIN_VERSION_'+counts_field, igains_vers
       Hyd_cdf_varget, CAL, 'CAL_ENERGY_VERSION', isweep_version, rec_start=modei-1
       header.intersweep_version= $
         min( [ min( isweep_version ), header.intersweep_version ] )
       header.rgains_version= min([ rgains_vers, header.rgains_version ])
       header.igains_version= min([ igains_vers, header.igains_version ])
       
       print, ' '
       print, 'averaging ('+strtrim(rnewi-recstart,2)+'/'+$
         strtrim(reccount,2)+')...', format='(A,$)'

       hyd_cdf_varget, CDF, 'COUNTS_'+counts_field, counts, $
         rec_start=offset, rec_count=nrecmode

       if usesangles and (l1pitch eq -1)  then begin
           gen_bfield, cdf, cdfcal=cal, rnewi, nrecmode, $
             /hydtim, /thetaphi, $
             btheta= thetab, bphi= phib, bvalid=bvalid, /lag_correct
           n= n_elements( thetab )/384
           thetab= reform( thetab, 384, n )
           phib= reform( phib, 384, n )
           bvalid= reform( bvalid, 384, n )
       endif

       if ( usesangles and (l1pitch ne -1) ) then begin
           hyd_cdf_varget, l1pitch, 'PITCH_ANGLE', ialpha, $
             rec_start=rnewi, rec_count=nrecmode
       endif

       split= modei eq 10 
       energy_align=0
       nmode6=0
       if modei eq 10 then begin
           spectra_l1_recalc_energy_index, energy_index, energy_seq
           if subblocks lt 6 then begin
               energy_align=1
               spectra_l1_1x2x_energy_align, /init, $
                 energy_index=energy_index, $
                 energy_seq=energy_seq
           endif else begin
               nmode6= 1
           endelse
       endif
       
       spectra_l1_average, /init, $
         energy_index=energy_index, energy_seq=energy_seq, $
         split=split, intdata=intdata, time_offset=time_offset, $
         nmode6= nmode6

       nen= n_elements( intdata(0).energy_interp )
       r55= indgen(nen) + 55 - (nen/2)

       header.mode_energy( r55, modei-1 ) = $
         10. ^ intdata(0).energy_interp( * )
       header.mode_energy( 0:54, modei-1 ) = $
         -1.0 * header.mode_energy( 0:54, modei-1 )
       header.mode_energy( 55, modei-1 ) = 0.0 ; very important!
       
       r= spectra_l1_correct( /init, rawid=CDF, calid=CAL, $
                              mode=modei, $
                              ele_bias=elebiasi, ion_bias=ionbiasi, $
                              /sigma_cc )
       
       r= spectra_l1_pitch( /init,  $
                            mode=modei, rele=rele, rion=rion, $
                            cdfcal= CAL )
       
; average each block...

       for ii= 0, nrecmode-1 do begin
           print, '.', format="(A,$)"

           if n_elements(gprogress) gt 1 and !d.name ne 'PS' then begin
               dd= float( ii + rnewi - recstart ) / ( ( reccount - 1 ) > 1 )
               oplot, [0.05,dd*0.9+0.05], [.5,.5], color=gprogress(5), thick=3.
               empty            ; empty graphics buffer
           endif

           icounts= counts(*,*,ii)
           datamask= ( (counts(*,*,ii) and -16384) eq 0 )

           if total(include_detectors) ne 0 then begin
               r= where( include_detectors eq 0 )
               if r(0) ne -1 then datamask(*,r)=0
           endif

;            sc_pot filter code was here!

           if ( block_status(ii+recstartoff) and 1 ) then begin ; overflow
               icounts=long(icounts)
               me=where(ov_block_num eq ii+rnewi+1) ; ii to day space
               for i=0, n_elements(me)-1 do begin
                   hyd_cdf_varget, CDF, 'OV_STEP_NUM', rec_start=me(i), st
                   hyd_cdf_varget, CDF, 'OV_DET_NUM', rec_start=me(i), det
                   hyd_cdf_varget, CDF, 'OV_COUNTS', rec_start=me(i), ovcounts
                   icounts(st-1,det-1)=ovcounts
               endfor
           endif

           counts1= spectra_l1_correct( icounts, ii+rnewi, sigma_cc= dcounts1 )

           if energy_align then begin
               spectra_l1_1x2x_energy_align, counts1, datamask
           endif

           if ( usesangles ) then begin
               if ( l1pitch eq -1 ) then begin ;calculate pitch
                   pitch= spectra_l1_pitch( $
                                            phase_start(ii+recstartoff),$
                                            thetab(*,ii),phib(*,ii), $
                                            bvalid(*,ii) $
                                          )
               endif else begin ; read it from l1 cache file
                   pitch= (float( ialpha(*,*,ii) )+0.5)*!pi/256
               endelse
           endif

;         calculate the four spectrograms

;         all
           spectra_l1_average, counts1, datamask, subblocks, $
             all_out, energy_seq=energy_seq

           if calc_all then begin
;              aligned
               datamask1= datamask
               r= where( pitch lt aligned_pr(0) or pitch gt aligned_pr(1) )
               if r(0) ne -1 then datamask1(r)= 0
               spectra_l1_average, counts1, datamask1, subblocks, $
                 sigma_counts= dcounts1, $
                 aligned_out, sigma_spec_out= sigma_aligned_out, energy_seq=energy_seq
               
;              opposed
               datamask1= datamask
               r= where( pitch lt opposed_pr(0) or pitch gt opposed_pr(1) )
               if r(0) ne -1 then datamask1(r)= 0
               spectra_l1_average, counts1, datamask1, subblocks, $
                 sigma_counts= dcounts1, $
                 opposed_out, sigma_spec_out= sigma_opposed_out, energy_seq=energy_seq
               
;              perp
               datamask1= datamask
               r= where( pitch lt perp_pr(0) or pitch gt perp_pr(1) )
               if r(0) ne -1 then datamask1(r)= 0
               spectra_l1_average, counts1, datamask1, subblocks, $
                 sigma_counts= dcounts1, $
                 perp_out, sigma_spec_out= sigma_perp_out, energy_seq=energy_seq
               
           endif
           
           itime= itime0[ii+recstartoff]
           
           if itime lt n_elements(data_all) then begin
               
               data_all( itime+indgen(subblocks) ).spectrum(r55) = all_out
               
               if calc_all then begin
                   data_up( itime+indgen(subblocks) ).spectrum(r55) = $
                     aligned_out
                   data_up( itime+indgen(subblocks) ).sigma(r55) = $
                     sigma_aligned_out
                   data_dn( itime+indgen(subblocks) ).spectrum(r55) = $
                     opposed_out
                   data_dn( itime+indgen(subblocks) ).sigma(r55) = $
                     sigma_opposed_out
                   data_perp( itime+indgen(subblocks) ).spectrum(r55) = $
                     perp_out
                   data_perp( itime+indgen(subblocks) ).sigma(r55) = $
                     sigma_perp_out
               endif
               
               if ( pot_quality(recstartoff+ii) and 2 ) eq 2 then begin
                   if subblocks eq 1 then begin
                       scpot_avg= $
                         total( hr_sc_pot( *, recstartoff+ii ) ) / 12 + 2.0
                       data_all( itime ).sc_pot= scpot_avg                     
                       if calc_all then begin
                           data_up( itime ).sc_pot= scpot_avg
                           data_dn( itime ).sc_pot= scpot_avg
                           data_perp( itime ).sc_pot= scpot_avg
                       endif
                   endif else begin
                       r= indgen( subblocks ) * ( 12 / subblocks )
                       sc_pot= hr_sc_pot( r, recstartoff+ii ) + 2.0
                       data_all( itime+indgen(subblocks) ).sc_pot= sc_pot
                       if calc_all then begin
                           data_up( itime+indgen(subblocks) ).sc_pot= sc_pot
                           data_dn( itime+indgen(subblocks) ).sc_pot= sc_pot
                           data_perp( itime+indgen(subblocks) ).sc_pot= sc_pot
                       endif
                   endelse
               endif
           endif ; itime lt n_elements( data_all )
       endfor
   endfor

   cdf_close, CDF
   cdf_close, CAL
   if (l1pitch ne -1) then cdf_close, l1pitch

   if n_elements( gprogress ) gt 1 and !d.name ne 'PS' then begin
       xyouts, 0.05, 0.6, gprogstr, color=0
       oplot, [0.05,0.95], [0.5,0.5], color=0, thick=3.
   endif

   header.reader_version= hydra_code_version( /spec )
   
   all= header
   all.pitch_range=[0,180]

   if calc_all then begin
       up=header
       up.pitch_range= up_range
       
       dn=header
       dn.pitch_range= dn_range
       
       perp=header
       perp.pitch_range= perp_range
   endif

   return
end

;           if ( pot_quality(recstartoff+ii) and 2 ) eq 2 then begin
;               sc_pot_en= rebin( hr_sc_pot( *, recstartoff+ii ), 384, /sample )
;               r= where( energy_seq( rele, 0 ) gt $
;                         -sc_pot_en(rele)/pot_fraction )
;               if r(0) ne -1 then begin
;                   ien_mask= rele(r)
;                   for idet=0,10,2 do datamask( ien_mask, idet )= 0
;               endif
;               r= where( energy_seq( rele, 1 ) gt -sc_pot_en(rele) )
;               if r(0) ne -1 then begin
;                   ien_mask= rele(r)
;                   for idet=1,11,2 do datamask( ien_mask, idet )= 0
;               endif
;           endif
