;***************************************************************************
;* file hydra_fv_slice.pro
;*
;* this files contains the routines needed to enable the extra mouse
;* feautures provided for in PAPCO. Mouse zooms is part of main papco.
;* Currently possible:
;*
;*    mouse views slice
;*    mouse views slice (polar)
;*    mouse accesses level zero
;*
;* Again here the procedure names must adhere to a strict naming convention so
;* that they can be automatically called when needed.
;*
;* The slice types supported by these routines must be ENABLED by the user, so
;* that PAPCO will call them. This is done by setting the value of
;* slice_type in draw_plot_type.pro (contained in papcoadd_plot_type.pro).
;* slice_type is defined in the following way:
;*
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 switches. Currently
;*                      Three are supported (this may be extended later)
;*                       1 = mouse zooms (default, always possible)
;*                       2 = mouse slices
;*                       4 = mouse calls level zero
;*                       8, 16, 32, 64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arythmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* The mouse does slice function uses the time returned by PAPCO and the
;* get_data_call fvified for the plot_type to read the data and to draw a
;* slice at the clicked position. This is commonly used in fvtra color plots
;* to draw a fvtrum at that time, or a pitch angle distribution etc, or to
;* call up any other plot at that time as fvified by the user.
;*
;* The mouse accesses level zero uses the time passed and calls up another
;* widget application to read and work with level zero data. This application
;* must be intirely independant of PAPCO and is only called from PAPCO via
;* a mouse click in the panel required for the time required.
;*
;* If your new data contains several sub-data types, one slice and/or level zero
;* routine has to be provided for each type, since in general they would do
;* different things. See $PAPCO_MODULES/crres_slices.pro for such an example.
;*
;*    PRO DrawSlice_hydra_fv, panelNr, seconds, yValue
;*        -> Draw a slice for hydra_fv
;*
;*    PRO levelzero_hydra_fv, seconds, yvalue
;*        -> This procedure calls the widget for level zero data acces
;*           for this plottype.
;*
;* MODIFICATION HISTORY:
;*    December 1995, written by R.Friedel at Goddard
;*     april 1996, R. Friedel
;*
;***************************************************************************

;***************************************************************************
;*
;* PROCEDURE:
;*      PRO DrawSlice_hydra_fv, panelNr, seconds, yValue
;*
;* DESCRIPTION:
;*	Draw a slice for hydra_fv, providing default functionality and
;*      calling a seperate window for the plot.
;*      slice_type in draw_plot_type must be set to +2 to enable this feature.
;*
;* INPUTS:
;*	panelNr	 	an integer
;*			the number of the panel for which the slice is to be
;*			drawn
;*	seconds		time of mouse click, a float
;*	yvalue		y-value of mouse click, a float
;*			the time and the y-value, for that the slice shall
;*			be drawn
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by PAPCO_DrawSLice (in papco_cursor)
;*
;* MODIFICATION HISTORY:
;*     april 1996, R. Friedel
;***************************************************************************
PRO DrawSlice_hydra_fv, panelNr, seconds, yValue

   COMMON PAPCO_DRAWSLICE, sliceWidgetData
   COMMON PLOT_COMPOSER, widgetData
   common papco_hydra_slice_control, hydra_sliceWidgetData

   common time, xut1, xut2

; add in here and other common blocks that you need to make your plot data
; available here. The common block included here is a dummy for illustration.
; t90 is the time array in t90 format.
; PAPCO uses the the getdatacall you set in papco_add_plot_type to read the
; data you need for the slice. As long as you then stay in one panel with the
; mouse, the data is not read again. When you switch panels, the new data is
; read.

   return

   common hydra_fv_data, $
     data_context, $            ; selects current data set
     fv_type_names, $
     fv_header, $
     fv_data, $
     fv_aligned_data, $
     fv_perp_data, $
     fv_opposed_data, $
     fv_skew_data, $
     hr_fv_data, $
     hr_fv_aligned_data, $
     hr_fv_perp_data, $
     hr_fv_opposed_data, $
     hr_fv_skew_data

   if seconds lt 0 or seconds gt (xut2-xut1) then begin
       erase
       message= 'Invalid Time Generated by PaPCo slice pointer. !c'+ $
         'Suggest you reDRAW then try again.'
       xyouts, 0.5, 0.5, align=0.5, message, charsize=1.5, /normal
       return
   endif

   convert_T90_to_date, xut1, year, doy, hour, minute, second, $
     month, cmonth, day_om

   date_str= string( year, format='(i4.4)' ) + $
     string( month, format='(i2.2)' ) + $
     string( day_om, format='(i2.2)' )

   if n_elements( hydra_sliceWidgetData ) eq 0 then begin
       hydra_sliceWidgetData= { hydra_sliceWidgetData_struct, $
                                type:fix(0) $
                              }
   endif


;   WIDGET_CONTROL,/HOURGLASS
   thisPlot=widgetData.plotsDrawn(panelNr)

; make same data plot array as used in p_hydra_fv

   thisPlot=widgetData.plotsDrawn(panelNr)
   type=thisPlot.typeVector

; set the panel window for the slice to full window

   !p.position=[0.15,0.15,0.9,0.9]

   data_context= hyd_dc( thisPlot.typevector(1), thisPlot.typevector(2) )
   exe_str='data=' + fv_type_names(data_context)+'_data'
   r=execute( exe_str )

   t90= data(*).time + fv_header(data_context).start_time - xut1

; pick closest block
;  this is kludgy

   rvalid= where( data(*).mode ne 0 and $
                  ( ( data(*).fvies and type(0)+1 ) eq type(0)+1 ) )

   if rvalid(0) eq -1 then return

   dd= seconds - t90(rvalid)
   r= where(dd ge 0)
   if r(0) ne -1 then ishow = rvalid(r(n_elements(r)-1)) else return

   if dd(r(n_elements(r)-1)) gt 13.8 then return

   mode= data(ishow).mode
   energy= fv_header(data_context).mode_energy(*,mode-1)
   fvtrum= data(ishow).fvtrum
   time= data(ishow).time + fv_header(data_context).start_time

   case type(0) of
       0:begin			;electrons
           fvies=-1
           rfv= where( energy lt 0 )
           energy= -reverse(energy(rfv))
           fvtrum= reverse(fvtrum(rfv))
           ctit='POLAR/HYDRA Electron Energy Fvtrum'
           xtitle='Observed Electron Energy, eV'
           ytitle='Corrected Counts / Energy!e2'
           yst=1e-10 & yen=1e1
       end
       1:begin			;ions
           fvies=1
           rfv= where( energy gt 0 )
           energy= energy(rfv)
           fvtrum= fvtrum(rfv)
           ctit='POLAR/HYDRA Ion Energy Fvtrum'
           xtitle='Observed Ion Energy, eV'
           ytitle='Corrected Counts / Energy!e2'
           yst=1e-11 & yen=1e-1
       end
   endcase

   onecount= fvtrum*0.0+1.0

   common hydra_user_configure_common, user_configuration_data, $
     user_configure_data_tmp

   if user_configuration_data.fv_shift then begin
       sc_pot= data(ishow).sc_pot
       logf= ( fvtrum / energy^2 )
       onecount=( onecount/energy^2 )
       energy= energy + (fvies*sc_pot)
       xtitle='Debye Energy, eV'
       xrange=[ ([1,0,10])(fvies+1),20000]
   endif else begin
       logf= fvtrum/energy^2
       onecount= onecount/energy^2
       xtitle='Observed Energy, eV'
       xrange=[10,20000]
   endelse

   r=where( fvtrum eq 1e-20, n_nodata )
   if n_nodata gt 0 then fvtrum(r) = -1

   circx= cos( findgen(21)/20 * 2 * !pi )
   circy= sin( findgen(21)/20 * 2 * !pi )
   usersym, circx, circy

   erase
   !x.ticks=0
   !x.minor=0
   !y.ticks=0
   !y.minor=0

   r=where( fvtrum gt 0.01 )
   if r(0) ne -1 then begin
       plot,energy(r),logf(r),$
         xtitle=xtitle,ytitle=ytitle, title=ctit, $
         xrange=xrange, xstyle=1, $
         yrange=[yst,yen], ystyle=1, $
         /xlog, /ylog, $
         color=1, psym=-8
       oplot, energy, onecount, linestyle=1
   endif else begin
       plot, energy, onecount, linestyle=1
   endelse

; these two info strings are user definable ouput strings to label the slice
; window.

   convert_t90_to_Date,time, year, day_oy, hour, min, sec, $
     month, cmonth, day_om

; convert_t90_to_Date trucates the seconds rather than rounding them.
   start_time= hydra_hms( time, secformat='(f5.2)' )
   end_time= hydra_hms( time+fv_header(data_context).time_resolution, $
                        secformat='(f5.2)' )

   info1=string(day_om,cmonth,year,format="('Date: ',i2.2,' ',a3,' ',i2.2,' ')")

   xyouts, 0.64, 0.87, info1, /normal, color=1
   xyouts, 0.64, 0.84, start_time+'-'+end_time, /normal, color=1

   xyouts, 0.64, 0.81, /normal, color=1, $
     'S/C Pot: '+strtrim( string(data(ishow).sc_pot,format="(f6.2)"),2 )

; Replace these labels and include your plot!

; user defined routine for plotting the slice included here. An example is in
; $PAPCO_MODULES/panels&slices/crres_slices.pro
; You typically need to include the commons from the read routine, a way
; of finding the index of the data you want to plot using the time "seconds"
; (in t90 format of course), plus a simple plot statement to plot the data.
; Two strings, info1 and info2, can be used to annotate the slice window
; by providing additional information, ie ephemeris data, etc.

   return

END


;***************************************************************************
;*
;* PROCEDURE:
;*      PRO DrawPolarSlice_hydra_fv, panelNr, seconds, yValue
;*
;* DESCRIPTION:
;*	Draw a slice for hydra_fv, providing default functionality and
;*      calling a seperate window for the plot.
;*      slice_type in draw_plot_type must be set to +2 to enable this feature.
;*
;* INPUTS:
;*	panelNr	 	an integer
;*			the number of the panel for which the slice is to be
;*			drawn
;*	seconds		time of mouse click, a float
;*	yvalue		y-value of mouse click, a float
;*			the time and the y-value, for that the slice shall
;*			be drawn
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by PAPCO_DrawSLice (in papco_cursor)
;*
;* MODIFICATION HISTORY:
;*     april 1996, R. Friedel
;***************************************************************************
PRO DrawPolarSlice_hydra_fv, panelNr, seconds, yValue

   COMMON PAPCO_DRAWSLICE, sliceWidgetData
   COMMON PLOT_COMPOSER, widgetData
   common papco_hydra_slice_control, hydra_sliceWidgetData

   common time, xut1, xut2

   COMMON PLOT_COMPOSER, widgetData
   common papco_hydra_slice_control, hydra_sliceWidgetData

; add in here and other common blocks that you need to make your plot data
; available here. The common block included here is a dummy for illustration.
; t90 is the time array in t90 format.
; PAPCO uses the the getdatacall you set in papco_add_plot_type to read the
; data you need for the slice. As long as you then stay in one panel with the
; mouse, the data is not read again. When you switch panels, the new data is
; read.

   thisPlot=widgetData.plotsDrawn(panelNr)
   species=thisPlot.typeVector(0)

   t90= double(xut1)+double(seconds)
   hydra_fv_control, t90, species=species
END



;***************************************************************************
;*
;* PROCEDURE:
;*      PRO DrawSlice_hydra_fv_survey, panelNr, seconds, yValue
;*
;* DESCRIPTION:
;*	Draw a slice for hydra_fv, providing default functionality and
;*      calling a seperate window for the plot.
;*      slice_type in draw_plot_type must be set to +2 to enable this feature.
;*
;* INPUTS:
;*	panelNr	 	an integer
;*			the number of the panel for which the slice is to be
;*			drawn
;*	seconds		time of mouse click, a float
;*	yvalue		y-value of mouse click, a float
;*			the time and the y-value, for that the slice shall
;*			be drawn
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by PAPCO_DrawSLice (in papco_cursor)
;*
;* MODIFICATION HISTORY:
;*     april 1996, R. Friedel
;***************************************************************************
PRO DrawSlice_hydra_fv_survey, panelNr, seconds, yValue

  DrawSlice_hydra_fv,panelNr, seconds, yValue

end


;***************************************************************************
;*
;* PROCEDURE:
;*      PRO levelzero_hydra_fv, seconds, yvalue
;*
;* DESCRIPTION:
;*	This procedure calls the widget for level zero data acces
;*      for this plottype.
;*      slice_type in draw_plot_type must be set to +4 to enable this feature.
;*
;* INPUTS:
;*	seconds		t90 time at which mouse is clicked
;*	yvalue		y-value at which mouse is clicked
;*
;* OUTPUTS:
;*      none
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	levelzero_hydra_fv, seconds, yvalue
;*
;* MODIFICATION HISTORY:
;*     april 1996, R. Friedel
;*
;***************************************************************************
pro levelzero_hydra_fv, seconds, yvalue

common time, xut1, xut2

; The actual procedure is very dependent on what the level zero widget
; application needs. An example is in $PAPCO_MODULES/addpapco_swe_plots.pro.
; Provided as paramters to be passed to the new application are the values
; of the mouse pointer when clicked.
;
; The main widget remains active and can be switched to (e.g. to get a new
; time off the graph) by using a "Parent" button in the new application
; which then executes the line
;            current_appl=xregistered('PAPCO_main')
; to switch back to the PAPCO main calling widget.

t90= xut1+seconds

hydra_fv_control, t90

END

PRO user_hydra_fv_slice_hydra_fv, panelNr, seconds, yValue, event=e, readdata=readdata

   COMMON PAPCO_DRAWSLICE, sliceWidgetData
   COMMON PLOT_COMPOSER, widgetData
   common papco_hydra_slice_control, hydra_sliceWidgetData

   common time, xut1, xut2

   COMMON PLOT_COMPOSER, widgetData
   common papco_hydra_slice_control, hydra_sliceWidgetData

; add in here and other common blocks that you need to make your plot data
; available here. The common block included here is a dummy for illustration.
; t90 is the time array in t90 format.
; PAPCO uses the the getdatacall you set in papco_add_plot_type to read the
; data you need for the slice. As long as you then stay in one panel with the
; mouse, the data is not read again. When you switch panels, the new data is
; read.

   thisPlot=widgetData.plotsDrawn(panelNr)
   species=thisPlot.typeVector(0)

   t90= double(xut1)+double(seconds)
   hydra_fv_control, t90, species=species, group= papco_getDrawBase()
END

