;+
; NAME: hydra_COLOR_BAR
;
; PURPOSE:
; This IDL-procedure plots a colorbar on the right side of the screen
;	
; CATEGORY:
;       Plotting (Color)
;
; CALLING SEQUENCE:
;
;       hydra_COLOR_BAR,ZMIN,ZMAX,[ztitle], zlog=zlog, $
;          bipolar=bipolar
;
; INPUTS:     in accordance with PLOT_CLR
;             ZMIN     float bottom value of color axis
;             ZMAX     float top value of color axis
;
;	
; PROCEDURE:  The color sequence is (and must be) linear, even for
;	      logarithmic z-scaling.
;             Axes are drawn using the AXIS- and LOGAXIS-procedures.
;             Viewport and data-screen relation are restored at the end.
;
; MODIFICATION HISTORY:
;       written jan 92 M.Fraenz,N.Krupp
;       customized (generalized, really) for hydra 9/20/96 by J.B.Faden
;
PRO hydra_COLOR_BAR, ZMIN,ZMAX,ztitle,zlog=zlog, bipolar=bipolar

  common papco_color, papco_color, r_papco, g_papco, b_papco

  IF N_PARAMS(0) LT 3 THEN ztitle=''
   zlog= keyword_set(zlog)
   bipolar= keyword_set(bipolar)
   
; find colorbar range
  cindex= where( papco_color(*,2) eq 1 )
  if (cindex(0) eq -1) then begin
      print, 'Color system failure, consult jbf@space-theory.physics.uiowa.edu'
      print, 'Using color set 0'
      cindex=0
  endif
  color_range= PAPCO_get_Colorindices( cindex )
  cbase= color_range(0)
  clen= color_range(1)-color_range(0)
  ctop= color_range(1)
  
; save current viewport and data-screen-relation:
   pa=!p
   xa=!x
   ya=!y
   colrat=clen/255.0
   !x.title=''
   !p.TITLE=''
   !y.title=''
   !p.ticklen=-.2

   scalex=(!p.position(2)-!p.position(0))
   scaley=(!p.position(3)-!p.position(1))
   !p.position(0)=!p.position(2)+scalex/11.
   !p.position(2)=!p.position(0)+scalex/30.0
   SIZE=SQRT(scaley/scalex)*1.5
   !x.range=[0.,100.]
   !y.range=[ZMIN,ZMAX]
   !x.style=4+1
;
   linlogbit= keyword_set( zlog )

   !y.style = 1+4
   zfac= DOUBLE(zmax-zmin)/clen
   PLOT,[0,100],[zmin,zmax],color=!p.background,/noerase
   
;*******************create color bar (linear color sequence)
   FOR I=1,clen DO BEGIN
       ZINT1=(I-1)*zfac+zmin
       ZINT2=(I)*zfac+zmin
       color= cbase+I-1
       POLYFILL,[0.,0.,100.,100.],[ZINT1,ZINT2,ZINT2,ZINT1],col=color
   ENDFOR
   
;*******************axes and annotation (only here do we have to distinguish
;                   linear and logarithmic scaling)
   IF zlog and not bipolar THEN BEGIN ; logarithmic
       logmin=ALOG10(ZMIN)
       logmax=ALOG10(ZMAX)
       !y.ticks=(logmax-logmin)
       !y.crange = alog10(!y.crange)
;	this is a trick to get around the range finding algorithm in LOGAXIS
       !p.color=1
       LOGAXIS,3,ticklen=-0.6
;       the main tick length is 0.6 times the width of the color bar
   ENDIF ELSE if not zlog and not bipolar then BEGIN ; linear
       !y.ticks=FIX(zfac)/20
       !x.style=4
       !p.color=1
       AXIS,yaxis=1,ystyle=1,charsize=1.0
   ENDIF ELSE begin             ; bipolar logarithmic
       ymax = !p.position(3)
       ymin = !p.position(1)
       yzero = scaley/2+ymin
       !p.color=1
       
;      ; positive axis
      !y.minor = 9
      !y.ticks = fix(alog10(ZMAX))
      for i=0,!y.ticks do begin
          !y.tickname(i) = fltstr(1,i,0)
          !y.tickv(i) = 10^i
      ENDFOR

      !y.range = [ 1, ZMAX ]
      !p.position(1) = yzero+(ymax-ymin)/128. 
      !p.position(3) = ymax
      PLOT,[0,100],[1,1],color=1, /noerase, /nodata, ystyle=5 ; set plot param
      axis, yaxis=1, /ylog, ticklen=-0.6 

;      ; negative axis
      !y.ticks = fix(alog10(-ZMIN))
      for i=0,!y.ticks do begin
          !y.tickname(i) = '-'+fltstr(1,i, 0)
          !y.tickv(i) = 10^i
      ENDFOR

      !y.range = [ -ZMIN, 1 ]
      !p.position(1) = ymin
      !p.position(3) = yzero
      PLOT,[0,100],[-zmin,-zmin],color=1,/noerase, /nodata, ystyle=5
      axis, yaxis=1, /ylog, ticklen=-0.6
      
;      ; reset plot position for Z axis title
      !p.position(1) = ymin
      !p.position(3) = ymax
      
  endelse
  
  !p.font=-1
  !p.position(0)=!p.position(2)
  !p.position(2)=!p.position(0)+scalex/8.0
  !y.range=[0.,100.]
  PLOT,[0.,100.],[0., 100.] , /noerase, /nodata,  ystyle = 5,  xstyle = 5
  XYOUTS,75.,50.,ZTITLE,alig=0.5,chars=SIZE,orient=-90.,color=1
  !x=xa
  !y=ya
  !p=pa
  return
end
