;==============================================================================
;
;  NAME:
;     
;     ddeis_epa_ratio
;
;  PURPOSE:
;     
;     This function calculates the probability distribution, 
;     'pratio', for the ratio of two count rates randomly sampled 
;     in an energy pitch-angle bin.  
;
;  ARGUMENTS:
;     
;     Input
;     -----
;
;     epa_out (STRUCTURE):
;
;        output from 'ddeis_epa_avg'
;        
;     control (STRUCTURE):
;
;        zlow: integer 
;
;           specifying the smallest count rate which can contribute to
;           a ratio calculation
;
;        binr: float
;
;           the ratio bin size for the 'pratio' histogram
;
;        r0: float
;
;           the smallest ratio to use in the construction of 'pratio'
;
;        r1: float
;
;           the largest ratio to use in the construction of 'pratio'
;        
;     Ouput
;     -----
;
;     ratio_out (STRUCTURE)
;
;        ratio_raw:  fltarr
;
;           the raw ratio measurements
;
;        rbar: fltarr
;
;           the centers of the bins used in the contruction of
;           'pratio'
;
;        pratio: fltarr
;
;           ratio histogram (computed using IDL built in function
;           'histogram')
;
;        rmax: float
;
;           center of ratio bin with maximum number of measurements
;
;        meanr: float
;
;           arithmetic mean of the data in 'ratio_raw'; 'meanr' is 
;           set to -1 if there are less than 10 count comparisons
;           in the spectrum under investigation
;
;        sigr: float
;
;           standard deviation of the ratios determined from the data
;           in 'ratio_raw'; 'sigr' is set to -1 if there are less 
;           than 10 count comparisons in the spectrum under investigation
;
;  USAGE:
;
;     IDL> ddeis_epa_ratio, epa_out, control, ratio_out
;  
;  HISTORY:
;
;     VERSION 1.00 -- 1/22/98 -- Written, John C. Dorelli
;
;=============================================================================

PRO ddeis_epa_ratio, epa_out, control, ratio_out

; sort raw data according to increasing energy

idx_bine = sort(epa_out.bine)

bina = epa_out.bina(idx_bine)
bine = epa_out.bine(idx_bine)
zraw = epa_out.zraw(idx_bine)

; compute ratios in each bin

m = 0
ncomp = 500
ratio = replicate(-1d20, ncomp)

; loop over bins

FOR i = 1, epa_out.nbe DO BEGIN
    FOR j = 1, epa_out.nba DO BEGIN
        idx_bin = where(bine EQ i AND bina EQ j AND zraw GE control.zlow)

        ; if there are at least two measurements in the bin, calculate
        ; all possible ratios

        IF n_elements(idx_bin) GE 2 THEN BEGIN
            zarray = zraw(idx_bin)
;            zarray = zarray(sort(zarray))

            ; loop over all possible comparisons within the bin (i, j)
            
            FOR k = 1, n_elements(zarray) DO BEGIN
                FOR l = k+1, n_elements(zarray) DO BEGIN
                    m = m+1
                    ratio(m-1) = 1.*zarray(l-1)/zarray(k-1)
                ENDFOR
            ENDFOR

        ENDIF 
    ENDFOR
ENDFOR

rok = where(ratio NE -1d20)
ratio = ratio(rok)

; compute mean and standard deviation

IF n_elements(ratio) GE 10 THEN BEGIN
    meanr = total(ratio)/n_elements(ratio)
    residr = total((ratio-meanr)^2)/(n_elements(ratio)-1)
    sigr = sqrt(residr)
ENDIF ELSE BEGIN
    print, 'DDEIS_EPA_RATIO:  not enough comparisons to compute mean ratio'
    meanr = -1
    sigr = -1
ENDELSE

; compute ratio probability distribution for selected energy range

binr = control.binr
r0 = control.r0
r1 = control.r1
pratio = histogram(ratio, binsize = binr, min = 0., max = 2.)

; fill in output structure

rbar = r0+binr*findgen(n_elements(pratio))+binr/2.
idx_maxr = where(pratio EQ max(pratio))
rmax = rbar(idx_maxr)

ratio_out = {$
              ratio_raw: ratio, $
              rbar: rbar, $
              pratio: pratio, $
              rmax: rmax, $
              meanr: meanr, $
              sigr: sigr}

END

