;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_skew, panel, typeVector, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type DATA_TYPE for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      typeVector    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              For this data type has the following meaning:
;*      typeVector(0) 0 for electrons
;*              1 for ions
;*              2 for both in one plot (iowa style).
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_skew, panel, typeVector, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     written april 1996, Reiner Friedel
;*
;***************************************************************************

pro p_hydra_skew, panel, typeVector, res=res, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           subtable=subtable, _EXTRA=extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Linestyles are also not useful. So the keywords
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; Comments here should include an explanation of the plot options used
; in type, swwitch or channel. Replace type in the call with whatever variables
; you need

    common hydra_spec_data, $
      data_context, $           ; selects current data set
      spec_header, $
      spec_data, $
      spec_aligned_data, $
      spec_perp_data, $
      spec_opposed_data, $
      hr_spec_data, $
      hr_spec_aligned_data, $
      hr_spec_perp_data, $
      hr_spec_opposed_data

   
; One of the common blocks should contain the plot data as returned by the
; data read procedure.
; The following common blocks are needed:

   common time, xut1,xut2		;common time limits in T90
   common ger_error, get_err_no, get_err_msg
   common yscale, yscl    ;man/auto yscaling
   common zscale, zscl
   common coordinateSystems, plotted_x, plotted_y
                                        ;info on coords, used by mouse
                                        ;                   functions
   common papco_color_names

   rfill= 1e-20                 ; rfill must be small wrt data

   panelset,panel			;sets the panel position viewport

   if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')

;  All Hydra panels use the same ticks.
   goodticks, xtickv, xminor
   xticks=n_elements(xtickv)-1
   
   aligned= 1
   perp= 2
   opposed= 3
   
   if res eq 0 then context=0 else context=5
   
   r= execute( 'data_all=' + spec_header(context).data )
   r= execute( 'data_up=' + spec_header(1+5*res).data )
   r= execute( 'data_perp=' + spec_header(2+5*res).data )
   r= execute( 'data_dn=' + spec_header(3+5*res).data )   
   
   time= data_all(*).time + spec_header(context).start_time - xut1
   dxut= xut2-xut1   
   
   rvis= where( time ge 0 and time le dxut, nvis )
   if nvis lt 2 then begin
       plotted_x = !x
       plotted_y = !y
       spec_header(context).reload_flag=1
       print, 'p_hydra_skew: internal error, can''t find valid data'
       return
   endif 

;  find common time structures.  Note that all the angle measurements
;    must be at the same times!  (This is okay, since we load them all
;    at once.
   r= hydra_union( data_all(rvis).time, data_up(*).time, all_idx, angles_idx )

   if all_idx(0) eq -1 then begin
       message, 'Internal Error: Data sets do not overlap.', /cont
       message, '  Try explicit reload of data set.', /cont
       plotted_x= !x
       plotted_y= !y
       return
   endif
   rvis= rvis( all_idx )

;  detect psi on
   r= where( data_all(rvis).psi_status eq 1 )
   psi_on= r(0) ne -1

;  detect interactive mode (enable pop-ups, etc.)
   interactive= (!d.name eq 'X' or !d.name eq 'WIN' )

;  check spec shift is the same for all spectrograms
   spec_shift= spec_header(context).options and 1L
   r= where( (spec_header(context+indgen(3)+1).options and 1L) ne spec_shift )
   if r(0) ne -1 then begin
       plot, [0,1], /nodata, xstyle=5, ystyle=5
       xyouts, .5,.5, align=0.5, $
         'Spectrograms have inconsistent s/c potential shift selections!c'+$
         'Make all selections similar and replot.'
       plotted_x= !x
       plotted_y= !y
       return
   endif
   
   dt= spec_header(context).time_resolution

   type= typeVector(2)   
   case type of 
       0: begin
           red_tit= 'Aligned'
           blue_tit= 'Opposed'
           up_matrix= data_up(angles_idx).spectrum(*)
           dup_matrix= data_up(angles_idx).sigma(*)
           dn_matrix= data_dn(angles_idx).spectrum(*)
           ddn_matrix= data_dn(angles_idx).sigma(*)
           all_matrix= data_all(rvis).spectrum(*)
           
           time=time(rvis)
           mode= data_up(angles_idx).mode
           mode_energy= spec_header(1+5*res).mode_energy


           rnv= where( dup_matrix eq rfill  or  ddn_matrix eq rfill  $
                       or all_matrix lt 0.1 )
           r= check_math(1,1)   ; turn off math warnings -- fill gives underflw
           err_matrix= dup_matrix + ddn_matrix 
           r= check_math(1,0)
           err_matrix(rnv)= rfill
           up_matrix(rnv)= rfill
           dn_matrix(rnv)= rfill

           spectra_sort, err_matrix, mode, mode_energy, error1, energy
           spectra_sort, up_matrix, mode, mode_energy, red1, energy
           spectra_sort, dn_matrix, mode, mode_energy, blue1, energy

       end
       1: begin                 ; anisotropy
           if res eq 0 and spec_header(1).data_version lt 2.0 then begin
               r= hydra_message( ['Perpendicular error not available,', $
                                  'using zero.'], /info )
           endif
           red_tit= 'Para'
           blue_tit= 'Perp'
           dup0= data_up(angles_idx).sigma(*)
           ddn0= data_dn(angles_idx).sigma(*)
           dperp0= data_perp(angles_idx).sigma(*)
           up0= data_up(angles_idx).spectrum(*)
           dn0= data_dn(angles_idx).spectrum(*)
           perp0= data_perp(angles_idx).spectrum(*)           
           all0= data_all(rvis).spectrum(*)
           
           mode= data_up(angles_idx).mode
           time=time(rvis)
           mode_energy= spec_header(1+5*res).mode_energy

           rnv= where( ( dup0 eq rfill and ddn0 eq rfill ) or $
                       dperp0 eq rfill or $
                       all0 lt 0.1 )           
           r= check_math(1,1)
           err0= dup0 + ddn0 + dperp0
           r= check_math(1,0)
           err0(rnv)= rfill
           
           message, 'Using unweighted average to bring together up '+$
             'and down measurements.', /cont
           nup0=  dup0 ne rfill
           ndn0=  ddn0 ne rfill
           red0= ( up0*nup0 + dn0*ndn0 ) / (nup0+ndn0)
           
           red0(rnv) = rfill
           perp0(rnv)= rfill
           
           spectra_sort, err0, mode, mode_energy, error1, energy
           spectra_sort, red0, mode, mode_energy, red1, energy
           spectra_sort, perp0, mode, mode_energy, blue1, energy
       end
       2: begin
           red_tit= 'Aligned'
           blue_tit= 'Opposed'
           up_matrix= data_up(angles_idx).spectrum(*)
           dup_matrix= data_up(angles_idx).sigma(*)
           dn_matrix= data_dn(angles_idx).spectrum(*)
           ddn_matrix= data_dn(angles_idx).sigma(*)
           all_matrix= data_all(rvis).spectrum(*)
           
           time=time(rvis)
           mode= data_up(angles_idx).mode
           mode_energy= spec_header(1+5*res).mode_energy

           rnv= where( dup_matrix eq rfill  or  ddn_matrix eq rfill  or all_matrix lt 0.1 )

           err_matrix= make_array( size=size( up_matrix), value=1.0 )

           err_matrix(rnv)= rfill
           up_matrix(rnv)= rfill
           dn_matrix(rnv)= rfill

           error_fact= zscl(panel(0),1)
           if error_fact eq 0. then error_fact=1.
           
           yes_red= up_matrix gt error_fact*sqrt(dup_matrix)
           yes_blue= dn_matrix gt error_fact*sqrt(ddn_matrix)

           spectra_sort, float(yes_red)*1000, mode, mode_energy, red1, energy
           spectra_sort, float(yes_blue)*1000, mode, mode_energy, blue1, energy
           spectra_sort, float(err_matrix), mode, mode_energy, error1, energy
           
           r= where( red1 eq 0 and blue1 eq 0 ) 
           if r(0) ne -1 then error1(r)= rfill
       end
   endcase

;  We need to insert fill data (and grid in time) this data before we
;  can average it.
;   dt= time(1:*) - time(0:*)
;   r= where(dt gt 0)
;   if r(0) ne -1 then dt=min(dt(r)) else dt=1
;   nn= max(time)/dt + 1
;   t90_grid= make_array( nn, /float, value=1e31 )
;   error1_grid= make_array( n_elements( error1(*,0) ), nn, /float, value=rfill)
;   red1_grid= make_array( n_elements( red1(*,0) ), nn, /float, value=rfill )
;   blue1_grid= make_array( n_elements( blue1(*,0) ), nn, /float, value=rfill )
;   r= long( time/dt )

;   error1_grid( *,r )= error1
;   blue1_grid( *,r )= blue1
;   red1_grid( *,r )= red1
;   t90_grid( r ) = time
   
;   t90= t90_grid
;   error1= error1_grid
;   blue1= blue1_grid
;   red1= red1_grid

   dc= data_context

   channelwidth=0.07		;sets % of channel width to get top
                                ;and bottom bounds

   case typeVector(0) of
       0:begin			;electrons
           spec='electrons'
           rspec=where(energy lt 0)
           en_e=energy(rspec)
           energy= reverse( -1.0*energy( rspec ) )
           yminarr=reverse(en_e)*(1-channelwidth)*(-1)
           ymaxarr=reverse(en_e)*(1+channelwidth)*(-1)
           specstr='Ele'
           zmax = 20.
           zmin = 1.
       end
       1:begin			;ions
           spec='ions'
           rspec=where(energy gt 0)
           en_i=energy(rspec)
           energy= energy(rspec)
           yminarr=reverse(en_i)*(1-channelwidth)
           ymaxarr=reverse(en_i)*(1+channelwidth)
           specstr='Ion'
           zmax = 200.
           zmin = 1.
           ztit='DDEIS Av. Ion Counts'
       end
   endcase
   
   case type of 
       0: ctit2= 'HYDRA!c'+specstr+' Skew'
       1: ctit2= 'HYDRA!c'+specstr+' Anisotropy'
       2: ctit2= 'HYDRA!c'+specstr+' Binary Skew'
   endcase
   
   if getenv('hydra_presentation_mode') eq '1' then begin       
       if spec_shift then cytit='E!dD!n(eV)' else $
         cytit='E!dobs!n(eV)'
   endif else begin
       if spec_shift then cytit='E Debye, eV' else $
         cytit='E obs, eV'
   endelse
       
   ctit2=ctit2
   
   ctit2=font+ctit2
   cytit=font+cytit
   
; handle axis range...
   ymin=min(yminarr) & ymax=max(ymaxarr)

; Y axis scaling
   if (yscl(panel(0),0) eq 1) then begin
       ymin=yscl(panel(0),1)
       ymax=yscl(panel(0),2)
       ylog= yscl(panel(0),3)
   endif else begin
       print, 'setting overplot range...'
       ylog=1
       yscl(panel(0),1)=ymin
       yscl(panel(0),2)=ymax
       yscl(panel(0),3)=ylog    ; ylog=1
   endelse
   
   if (zscl(panel(0),0) eq 1) then begin
       zmin=zscl(panel(0),1)
       zmax=zscl(panel(0),2)
   endif else begin
       zscl(panel(0),1)=zmin
       zscl(panel(0),2)=zmax
   endelse

   if type eq 2 then begin
       zmin=1.
       zmax=10.
   endif
   
; set up extra plot keywords, common ones first
  extra_plotPar_common={ xrange:[0,xut2-xut1], xstyle:1, $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xminor:xminor, xtickv:xtickv, xticks:xticks, $
                         ytitle:cytit, yticklen:-0.010, ylog:ylog, $
                         yrange:[ymin,ymax], ystyle:1 }

; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

   if n_elements(extra_par) EQ 0 then $
     extra_par={color:1}

; add keyword structure set here with the one passed in
;  p_hydra_skew: note this_par is not used.
   extra_plotPar=create_struct( extra_plotPar_common, extra_par )

   axis_plotPar= create_struct( extra_plotPar_common, {color:1} )


; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot

   if panel(0) eq 0 then $
     xtickformat= 'hydra_timeticks' $
   else $
     xtickformat= 'noticks'

; Extract species and make it right-side up.
   if spec eq 'electrons' then begin
       red1= rotate(red1(rspec,*),3)
       blue1= rotate(blue1(rspec,*),3)
       error1= rotate(error1(rspec,*),3)
   endif else begin
       red1= transpose(red1(rspec,*))
       blue1= transpose(blue1(rspec,*))
       error1= transpose(error1(rspec,*))
   endelse
   
;do color plot and color bar plot
   
;  format yarr and xarr for hydra_plotcolor
   x1= time
   x2= time+dt
   t= reform( [ x1, x2], n_elements(x1), 2 )
   e1= alog10(energy)
   n=n_elements(e1)-1
   de= e1(n) - e1(n-1)
   e2= [e1(1:*),e1(n)+de ]
   e= reform( 10^[ e1, e2], n_elements(e1), 2 )

   zmat = red1 - blue1
   hydra_plotcolor, zmat, t, e, zrange=[zmin,zmax], $
     _extra= axis_plotPar, $
     xtickformat=xtickformat, nodata=rfill, ytickformat='hyd_log_ticks', $
     special_colors=[[0.0,black]], diff_error2= error1, dx=dt

   if type eq 2 then begin
       urd= [!x.crange(1),([!y.crange(1),10^!y.crange(1)])(!y.type)]
       ur= convert_coord( urd, /data, /to_device )
       ps_fact=!d.x_px_cm/40.
       xyouts, ps_fact*(-100), ps_fact*(-100), /device, $
         strtrim(string(error_fact,form='(f20.1)'),2), $
         color= get_color_index('white'), align=0.5, width=w, charsize=1.
       w= (convert_coord( [w,0], /norm, /to_device ))(0)
       polyfill, ur(0)-ps_fact*50+ps_fact*[0,w+5,w+5,0,0], $
         ur(1)-ps_fact*20+ps_fact*[0,0,15,15,0], $
         color= get_color_index('white'), /device
       rr= convert_coord( ur(0)-50+ps_fact*[0,w+5,w+5,0,0], $
                          ur(1)-ps_fact*20+ps_fact*[0,0,15,15,0], $
                          /device, /to_data )
       oplot, rr(0,*), rr(1,*), color= get_color_index('black')
       xyouts, ur(0)-ps_fact*50+(w+ps_fact*5)/2, ur(1)-ps_fact*16, /device, $
         strtrim(string(error_fact,form='(f20.1)'),2), $
         color= get_color_index('black'), align=0.5, charsize=1.
   endif

; find colorbar range
   common papco_color,pap_color
   cindex= where( pap_color(*,2) eq 1 ) ; search the active vector
   if (cindex(0) eq -1) then begin
     print, 'Color system failure, consult jbf@space-theory.physics.uiowa.edu'
     print, 'Using color set 0'
     cindex=0
   endif
   color_range= PAPCO_get_Colorindices( cindex )
   colorbase= color_range(0)
   ncolors= color_range(1)-color_range(0)
   colortop= color_range(1)

   colormid= ( colorbase + colortop ) / 2

   hydra_color_bar, zrange=[zmax, zmin], /zlog, ztitle=blue_tit, $
     ypos=[0.0,0.5], color_range=[ colorbase, colormid ]
   hydra_color_bar, zrange=[zmin, zmax], /zlog, ztitle=red_tit, $
     ypos=[0.5,1.0], color_range=[ colormid+1, colortop ], /black_bar
   hydra_color_bar, zrange=[zmax, zmin], /zlog, ztitle='', $
     ypos=[0.0,1.0], color_range=[ colorbase, colormid ], /nobar

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.

  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
     x_side_label,panel,'time (UT)!CDate'
  endif

; store the coordinate information into common block

   plotted_x = !x
   plotted_y = !y

; plot info right of the panel. If you want overplots to have their own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION. Since here colorbars are plotted labels are rotated
; by 90 degrees and then plotted.

  right_side_label,panel,ctit2,/rot90

; Version Control: Check Version Quality
   if data_context lt 5 then begin
       plot_name= 'papco_skew_sv' 
       input_file= 'survey'
   endif else begin
       plot_name= 'papco_skew_l1'
       input_file= 'ddeis_l1_cal'
   endelse
   date= convert_secondstotime( spec_header(dc).start_time, /t90, country=1 )
   date= strmid( date, 6, 2 ) +  strmid( date, 0, 2 ) + strmid( date, 3, 2 )

   plot_version= hyd_prod_vers_v01( product=plot_name, $
                                    dependency_names= $
                                    [ input_file, 'papco_spec_code' ], $
                                    dependency_vers= $
                                    [ spec_header(dc).data_version, $
                                      hydra_code_version(/spec) ] $
                                  )
   

;  information useful for pop-up widgets
   interactive= (!d.name eq 'X' or !d.name eq 'WIN' )
   common plot_composer, widgetData
   if n_elements( widgetData ) ne 0 then $
     parent= widget_info( widgetData.dw_draw, /parent ) $
   else $
     parent=0


   if plot_version gt 0.00 then begin   
       qf= hyd_prod_qual_v01( product_type=plot_name, $
                              date='19'+date, $
                              product_version=plot_version )
   endif else begin
       qf=-1       
       dependency_names= $
         [ input_file, 'papco_skew_code' ]
       dependency_vers= $
         [ spec_header(dc).data_version, $
           hydra_code_version(/skew) ]
       print, plot_name, '[', dependency_names + '=' + $
         strtrim( dependency_vers, 2 ), $
         '] --> ', strtrim( plot_version, 2 )   
   endelse

   if qf gt -1 then begin
       if interactive then begin
           case qf of
               1: if spec_header(dc).message_flag eq 1 then begin
                   r= widget_message(['Plot is not','the best available.'], $
                                     dialog_parent=parent, /info )
                   spec_header(dc).message_flag = 0
               endif
               2: if spec_header(dc).message_flag eq 1 then begin
                   r= widget_message(['Plot contains','known flaws.'], $
                                     dialog_parent=parent, /info )
                   spec_header(dc).message_flag = 0
               endif
               3:
               else:
           endcase
       endif
       case qf of
           1: qualstr='(NBA)'
           2: qualstr='(F)'
           3: qualstr='(BA)'
           else:
       endcase
   endif

   if ( psi_on and interactive ) then begin
       r= widget_message( [ 'PSI power supply on', 'during this interval' ], $
                          dialog_parent=parent, /info )
   endif

  if qf ge 0 then begin
       if data_context gt 4 then res=1 else res=0
       tresstr=(['pskewsv','pskewl1'])(res)      
       footprint_str= tresstr + $
         ':'+ strtrim( string(plot_version,format='(f9.2)'), 2 ) + qualstr
       hydra_footprint_new, string= footprint_str
   endif else begin
       if data_context le 4 then begin
           hydra_footprint_new, $
             survey=spec_header(dc).data_version, $
             code= hydra_code_version(/skew), notes='(??)'
       endif else begin
           hydra_footprint_new, $
             ddeis_l1_cal=spec_header(dc).data_version, $
             code= hydra_code_version(/skew), notes='(??)'
       endelse
   endelse
end
