;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_spp, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hydra_spp for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_spp, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_spp, panel, type, $
                 OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                 _EXTRA=extra_par, subtable=subtable

; The following common blocks are needed:

common time, xut1,xut2          ;common time limits in T90
common mjdt, mjdt1, mjdt2
common coordinateSystems, plotted_x, plotted_y
                                ;info on coords used by mouse
common hydra_spp_data, spp_header, spp_data
common hydra_label_maker_block
if (n_elements(spp_spec_lab) eq 0) then hydra_label_maker, /spp
common papco_color_names

panelset,panel			;sets the panel position viewport

oldy=!y
oldx=!x

goodticks, xtickv, xminor
xticks=n_elements(xtickv)-1

; set duplex font if big_gif is device ; duplex roman
if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')

; the plot calls normally use the arrays t90 for the time axis and yray (ymat
; for color plots) for the y or z axis.
dxut= xut2-xut1

ivar = spp_header.data_var
rstr = spp_res_rstr(spp_header.data_res)
ctit2 = 'SPP!s!d'+rstr+'!r!u'+spp_data.tstr+'!n '
ctit2 = font+ctit2+spp_var_lab(ivar,2)
cytit = font+spp_var_lab(ivar,3)

gd = where(spp_data.ydat ne -1.e+31)
t90 = spp_data.xdat(gd) + ( spp_header.start_time - xut1 )
ydat = spp_data.ydat(gd)
ylog = spp_data.ylog

ctit=' '                        ; Do not use this variable!

; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images

; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF

index=where((t90 ge 0) and (t90 le dxut),count)
if count ge 2 then begin
    t90=t90(index)
    ydat=ydat(index)
end

; insert breaks where there are data dropouts. (grid the data)
dt= t90(1:*) - t90(0:*)
r= where(dt gt 0)
if r(0) ne -1 then dt=min(dt(r)) else dt=1
nn= max(t90)/dt + 1
ydat_grid= make_array( nn, /float, value=1e31 )
t90_grid= make_array( nn, /float, value=0 )
r= long( t90/dt )
ydat_grid( r ) = ydat
t90_grid( r ) = t90

t90= t90_grid
ydat= ydat_grid


; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin
;         yscl(*,2)  is then ymax
;         yscl(*,3)  is ylog
; when scaling is set to automatic, the automatic y limits are loaded in.

y_kw= hydra_ytick_parm( ydat, panel=panel(0), ylog=ylog, fill=1e31 )

; set up extra plot keywords, common ones first

extra_plotPar_common={xrange:[0,dxut], $
                      noerase:1, ticklen:-0.03, xticklen:-0.03, $
                      xtickv:xtickv, xminor:xminor, xticks:xticks, $
                      yticklen:0.010, ytitle:cytit, $
                      ythick:1.3, xthick:1.3, max_value:1e30 }

extra_plotPar_common= $
  create_struct( extra_plotPar_common, y_kw )

; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

if n_elements(extra_par) EQ 0 then $
  extra_par={color:1}

; add keyword structure set here with the one passed in

extra_plotPar= create_struct( extra_plotPar_common, extra_par )

axis_plotPar= create_struct( extra_plotPar_common, {color:1} )

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot

if panel(0) eq 0 then BEGIN
    xtickformat='hydra_timeticks'
ENDIF ELSE BEGIN 
    xtickformat='noticks'
ENDELSE

; plot the axis in color=1, and only plot if if you're not overplotting
IF NOT keyword_set(overplot) THEN BEGIN 
    plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
      xstyle=1, ystyle=1, xtickformat=xtickformat
    if panel(0) eq 0 then $
      x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
ENDIF 

; now do the actual data plot
hydra_plot, t90, ydat, _extra=extra_plotPar, xstyle=5, ystyle=5, $
  overplot= keyword_set(overplot)

if not keyword_set(overplot) then begin
    plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
      xstyle=1, ystyle=1, xtickformat=xtickformat
    if panel(0) eq 0 then $
      x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
endif

; store the coordinate information into common block
plotted_x = !x
plotted_y = !y

!x=oldx
!y=oldy

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015

if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
    r= where( (xtickv+xut1) mod 86400 eq 0, count )
    if r(0) ne -1 then begin
        xtickv= xtickv(r)
        xticks= count
    endif else begin
        xtickv= xtickv([0,n_elements(xtickv)-1])
        xticks= 1
    endelse
    
    axis_plotPar.xtickv= xtickv
    axis_plotPar.xticks= xticks
    
    plot, t90, ydat, /NODATA, $
      _extra=axis_plotPar, xtickformat='hydra_juldate', $
      xstyle=1, ystyle=5
    
endif

; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
right_side_label,panel,lfs+ctit2, _extra=extra_plotPar

hydra_footprint, spp_header

message, 'DONE', /cont

return
END 





