pro hydra_colorbar, zmat=zmat, zlog=zlog, zrange=zrange, $
                    colors=colors, _extra=e, help, ztitle=ztitle, $
                    zword=zword, xpos=xpos, ypos=ypos, ctable=ctable, $
                    inset=inset, position_out=position_out, $
                    ticklen=ticklen, charsize=charsize, $
                    no_tv=no_tv
;+
; NAME:
;       hydra_colorbar
;
;
; PURPOSE:
;
;       Draws a color bar to the right side of the last plot, plotting
;       the map from color to "Z" value.
;
; CATEGORY:
;
;       color plotting
;
; CALLING SEQUENCE:
;
;       hydra_colorbar, zrange=[min,max], /zlog, ztitle='Z,units'
; 
; KEYWORD PARAMETERS:
;       zmat=z:       supply the z matrix for autoranging. 
;       ctable=[,]:   ctable(0) is first color index, ctable(1) is last.
;       ztitle='':    string to print along axis.
;       zrange=[,]:   like xrange, zrange(0) is first color, zrange(1) is last
;       /zlog:        logarythmic z axis.
;       xpos=[,]:     specify position of colorbar relative to the position
;                     vector x position.  0 is the left edge of plot,
;                     1 is the right side.  For example, default is [ 1.05, 1.10 ]
;       ypos=[,]:     specify position of colorbar relative to yaxis.
;       inset=n       position colorbar as inset in current plot.  
;                         n  is the corner: 0 is ul, 1 is ur, 2 is lr, 3 is ll, 4 is ul
;       zword='':     string to print above the colorbar.
;       position_out  colorbar position in normal coordinates
;       ticklen=t     length of ticks, relative to bar width (default is 0.1)
;       /no_tv        draw colorbar as sequence of polygons, not TV image
;                        The ps file may be slightly larger, but may be
;                        more portable.                        
;
; PROCEDURE:
;       locate corners of image, draw tv image. draw axis with "axis"
;       procedure.  xyouts zword if specified.
;
; EXAMPLE:
;       hydra_plotcolor, Z, time, energy, $
;             zrange=[0.1,100], /zlog, title='POLAR/HYDRA Ions'
;       hydra_colorbar, zrange=[0.1,100], /zlog, ztitle='Counts'
;
; MODIFICATION HISTORY:
;       Log started October 15, 1998
;        written, JBF, ????
;        Oct 22, 1999 JBF  added no_tv keyword--use polyfills
;                                    instead of tv.
;-

if n_elements( inset ) gt 0 then begin
    ixpos0= [ 0.08, 0.87, 0.87, 0.08, 0.08 ]    
    ixpos1= [ 0.13, 0.92, 0.92, 0.13, 0.13 ]    
    iypos0= [ 0.5, 0.5, 0.05, 0.05, 0.5 ]
    iypos1= [ 0.95, 0.95, 0.5, 0.5, 0.95 ]
    
    xpos= [ interpolate( ixpos0, inset ), interpolate( ixpos1, inset ) ]
    ypos= [ interpolate( iypos0, inset ), interpolate( iypos1, inset ) ]
endif

if n_elements( ypos ) ne 2 then ypos=[ 0, 1. ]
if n_elements( xpos ) ne 2 then xpos=[ 1.05, 1.10 ]

if n_elements( ctable ) eq 2 then colors=ctable
zlog=keyword_set(zlog)

if n_elements( ztitle ) eq 0 then ztitle=''

if n_elements( ticklen ) eq 0 then ticklen=-0.2

if n_elements( charsize ) eq 0 then charsize= !p.charsize

if not keyword_set(zrange) then begin
    if keyword_set(zmat) then begin
        zrange=dblarr(2)
        zrange(0)=min(zmat)
        zrange(1)=max(zmat)  
    endif else begin
        print, 'COLORBAR: no z range specified.'
        return
    endelse
end

if n_elements(colors) eq 0 then begin
    hydra_setcolors, cs
    colors= cs.colortable
endif
ncolors= colors(1)-colors(0)+1

ylog=!y.type
xlog=!x.type

if (ylog) then begin
    ycrange=10^!y.crange
endif else begin
    ycrange=!y.crange
endelse

if (xlog) then begin
    xcrange=10^!x.crange
endif else begin
    xcrange=!x.crange
endelse

ll= convert_coord( xcrange(0), ycrange(0), /data, /to_normal )
ur= convert_coord( xcrange(1), ycrange(1), /data, /to_normal )

llb= ll
urb= ur

llb(0)= ur(0)+ (ur(0)-ll(0)) * (xpos(0)-1.)
urb(0)= ur(0)+ (ur(0)-ll(0)) * (xpos(1)-1.)

llb(1)= ll(1)+ (ur(1)-ll(1)) * ypos(0)
urb(1)= ll(1)+ (ur(1)-ll(1)) * ypos(1)

cmap= [1,1] # ( (indgen(ncolors))+colors(0) )

xsizen= urb(0)-llb(0)
ysizen= urb(1)-llb(1)

pold= !p
xold= !x
yold= !y

position= [ llb(0), llb(1), urb(0), urb(1) ]
shift= urb(0)-1. > 0
position= position - [ shift, 0, shift, 0 ]

use_tv= not keyword_set(no_tv)
if use_tv then begin
    if !d.name eq 'PS' then begin
        hyd_tv, byte(cmap), llb(0), llb(1), $
          xsize=xsizen, ysize=ysizen, /normal
    endif else begin
        sized= ( convert_coord( xsizen, ysizen, /norm, /to_device ) ) > 2
        hyd_tv, congrid( cmap, sized(0), sized(1) ), llb(0), llb(1), /norm
    endelse
endif else begin
    plot, findgen(100), /nodata, /noerase, $
      position=position, $
      xstyle=5, ystyle=5, yrange=colors+[0,1], xrange=[0,1], /normal
    for i=colors(0),colors(1) do begin
        polyfill,[0,1,1,0,0],[0,0,1,1,0]+i, color=i
    endfor
endelse

plot, findgen(100), /nodata, /noerase, $
  position=position, $
  xstyle=5, yrange=zrange, ystyle=5, xrange=[0,1], /normal

position_out= position

;  draw background for inset characters.
if n_elements( inset ) gt 0 then begin
    if n_elements(e) gt 0 then begin    
        axis, yaxis=1, ylog=zlog, yrange=zrange, ystyle=1, _extra=e, $
          charthick=3.0, color=!p.background, charsize=charsize, ticklen=ticklen, $
          ytitle=ztitle
    endif else begin
        axis, yaxis=1, ylog=zlog, yrange=zrange, ystyle=1, charthick=3.0, $
          color=!p.background, charsize=charsize, ticklen=ticklen, $
          ytitle=ztitle
    endelse
endif

if n_elements(e) gt 0 then begin    
    axis, yaxis=1, ylog=zlog, yrange=zrange, ystyle=1, _extra=e, ticklen=ticklen, $
      charsize=charsize, ytitle=ztitle
endif else begin
    axis, yaxis=1, ylog=zlog, yrange=zrange, ystyle=1, ticklen=ticklen, $
      charsize=charsize, ytitle=ztitle
endelse

; print the ztitle
;label_width=0.075               ; hard to know
;x= position(2) + label_width
;y= ( position(1) + position(3) ) / 2.
;if n_elements( inset ) gt 0 then begin
;    xyouts, x, y, ztitle, /normal, charsize=charsize, charthick=3.0, $
;      align=0.5, orientation=270, color=!p.background
;endif
;xyouts, x, y, ztitle, /normal, charsize=charsize, align=0.5, orientation=270

plot, findgen(100), /nodata, /noerase, $
  position=[ llb(0), llb(1), urb(0), urb(1) ], $
  xstyle=5, yrange=[0,1], ystyle=5, xrange=[0,1], /normal

if n_elements( inset ) gt 0 then begin
    oplot, [ 0,0,1,1,0], [ 0,1,1,0,0 ]
endif

if n_elements(zword) gt 0 then $
  xyouts, (urb(0)+llb(0))/2, urb(1)+(urb(1)-llb(1))*0.03, $
  align=0.5, zword, /normal, charsize=charsize*1.25

!p= pold
!x= xold
!y= yold

return
end



