;==============================================================================
;
;  NAME:
;     
;     ddeis_block_get
;
;  PURPOSE:
;     
;     This function gets the block of Hydra-DDEIS data closest to an
;     input time string, 'time.'
;
;  ARGUMENTS:
;     
;     Input
;     -----
;
;     date (INTEGER):
;
;        year, month and date in the format yyyymmdd (e.g., 19960529
;        is May 29, 1996) 
;        
;     time (STRING):
;
;        hours, minutes and seconds in the format 'hh:mm:ss' 
;        (e.g., '04:30:00')
;
;     
;     Ouput
;     -----
;
;     block_data, mode_info (STRUCTURE):
;
;        see level1 library routine 'init_level1' for a description of
;        these structures
;
;     file_flag (INTEGER):
;
;        0: level1 file not found
;        1: level1 file found
;
;  KEYWORDS:
;
;     next:
;
;        get the next block of data
;
;     previous:
;
;        get the previous block of data
;
;  USAGE:
;
;     IDL> ddeis_block_get, 19960529, '04:30:00', block_data,
;             mode_info, file_flag 
;  
;  HISTORY:
;
;     VERSION 1.00 -- 1/16/98 -- Written, John C. Dorelli
;     VERSION 1.01 -- 1/23/98 -- corrected bug in implementation of
;                                   'next' and 'previous' keywords
;
;=============================================================================

PRO ddeis_block_get, date, time, block_data, mode_info, file_flag, $
                     next = next, previous = previous

COMMON store1, path_store, t0_store, block_data_store, mode_info_store, $
  blocknum_store

arg_path = date
arg_t0 = time

file_flag = 1

; initialize leve1 file if the 'date' argument has changed

IF NOT keyword_set(path_store) THEN BEGIN
    path_store = arg_path
    path = arg_path
    init_level1, path, block_data, mode_info
    IF mode_info.footprint.l1_version EQ 0 THEN BEGIN
        file_flag = 0
        return
    ENDIF
    mode_info_store = mode_info
    block_data_store = block_data
ENDIF ELSE BEGIN
    IF path_store NE arg_path THEN BEGIN
        close_level1
        path_store = arg_path
        path = arg_path
        init_level1, path, block_data, mode_info
        IF mode_info.footprint.l1_version EQ 0 THEN BEGIN
            file_flag = 0
            return
        ENDIF
        mode_info_store = mode_info
        block_data_store = block_data
    ENDIF ELSE BEGIN
        mode_info = mode_info_store
        block_data = block_data_store
    ENDELSE
ENDELSE

; compute time in seconds from 'time' argument

hour = strtrim(strmid(arg_t0, 0, 2), 1)
min = strtrim(strmid(arg_t0, 3, 2), 1)
sec = strtrim(strmid(arg_t0, 6, 2), 1)

tsec = hour*3600L+min*60L+sec*1L
thour = tsec/3600.

; get the block of Hydra-DDEIS data closest to 'time'

IF NOT keyword_set(t0_store) THEN BEGIN
    t0_store = arg_t0
    blockindex = where(mode_info.rectime GE thour)
    blocknum = blockindex(0) - 1
    IF blocknum LT 0 THEN blocknum = 0
    read_level1, blocknum, block_data, mode_info
    mode_info_store = mode_info
    block_data_store = block_data
    blocknum_store = blocknum
    return
ENDIF ELSE BEGIN
    IF t0_store NE arg_t0 OR path_store NE arg_path THEN BEGIN
        t0_store = arg_t0
        blockindex = where(mode_info.rectime GE thour)
        blocknum = blockindex(0) - 1
        IF blocknum LT 0 THEN blocknum = 0
        read_level1, blocknum, block_data, mode_info
        mode_info_store = mode_info
        block_data_store = block_data
        blocknum_store = blocknum
        return
    ENDIF ELSE BEGIN
        IF keyword_set(previous) THEN BEGIN
            blocknum = blocknum_store-1
            read_level1, blocknum, block_data, mode_info
        ENDIF
        IF keyword_set(next) THEN BEGIN
            blocknum = blocknum_store+1
            read_level1, blocknum, block_data, mode_info
        ENDIF
        IF NOT keyword_set(previous) AND $
          NOT keyword_set(next) THEN BEGIN
            block_data = block_data_store
            mode_info = mode_info_store
            blocknum = blocknum_store
        ENDIF
        block_data_store = block_data
        mode_info_store = mode_info
        blocknum_store = blocknum
    ENDELSE
ENDELSE

END



