;==============================================================================
;
;  NAME:
;     
;     ddeis_epa_cuts
;
;  PURPOSE:
;     
;     This function calculates a cut of Hydra-DDEIS data as a function
;     of energy in a specified pitch-angle range.
;
;  ARGUMENTS:
;     
;     Input
;     -----
;
;     date (INTEGER):
;
;        year, month and date in the format yyyymmdd (e.g., 19960529
;        is May 29, 1996) 
;        
;     time (STRING):
;
;        hours, minutes and seconds in the format 'hh:mm:ss' 
;        (e.g., '04:30:00')
;
;     de (FLOAT):
;
;        energy bin size (delta_E/E)
;  
;     da (FLOAT):
;
;        pitch-angle bin size (degrees)
;
;     index (INTEGER):
;
;        bin index to be used in the cut
;     
;     Ouput
;     -----
;
;     energy (FLOAT):
;
;        1-D array of energies (eV)
;
;     f (DOUBLE):   
;
;        1-D array of distribution function data (CGS units) 
;
;  KEYWORDS:
;
;     next:
;
;        get the next block of data
;
;     previous:
;
;        get the previous block of data
;
;  USAGE:
;
;     IDL> ddeis_epa_cuts, 19960529, '04:30:00', 0.06, 30., 1, energy, f
;  
;  HISTORY:
;
;     VERSION 1.00 -- 1/16/98 -- Written, John C. Dorelli
;
;=============================================================================

PRO ddeis_epa_cuts, date, time, de, da, index, energy, f, _extra = e

; prepare input to 'ddeis_block_process'

control1 = {$
             correct: 1, $
             potential: 1, $
             filter: [0., 10000., 0.1, 20000, 0., !pi], $
             ztype: 0, $
             nseg: 1}

; prepare input to 'ddeis_epa_avg'

control2 = {$
             enmin: 0.1, $
             enmax: 20000., $
             de: de, $
             da: da, $
             sigma: 1}

; process one block of data

ddeis_block_get, date, time, block_data, mode_info, file_flag, _extra = e
IF NOT file_flag THEN BEGIN
    print, 'DDEIS_EPA_CUTS:  level1 file not found....'
    return
ENDIF
ddeis_block_process, block_data, mode_info, control1, z_ele, z_ion, err1

; perform energy-pitch-angle bin average

z = z_ele(0)
ddeis_epa_avg, z, control2, out

; create 1-D cut arrays

energy = out.ebar
nba = n_elements(out.abar)
IF index LT 1 OR index GT nba THEN BEGIN
    print, 'DDEIS_EPA_CUTS:  pitch-angle bin index out of range;'
    print, 'setting ''index'' equal to one (first pitch-angle bin).'
    index = 1
ENDIF
f = out.zavg(index-1, *)
idx = where(f NE -1d20)
energy = energy(idx)
f = f(idx)

END



