;==============================================================================
;
;  NAME:
;     
;     ddeis_fp_to_gz
;
;  PURPOSE:
;     
;     This function transforms the Hydra-DDEIS measurmements from
;     fixed payload (FP) to gyro-z (GZ) coordinates.
;
;  ARGUMENTS:
;     
;     z (STRUCTURE):
;        
;        structures containing processed data output from 
;        'ddeis_block_process.'
;
;     gzcoord (STRUCTURE):
;
;        structure containing the average pitch-angles and 
;        average gyrophases of the Hydra-DDEIS measurements contained
;        in the structure 'z' above.  The fields are described below:
;
;        alpha:  fltarr(z.n, 12) -- the average particle pitch-angle
;
;        beta:  fltarr(z.n, 12) -- the average particle gyrophase
;
;  USAGE:
;
;     IDL> ddeis_fp_to_gz, z, gzcoord
;  
;  HISTORY:
;
;     VERSION 1.00 -- 1/14/98 -- Written, John C. Dorelli
;
;=============================================================================

PRO ddeis_fp_to_gz, z, gzcoord

; compute average magnetic field direction and rotation matrix
   
thetab_avg = total(z.thetab)/n_elements(z.thetab)
phib_avg = total(z.phib)/n_elements(z.phib)

rg = [[-sin(phib_avg),  cos(phib_avg),  0],  $
      [-cos(thetab_avg)*cos(phib_avg),  -sin(phib_avg)*cos(thetab_avg),  $
       sin(thetab_avg)], $
      [sin(thetab_avg)*cos(phib_avg),  sin(thetab_avg)*sin(phib_avg), $
       cos(thetab_avg)]]

; compute velocity space unit vectors in FP coordinates

vhat1 = sin(z.theta)*cos(z.phi)
vhat2 = sin(z.theta)*sin(z.phi)
vhat3 = cos(z.theta)

; transform 'vhat' to GZ coordinates

vhatg1 = rg(0, 0)*vhat1 + rg(0, 1)*vhat2 + rg(0, 2)*vhat3
vhatg2 = rg(1, 0)*vhat1 + rg(1, 1)*vhat2 + rg(1, 2)*vhat3
vhatg3 = rg(2, 0)*vhat1 + rg(2, 1)*vhat2 + rg(2, 2)*vhat3

; compute average pitch-angles and gyrophases

gzcoord = {$
           alpha: fltarr(z.n, 12), $
           beta: fltarr(z.n, 12), $
           thetab_avg: thetab_avg, $
           phib_avg: phib_avg}

gzcoord.alpha = acos(vhatg3)
gzcoord.beta = (atan(vhatg2, vhatg1) + 2.*!pi) MOD (2.*!pi)

END






