;***************************************************************************
;* 
;* PROCEDURE:     
;*      p_hydra_spec, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type DATA_TYPE for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale     
;*
;* INPUTS:       
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              For this data type has the following meaning:   
;*
;*      type(0) 0 for electrons
;*              1 for ions
;*              2 for both in one plot (iowa style).
;*
;*      type(2) 0 data from all pitch angles
;*              1 only field-aligned measurements
;*              2 only perpendicular 
;*              3 only field opposed measurements
;*
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.   
;* 
;* CALLING SEQUENCE:
;*       p_hydra_spec, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;*
;***************************************************************************

pro p_hydra_spec, panel, type, $
              OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                  subtable=subtable, _EXTRA=extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Linestyles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; Comments here should include an explanation of the plot options used
; in type, swwitch or channel. Replace type in the call with whatever variables
; you need

   common hydra_spec_data, $
     data_context, $            ; selects current data set
     spec_type_names, $
     spec_header, $
     spec_data, $
     spec_aligned_data, $
     spec_perp_data, $
     spec_opposed_data, $
     spec_skew_data, $
     hr_spec_data, $
     hr_spec_aligned_data, $
     hr_spec_perp_data, $
     hr_spec_opposed_data, $
     hr_spec_skew_data

   common hydra_user_configure_common, user_configuration_data, $
     user_configure_data_tmp

   if n_elements( user_configuration_data ) eq 0 then $
     hydra_rw_user_configuration, /read
       
   
; One of the common blocks should contain the plot data as returned by the
; data read procedure. 
; The following common blocks are needed:
   
   common mjdt, mjdt_start, mjdt_end ;common time limits in mjdt
   common time, xut1, xut2
;   common hydra_yscale, yscl          ;man/auto yscaling
   common yscale, yscl
   common zscale, zscl
;   help, zscl
   common coordinateSystems, plotted_x, plotted_y  
                                ;info on coords, used by mouse
                                ;                   functions
   common papco_color_names

   panelset,panel               ;sets the panel position viewport
   PAPCO_set_subtable, subtable ;set the color space
   
;  HYDRA-IOWA uses gif file output, which is converted to pcl.  The
;  following line uses a double stroke font for this case.
   if !d.name eq 'Z' then font='!5' else font='!3'
   
;  select ticks that fall on even minutes, hours, days, etc...
   goodticks, xtickv, xminor
   xticks= n_elements(xtickv)-1

;  copy into our local buffer the data set.
   exe_str='data=' + spec_type_names(data_context)+'_data' 
   r=execute( exe_str )         ; neat!

   pitch_range= spec_header( data_context ). pitch_range

   t90= data(*).time + ( spec_header(data_context).start_time - xut1 )
   dt= t90(1)-t90(0)
   dxut= xut2-xut1

   rvis= where( t90 ge 0 and t90 le dxut and data(*).mode gt 0, nvis )
   if nvis lt 2 then begin
       plotted_x = !x
       plotted_y = !y
       print, 'p_hydra_spec: no data in interval'
       return
   endif

   spec_matrix= data(rvis).spectrum(*)
   t90=t90(rvis)

   sc_pot= data(rvis).sc_pot
   sc_pot_qual= data(rvis).sc_pot_qual

;  find the unique energy levels that are visible in this period.
   mode_energy= spec_header(data_context).mode_energy
   modes= data(rvis).mode
   mode_index= modes( uniq( modes ) )-1
   if (n_elements(mode_index) gt 1) then $
    mode_index = mode_index(uniq(mode_index,sort(mode_index)))
   mode_energy_part= mode_energy(*,mode_index)
   r = where(mode_energy_part NE 0.0)
   energy= mode_energy_part( where( mode_energy_part ne 0.0 ) )
   u = uniq( energy, sort(energy) )
   energy= energy(u)

;  form a map from energy index to unique energy level number.
   nen= ( size( mode_energy ) )(1)
   block_energy_index = make_array( size=size(mode_energy), /int)
   for m=0, n_elements(mode_index)-1 do begin
       mb = mode_index(m)
       for j=0, nen-1 do begin
           block_energy_index(j,mb) = where(energy EQ mode_energy(j,mb))
       endfor
   endfor

;  form the spectrogram
   neng= n_elements( energy )

   avg1 = make_array( neng, nvis, /float, value=1.e-20 )
   if (n_elements(mode_index) eq 1) then begin ; no mode changes, simpler case
       mb = mode_index(0)
       r = where(block_energy_index(*,mb) ge 0)
       avg1 = spec_matrix(r,*)
   endif else begin
       for im=0,n_elements(mode_index)-1 do begin
           mb = mode_index(im)
           r = where(block_energy_index(*,mb) ge 0)
           for en=0, n_elements(r)-1 do begin
               ir = where(data(rvis).mode EQ mb+1)
               avg1(block_energy_index(r(en),mb),ir) = $
                 spec_matrix(r(en),ir)
           endfor
       endfor
   endelse

   if data_context lt 5 then res='sv' else res='hr'

   dc= data_context

   if data_context lt 5 then $
     plot_name= 'PAPCO_SPEC_SV' $
   else $
     plot_name= 'PAPCO_SPEC_L1'

   plot_version= hydra_check_version( plot_name, $
                                      [ spec_header(dc).data_version, -1, $
                                        spec_header(dc).reader_version ], qf )

   if qf eq 0 then $
     r= widget_message( ['Plot is not','the best available.'], /info )
   

   spec_shift= user_configuration_data.spec_shift

   channelwidth=0.07		;sets % of channel width to get top
				;and bottom bounds
   case type(0) of
     0:begin			;electrons
         rele=where(energy lt 0)
         ymat=rotate(avg1(rele,*),3)
         en_e=energy(rele)
         energy= -reverse(en_e)
         yminarr=reverse(en_e)*(1-channelwidth)*(-1)
         ymaxarr=reverse(en_e)*(1+channelwidth)*(-1)
         yst=yminarr(0)
         yen=ymaxarr(n_elements(ymaxarr)-1)
         ;invert the array to have increasing energy y-scale
         ctit2='HYDRA!cEle'
         this_Par={ylog:1}
         zmax = 500.
         r = where( ymat gt zmax , count )
         if ( count gt 0 ) then zmax = 4000.
         zmin = 0.1
       end
     1:begin			;ions
         rspecies=where(energy gt 0)
         ymat=transpose(avg1(rspecies,*))
         en_i=energy(rspecies)
         energy= en_i
         yminarr=reverse(en_i)*(1-channelwidth)
         ymaxarr=reverse(en_i)*(1+channelwidth)
         yst=yminarr(0)
         yen=ymaxarr(n_elements(ymaxarr)-1)
         ctit2='HYDRA!cIon'
         this_Par={ylog:1}
         zmax = 100.
         zmin = 0.1
         sc_pot= -1.0 * sc_pot
       end
   endcase

   if spec_shift then cytit='E Debye, eV' else $
     cytit='E obs,eV'

   if user_configuration_data.spec_shift then $
     hydra_interp, ymat, sc_pot, energy

   if user_configuration_data.spec_dJdE then begin
       efluxfact = 187970. 
       ztit= 'dJ/dE'
   endif else begin
       efluxfact =1.
       ztit= 'counts'
   endelse

   if pitch_range(1) - pitch_range(0) ne 180 then begin
       if pitch_range(0) eq 0 then $
         ctit2= ctit2+ ' !9#!x'
       if pitch_range(1) eq 180 then $
         ctit2= ctit2+ ' -!9#!x' 
       if total(pitch_range)/2 eq 90 then $
         ctit2= ctit2+ ' !9x!x'
       
       ctit2=ctit2+'!c!a!i'+strtrim(fix(pitch_range(0)),2)+'<!7a!x<'+$
         strtrim(fix(pitch_range(1)),2)+'!n'
   endif

   ctit=' '
   
   ctit2=font+ctit2
   cytit=font+cytit

; axis range vector...
   if n_elements(yscl) eq 0 then yscl=dblarr(20,4)


; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data. 
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

  ymin=min(yminarr) & ymax=max(ymaxarr)

; Y axis scaling
  yscl(panel(0),0) = overplot
  if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1)
      yen=yscl(panel(0),2)
  endif else begin
      print, 'setting overplot range...'
      yscl(panel(0),1)=ymin
      yscl(panel(0),2)=ymax
      yscl(panel(0),3)=1        ; ylog=1      
  endelse
  
; "Z" axis scaling
  if (zscl(panel(0),0) eq 1) then begin
      zmin=zscl(panel(0),1)
      zmax=zscl(panel(0),2)
  endif else begin
      zscl(panel(0),1)=zmin
      zscl(panel(0),2)=zmax
  endelse

; set up extra plot keywords, common ones first

  extra_plotPar_common={ xrange:[0,xut2-xut1], xstyle:1, $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xminor:xminor, xtickv:xtickv, xticks:xticks, $
                         ytitle:cytit, yticklen:-0.010, ylog:1, $
                         yrange:[ymin,ymax], ystyle:1 }
  
; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.
  
  if n_elements(extra_par) EQ 0 then $
    extra_par={color:1}

; add keyword structure set here with the one passed in

   extra_plotPar_common=create_struct(extra_plotPar_common, extra_par)

; add keyword structure specific to the subtype of plot chosen here
  
   extra_plotPar_common=create_struct(extra_plotPar_common, this_par)

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot 

   if panel(0) eq 0 then $
     extra_plotPar=create_struct({xtickformat:'hydra_timeticks'}, $
                                 extra_plotPar_common) $
   else $
     extra_plotPar=create_struct({xtickformat:'noticks'}, $
                                 extra_plotPar_common)


; plot the axis in color=1, and only plot if if you're not overplotting
   color1_plotpar=create_struct({color:1}, extra_plotPar)

; extend this structure by ctit if the panel is topmost
   if (panel(1) eq panel(0)+panel(2)) then $
     color1_plotpar=create_struct({title:ctit}, $
                                  color1_plotpar)

; store the coordinate information into common block
   plotted_x = !x
   plotted_y = !y

   hydra_plot_clr, ymat, t90, yminarr, zrange=[zmin,zmax], /zlog, $
     nodata=1e-20, _extra=color1_plotPar
   hydra_color_bar,zmin*efluxfact,zmax*efluxfact,ztit,/zlog

; overplot space craft potential
   if user_configuration_data.spec_shift eq 0 then begin

       rokay= where( ( sc_pot_qual and 12 ) eq 0 )
       if rokay(0) ne -1 then oplot, t90(rokay), sc_pot(rokay), color=black, $
         min_value=0, max_value=1000
       
       rwarn= where( ( sc_pot_qual and 12 ) gt 0 )
       if rwarn(0) ne -1 then oplot, t90(rwarn), sc_pot(rwarn), color=black, $
         linestyle=2, min_value=0, max_value=1000 ; dotted
   endif

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.

  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
     r= where( (xtickv+xut1) mod 86400 eq 0, count )
     if r(0) ne -1 then begin
         xtickv= xtickv(r)
         xticks= count
     endif else begin
         xtickv= xtickv([0,n_elements(xtickv)-1])
         xticks= 1
     endelse

     plot, t90, t90, /NODATA, $
       xrange=extra_plotPar_common.xrange, $
       color=1, xticks=xticks, xtickv=xtickv, $
       xtickformat='hydra_juldate', xminor=0, $
       xstyle=1, ystyle=5, xticklen=-0.03
     x_side_label,panel,font+'time (UT)!CDate'
      
  endif

; plot info right of the panel. If you want overplots to have their own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION. Since here colorbars are plotted labels are rotated
; by 90 degrees and then plotted.

   if not keyword_set(overplot) then $
     right_side_label,panel,ctit2,/rot90

;  print plot identifier
   if qf eq 1 then qstr='(BA)' $
   else if qf eq 0 then qstr='(NBA)' $
   else qstr=''

   if qf ge 0 then begin
       footprint_str= strlowcase( plot_name ) + $
         ':'+ strtrim( string(plot_version,format='(f9.2)'), 2 ) + qstr 
       xyouts, 0.99, !p.position(3), /normal, $
         footprint_str, charsize= !p.charsize*0.7, $
         orientation=-90., color=black
   endif else hydra_footprint, spec_header(data_context)
end 

