PRO dp_hydra_moments, date, time, sc_potential, density, t, u, q, $
                      temp_tensor, b_avg, b_avg_sig, b_avg_status, $
                      b_avg_angle_cone, $
                      diagonalization_status, tparl, tperp, gyrotropy, skew, $
                      angle_bavg_bproxy, angle_bavg_bproxy_sig, b_proxy, $
                      data, charge, mass

; PURPOSE:  Calculate derived parameters from the moments quantities

   myname = 'dp_hydra_moments'

   ntime = n_elements(time)

; Define the payload 4-vectors
   
   u_pay = fltarr(4,ntime)
   u_pay(0:2,*) = u
   u_pay(3,*) = sqrt(total(u_pay(0:2,*)^2, 1, /double))
   u_pay =  u_pay*1.e-5         ;km/s
   
   b_pay = fltarr(4,ntime)
   b_pay(0:2,*) = b_avg
   b_pay(3,*) = sqrt(total(b_pay(0:2,*)^2, 1, /double))
   
   b_pay_sig = fltarr(4,ntime)
   b_pay_sig(0:2,*) = b_avg_sig
   
; Define the 'non-zero b' filter 
   gdb = where(b_pay(3,*) GT 0.)
   
   if gdb(0) ne -1 then begin
       b_pay_sig(3,gdb) = $
         sqrt( (b_pay(0,gdb)/b_pay(3,gdb)*b_pay_sig(0,gdb))^2 + $
               (b_pay(1,gdb)/b_pay(3,gdb)*b_pay_sig(1,gdb))^2 + $
               (b_pay(2,gdb)/b_pay(3,gdb)*b_pay_sig(2,gdb))^2 )
   endif

   q_pay = fltarr(4,ntime)
   q_pay(0:2,*) = q
   q_pay(3,*) = sqrt(q(0,*)^2+q(1,*)^2+q(2,*)^2)

   bproxy = fltarr(4,ntime)
   bproxy(0:2,*) = b_proxy
   bproxy(3,*) = sqrt( bproxy(0,*)^2 + bproxy(1,*)^2 + bproxy(2,*)^2 )
   
; Rotate the flow speed and mag. field  and heat flux from fixed 
; payload to gsm----------
   time =  reform(time)
   print, myname+': Rotating u to GSM...'
   hydra_payload_to_gsm,date,time,u_pay,u_gsm,istat,rate=0
   IF (istat EQ 0) THEN BEGIN
      print, myname+': GSM ROTATION of u_pay NOT SUCCESSFUL!'
      u_gsm = fltarr(4,ntime)
   ENDIF
   print, myname+': Rotating b to GSM...'
   hydra_payload_to_gsm,date,time,b_pay,b_gsm,istat,rate=0
   IF (istat EQ 0) THEN BEGIN
      print, myname+': GSM ROTATION of b_pay NOT SUCCESSFUL!'
      b_gsm = fltarr(4,ntime)
   ENDIF
   print, myname+': Rotating q to GSM...'
   hydra_payload_to_gsm,date,time,q_pay,q_gsm,istat,rate=0
   IF (istat EQ 0) THEN BEGIN
      print, myname+': GSM ROTATION of q_pay NOT SUCCESSFUL!'
      q_gsm = fltarr(4,ntime)
   ENDIF

; Rotate the flow speed and mag. field  and heat flux from fixed 
; payload to gse----------
   time =  reform(time)
   print, myname+': Rotating u to GSE...'
   hydra_payload_to_gse,date,time,u_pay,u_gse,istat,rate=0
   IF (istat EQ 0) THEN BEGIN
      print, myname+': GSE ROTATION of u_pay NOT SUCCESSFUL!'
      u_gse = fltarr(4,ntime)
   ENDIF
   print, myname+': Rotating b to GSE...'
   hydra_payload_to_gse,date,time,b_pay,b_gse,istat,rate=0
   IF (istat EQ 0) THEN BEGIN
      print, myname+': GSE ROTATION of b_pay NOT SUCCESSFUL!'
      b_gse = fltarr(4,ntime)
   ENDIF
   print, myname+': Rotating q to GSE...'
   hydra_payload_to_gse,date,time,q_pay,q_gse,istat,rate=0
   IF (istat EQ 0) THEN BEGIN
      print, myname+': GSE ROTATION of q_pay NOT SUCCESSFUL!'
      q_gse = fltarr(4,ntime)
   ENDIF
   
; Magnitudes
   b_mag = b_pay(3,*)           ; = b_gsm(3,*) = b_gse(3,*)
   u_mag = u_pay(3,*)           ; = u_gsm(3,*) = u_gse(3,*)
   q_mag = q_pay(3,*)           ; = q_gsm(3,*) = q_gse(3,*)

; Calculate the derived parameters
   mean_energy =  t*1.4e-16/1.6e-12 ;eV
   ktparl =  tparl*1.4e-16/1.6e-12 ;eV
   ktperp =  tperp*1.4e-16/1.6e-12 ;eV
   anisotropy = fltarr(n_elements(ktparl))
   gd = where(ktperp GT 0.)
   anisotropy(gd) =  ktparl(gd)/ktperp(gd)
   pres_perp = density*ktperp*1.6e-12 ;cgs
   pres_parl = density*ktparl*1.6e-12 ;cgs
   
; Flow speed components
   u_parl = fltarr(ntime)
   u_parl(gdb) = 1./(b_mag(gdb))*(u_pay(0,gdb)*b_pay(0,gdb) + u_pay(1,gdb)*b_pay(1,gdb) + u_pay(2,gdb)*b_pay(2,gdb)) ;km/s
   
   u_perp_gsm =  fltarr(4,ntime)
   u_perp_pay =  fltarr(4,ntime)
   for ic=0,2 do begin
       u_perp_gsm(ic,gdb) =  u_gsm(ic,gdb) - u_parl(gdb)*b_gsm(ic,gdb)/b_mag(gdb) ;km/s
       u_perp_pay(ic,gdb) =  u_pay(ic,gdb) - u_parl(gdb)*b_pay(ic,gdb)/b_mag(gdb) ;km/s
   endfor
   u_perp_pay(3,*) =  sqrt(total(u_perp_pay(0:2,*)^2,1,/double)) ; km/s
   u_perp_gsm(3,*) =  sqrt(total(u_perp_gsm(0:2,*)^2,1,/double)) ; km/s
   
; Alfven and mach number quantities
   mass_ion = 1.67e-24          ; cgs, hydrogen
   bgauss = b_mag*1.e-9*1.e+4   ; cgs
   rho = density*mass_ion       ; cgs
   v_alf = bgauss/sqrt(4.*(!pi)*rho) ; cgs
   v_alf = v_alf*1.e-5          ; km/s
   m_tot = fltarr(ntime)
   m_tot(gdb) = u_mag(gdb)/v_alf(gdb)
   m_parl = fltarr(ntime)
   m_parl(gdb) = u_parl(gdb)/v_alf(gdb)
   m_perp = fltarr(ntime)
   m_perp(gdb) = u_perp_gsm(3,gdb)/v_alf(gdb)
   
; Heat flux components
   q_parl = fltarr(ntime)
   q_parl(gdb) = 1./(b_mag(gdb))*(q_gsm(0,gdb)*b_gsm(0,gdb) + q_gsm(1,gdb)*b_gsm(1,gdb) + q_gsm(2,gdb)*b_gsm(2,gdb))
   q_perp_gsm =  fltarr(4,ntime)
   q_perp_gsm(0,gdb) = q_gsm(0,gdb) - q_parl(gdb)*b_gsm(0,gdb)/b_mag(gdb)
   q_perp_gsm(1,gdb) = q_gsm(1,gdb) - q_parl(gdb)*b_gsm(1,gdb)/b_mag(gdb)
   q_perp_gsm(2,gdb) = q_gsm(2,gdb) - q_parl(gdb)*b_gsm(2,gdb)/b_mag(gdb)
   q_perp_gsm(3,*) =  sqrt(q_perp_gsm(0,*)^2 + q_perp_gsm(1,*)^2 + q_perp_gsm(2,*)^2)
   q_perp_pay =  fltarr(4,ntime)
   q_perp_pay(0,gdb) = q_pay(0,gdb) - q_parl(gdb)*b_pay(0,gdb)/b_mag(gdb)
   q_perp_pay(1,gdb) = q_pay(1,gdb) - q_parl(gdb)*b_pay(1,gdb)/b_mag(gdb)
   q_perp_pay(2,gdb) = q_pay(2,gdb) - q_parl(gdb)*b_pay(2,gdb)/b_mag(gdb)
   q_perp_pay(3,*) =  sqrt(q_perp_pay(0,*)^2 + q_perp_pay(1,*)^2 + q_perp_pay(2,*)^2)
   qparl_over_bmag = fltarr(ntime)
   qparl_over_bmag(gdb) = q_parl(gdb)/b_mag(gdb) ; erg/cm^2/s/nT
   
; Heat flux and velocity organization with b_pay
   qx = q_pay(0,gdb)
   qy = q_pay(1,gdb)
   qz = q_pay(2,gdb)
   qmod = q_pay(3,gdb)
   
   ux = u_pay(0,gdb)
   uy = u_pay(1,gdb)
   uz = u_pay(2,gdb)
   umod = u_pay(3,gdb)
   
   bx = b_pay(0,gdb)
   by = b_pay(1,gdb)
   bz = b_pay(2,gdb)
   bmod = b_pay(3,gdb)
   
   bxsig = b_pay_sig(0,gdb)
   bysig = b_pay_sig(1,gdb)
   bzsig = b_pay_sig(2,gdb)
   bmodsig = b_pay_sig(3,gdb)
   
   cosangle_q_bavg = fltarr(ntime)
   cosangle_u_bavg = fltarr(ntime)
   cosangle_q_bavg(gdb) = (qx*bx + qy*by + qz*bz)/qmod/bmod
   cosangle_u_bavg(gdb) = (ux*bx + uy*by + uz*bz)/umod/bmod
   
   dhdbx = qx/qmod/bmod
   dhdby = qy/qmod/bmod
   dhdbz = qz/qmod/bmod
   dhdbmod = -1.*cosangle_q_bavg(gdb)/bmod
   cosangle_q_bavg_sig = fltarr(ntime)
   cosangle_q_bavg_sig(gdb) = sqrt( (dhdbx*bxsig)^2 + (dhdby*bysig)^2 + $
                                   (dhdbz*bzsig)^2 + (dhdbmod*bmodsig)^2 )
   
   dhdbx = ux/umod/bmod
   dhdby = uy/umod/bmod
   dhdbz = uz/umod/bmod
   dhdbmod = -1.*cosangle_u_bavg(gdb)/bmod
   cosangle_u_bavg_sig = fltarr(ntime)
   cosangle_u_bavg_sig(gdb) = sqrt( (dhdbx*bxsig)^2 + (dhdby*bysig)^2 + $
                                   (dhdbz*bzsig)^2 + (dhdbmod*bmodsig)^2 )

; ENERGY FLUX =====================================
   kb = 1.38d-16                ; ergs/K
   tracep = density(*)*kb*(temp_tensor(0,0,*)+temp_tensor(1,1,*)+temp_tensor(2,2,*))
   vec1 = q_pay(0:2,*)          ; cgs
   vec2 = dblarr(3,ntime)
   vec3 = dblarr(3,ntime)
   vec4 = dblarr(3,ntime)
   for ic=0,2 do begin 
       vec2(ic,*) = $           ; cgs
         density(*)*kb*temp_tensor(ic,0,*)*u_pay(0,*)*1.d+5 + $
         density(*)*kb*temp_tensor(ic,1,*)*u_pay(1,*)*1.d+5 + $
         density(*)*kb*temp_tensor(ic,2,*)*u_pay(2,*)*1.d+5
       vec3(ic,*) = .5d0*mass*density(*)*(u_pay(3,*)*1.d+5)^2*u_pay(ic,*)*1.d+5
       vec4(ic,*) = .5d0*tracep(*)*u_pay(ic,*)*1.d+5
   endfor
   eflux = vec1 + vec2 + vec3 + vec4

; 2/3/99 New variables
   eflux_parl = dblarr(ntime)
   eflux_parl(gdb) = 1./(b_mag(gdb))*(eflux(0,gdb)*b_pay(0,gdb) + eflux(1,gdb)*b_pay(1,gdb) + eflux(2,gdb)*b_pay(2,gdb)) ; ergs/cm^2/s   
   eflux_parl_over_b = dblarr(ntime)
   eflux_parl_over_b(gdb) = eflux_parl(gdb)/b_mag(gdb)
   
; Index array
   r= lindgen(ntime)
   
; Define the output structure
   
; One-dimensional arrays
   data(r).time = reform(time)  ; This will later change to seconds since start time (ssst)
   data(r).time_ssm = reform(time) ; This will hold the time in seconds since midnight (ssm)
   data(r).date = date
   data(r).mean_energy= reform(mean_energy)
   data(r).density = reform(density)
   data(r).ktparl = reform(ktparl)
   data(r).ktperp = reform(ktperp)
   data(r).pres_parl = reform(pres_parl)
   data(r).pres_perp = reform(pres_perp)
   data(r).gyrotropy = reform(gyrotropy)
   data(r).skew = reform(skew)
   data(r).anisotropy = reform(anisotropy)
   data(r).u_parl = reform(u_parl)

; New 2/2/99
   cgs2mapms = 1.6d-19/4.8d-10*100.^2*1.d+6 ; esu/s/cm^2 to microAmps/m^2
   data(r).qnu_parl = data(r).u_parl*1.d+5*charge*data(r).density*cgs2mapms

;==========================DIAGS==========================================
;   if (charge lt 0) then openw, unit, 'dp_hydra_moments_ele_data.txt', /get_lun else openw, unit, 'dp_hydra_moments_ion_data.txt', /get_lun
;   for ii = 0, n_elements(r)-1 do begin
;       printf, unit, data(r(ii)).time_ssm, $
;         u_pay(0,r(ii)), u_pay(1,r(ii)), u_pay(2,r(ii)), $
;         b_pay(0,r(ii)), b_pay(1,r(ii)), b_pay(2,r(ii)), b_pay(3,r(ii)), $
;         density(r(ii)), $
;         data(r(ii)).qnu_parl
;   endfor
;   close, unit
;   free_lun, unit
;==========================DIAGS==========================================


; New 2/3/99
   data(r).eflux_parl = reform(eflux_parl) ; ergs/cm^2/s
   data(r).eflux_parl_over_b = reform(eflux_parl_over_b) ; ergs/cm^2/s/nT

   data(r).v_alf = reform(v_alf)
   data(r).m_tot = reform(m_tot)
   data(r).m_parl = reform(m_parl)
   data(r).m_perp = reform(m_perp)
   data(r).sc_potential =  reform(sc_potential)
   data(r).q_parl =  reform(q_parl)
   data(r).qparl_over_bmag = reform(qparl_over_bmag)
   data(r).b_pay_angle_cone = reform(b_avg_angle_cone)
   data(r).b_pay_status = reform(b_avg_status)
   data(r).diagonalization_status = reform(diagonalization_status)
   data(r).angle_bavg_bproxy = reform(angle_bavg_bproxy)
   data(r).angle_bavg_bproxy_sig = reform(angle_bavg_bproxy_sig)
   data(r).cosangle_q_bavg = reform(cosangle_q_bavg)
   data(r).cosangle_q_bavg_sig = reform(cosangle_q_bavg_sig)
   data(r).cosangle_u_bavg = reform(cosangle_u_bavg)
   data(r).cosangle_u_bavg_sig = reform(cosangle_u_bavg_sig)
   
; Arrays of more than one dimension
   data(r).b_pay = b_pay
   data(r).b_pay_sig = b_pay_sig
   data(r).b_proxy = bproxy
   data(r).b_gsm = b_gsm
   data(r).b_gse = b_gse
   data(r).u_pay = u_pay
   data(r).u_gsm = u_gsm
   data(r).u_gse = u_gse
   data(r).u_perp_gsm = u_perp_gsm
   data(r).u_perp_pay = u_perp_pay
   data(r).q_pay = q_pay
   data(r).q_gsm = q_gsm
   data(r).q_gse = q_gse
   data(r).q_perp_gsm = q_perp_gsm
   data(r).q_perp_pay = q_perp_pay
   
   return
END                             ;*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*

