function hydra_check_version, plot_identifier, versions, $
                              quality_flag, verbose=verbose, $
                              version_strings=version_strings, $
                              use_web_file=use_web_file, echo=echo
;
;   Read plot version summary file.
;
;   INPUT:  plot_identifier = string indicating type of plot
;   INPUT:  versions = float array of versions to test
;
;   OUTPUT: plot_version = absolute Plot Version
;   OUTPUT: plot_number = absolute HYDRA Plot Number
;   OUTPUT: quality_flag = .true. if it is up-to-date
;   
;   OUTPUT: deficiency returns a string stating what needs to be updated
;

   quality_flag=-1 

   if n_params() eq 0 then begin
       print, 'usage: plot_version= hydra_check_version( plot_identifier, versions, qf )'
       print, '          plot_identifier  is a string identifying type of plot'
       print, '          versions         is float array [ DATA_VERS, LIB_VERS, CODEV ]'
       print, '          qf               is 1 if plot is latest version, 0 otherwise.'
       return, -1
   endif

   COMMON plot_version_web_comm, define, plot_vers, nversions, nplots, nparts

   if getenv( 'hydra_check_version_verbose' ) ne '' then verbose=1

   IF NOT keyword_set(define) THEN BEGIN 
;   if 1 then begin
       
      IF keyword_set(verbose) THEN print, 'Reading plot_version Record.'
      message, 'Getting Hydra version information via the net...', /cont

      filename = 'plot_version_summary_v00.dat'
      
      wget_exe= getenv( 'wget_exe' )
      if wget_exe eq '' then begin
          message, 'environment variable wget_exe does not exist', /cont
          qf=-1
          return, -1
      endif

      wget_timeout= '-T20'      ; seconds
      wget_options= '-q'
      command = wget_exe+' '+wget_timeout+' '+wget_options+' '+ $
        ' http://supras.physics.uiowa.edu/hydra_data/versions/'+ $
        filename
      
      spawn, command+' -O - ', unit=unit
      
      nversions = 100
      nplots = 100
      nparts = 6
      
      plot_vers1 = {plot_v1, $
                    plotv: float(0.0), $
                    nparts: 0L, $
                    parts: fltarr(nparts) }

      plot_vers2 = {plot_v2, $
                    nversions: 0L, $
                    identifier: string(""), $
                    versions: replicate(plot_vers1, nversions)}

      plot_vers = replicate(plot_vers2, nplots)

      line = string("")

      iplot = -1

      readf, unit, line
      IF keyword_set(echo) THEN print, line
      WHILE (strmid(line, 0, 3) NE 'END') DO BEGIN
         CASE strmid(line,0,4) OF
            'DDEI': BEGIN
               ident = (str_sep(strcompress(line),' '))(1)
               iplot = iplot+1
               iversion = -1
               plot_vers(iplot).nversions = iversion
               plot_vers(iplot).identifier = ident
            END
            'VERS': BEGIN
               iversion = iversion+1
               plot_vers(iplot).nversions = iversion
               line1 = str_sep(strcompress(line),' ')
               nums = float(line1(1:n_elements(line1)-1))
               plot_vers(iplot).versions(iversion).plotv = nums(0)
               plot_vers(iplot).versions(iversion).nparts = n_elements(nums)-1
               plot_vers(iplot).versions(iversion).parts = $
                nums(1:n_elements(nums)-1)
            END
            ELSE: BEGIN
            END
         ENDCASE 
         readf, unit, line
         IF keyword_set(echo) THEN print, line
      ENDWHILE 
      free_lun, unit
      close, unit
      
      IF keyword_set(use_web_file) THEN BEGIN 
         spawn, 'rm '+file
      ENDIF 
      define = 1
   ENDIF ELSE BEGIN
      IF keyword_set(verbose) THEN print, 'Not Reading plot_version Record.'
   ENDELSE

   nv = n_elements(versions)
   iver = lonarr(nv)
   FOR i=0,nv-1 DO BEGIN
      IF (versions(i) NE -1.0) THEN iver(i) = 0 ELSE iver(i) = 1
   ENDFOR 
   

   IF keyword_set(verbose) THEN BEGIN 
      print, 'INPUT Version Parameters: '
      FOR i=0,n_elements(versions)-1 DO BEGIN
         IF (versions(i) NE -1.0) THEN $
          print, i, versions(i) ELSE print, i, '*'
      ENDFOR 
   ENDIF 

   vers = long(versions*100+0.01)/100.0

   iplot = 0
   WHILE (plot_vers(iplot).identifier NE plot_identifier) AND $
    (iplot LT nplots-1) DO BEGIN
      iplot = iplot+1
   ENDWHILE 
   IF (plot_vers(iplot).identifier NE plot_identifier) THEN BEGIN
      message, 'HYDRA PLOT IDENTIFIER "'+plot_identifier+$
        '" NOT REGISTERED. ABORTING.', /cont
      return, -1
   ENDIF 
   plot_number = iplot
   nv = plot_vers(iplot).nversions

   iversion = 0

   logical_v = 1
   FOR i=0,plot_vers(iplot).versions(iversion).nparts-1 DO BEGIN 
      logical_v = logical_v AND $
       ((plot_vers(iplot).versions(iversion).parts(i) EQ vers(i)) OR iver(i))
   ENDFOR
;   print, vers
;   print, plot_vers(iplot).versions(iversion).parts
   WHILE ((NOT logical_v) AND (iversion LT nv)) DO BEGIN
      iversion = iversion + 1

      logical_v = 1
      FOR i=0,plot_vers(iplot).versions(iversion).nparts-1 DO BEGIN 
         logical_v = logical_v AND $
          ((plot_vers(iplot).versions(iversion).parts(i) EQ vers(i)) $
           OR iver(i))
      ENDFOR
;      print, vers
;      print, plot_vers(iplot).versions(iversion).parts
      
   ENDWHILE 
   IF (logical_v) THEN BEGIN
      plot_version = plot_vers(iplot).versions(iversion).plotv
      IF (plot_version EQ plot_vers(iplot).versions(0).plotv) THEN BEGIN
         quality_flag = 1      
      ENDIF ELSE BEGIN
         quality_flag = 0
      ENDELSE 
   ENDIF ELSE BEGIN
      plot_version = 0.00
      quality_flag = 0
   ENDELSE 
   label_string = ["FILEV", "LIBV", "CODEV"]

   return, plot_version
END

