;==============================================================================
;
;  NAME:
;     
;     ddeis_block_process
;
;  PURPOSE:
;     
;     This function performs the following operations on one block of
;     Hydra-DDEIS data:
;
;        1)  organizes the electron and ion data into separate time-
;            ordered structures
;        2)  processes the data according to the data specified in 
;            the input control structure 'control'
;        3)  selects the type of zdata specified in the input control
;            structure 'control'
;        4)  divides the electron and ion time-ordered structures into
;            'control.nseg' equal segments
;
;  ARGUMENTS:
;     
;     block_data, mode_info (STRUCTURES):
;        
;        structures containing one block of unprocessed Hydra-DDEIS
;        level one data.  These structures are defined by the Hydra
;        level one library routines 'init_level1' and 'read_level1.'
;     
;     control (STRUCTURE):
;         
;        a structure which controls the behavior of 'ddeis_process_block.'
;        A description of the fields of this structure follows:
;
;           correct (INTEGER):
;
;              1:  implement intersweep, background, channeltron
;                  efficiency, and dead time corrections
;              0:  none of the above corrections
;
;           potential (INTEGER):
;
;              1:  implement spacecraft potential correction
;              0:  no spacecraft potential correction
;
;           filter:  intarr(10)
;
;              an array which controls the data filtering process.
;              This array determines how the datamask array 
;              'block_data.datamask' is to be updated:
;
;                 filter(0:1):  allowed count range
;                 filter(2:3):  allowed energy range
;                 filter(4:5):  allowed pitch-angle range
;
;           ztype (INTEGER):
;
;              0:  distribution function
;              1:  counts
;                  NOTE:  If the counts are raw counts, then the
;                         standard deviation is discretization + Poisson.
;
;           nseg (INTEGER):
;
;              The separate ion and electron time-ordered structures
;              will be divided into 'nseg' equal segments.  Any
;              remaining data in the block is discarded.
;
;     z_ele, z_ion ('nseg'-dimensional STRUCTURE ARRAYS):
;
;        structure arrays containing the electron and ion data,
;        respectively, in order of increasing time.  Each element of
;        the array contains a structure with the following fields ('n'
;        is the number of data points in each segment; 'ndet' is the
;        number of DDEIS detectors):
;
;           time:  dblarr(n)
;
;              time in seconds since midnight.  'n' is the number of
;              data points in each segment.
;
;           theta, phi:  fltarr(n, ndet)
;              
;              the incoming particle polar and azimuthal angles
;              in fixed payload coordinates (radians)
;
;           thetab, phib:  fltarr(n, ndet)
;
;              the polar and azimuthal magnetic field angles in fixed 
;              payload coordinates (radians)
;
;           alpha:  fltarr(n, ndet)
;
;              particle pitch-angle in fixed payload coordinates (radians)
;
;           energy, energy_old:  fltarr(n, ndet)
;
;              energies after and before, respectively, the
;              spacecraft potential correction (eV)
;
;           z, sig:  dblarr(n, ndet)
;
;              zdata and associated standard deviations (from error
;              propagation of count discretization and Poisson
;              errors)(CGS units).
;
;           dm:  bytarr(n, ndet)
;
;              a 'data mask' which eliminates invalid counts (e.g.,
;              glint); 0 -> invalid, 1 -> valid
;
;           idx:  intarr(n)
; 
;              time-ordered indices of the species separated data in
;              the 'block data' arrays
;
;           n:  (INTEGER)
;
;              number of species separated data points after filtering
;
;     err:  intarr(10)
;
;        an array of error flags.  A value of 0 indicates normal exit.
;        values of 1 indicate the following for each element:
;
;           err(0):  no electron data found
;           err(1):  no ion data found
;           err(2):  no spacecraft potential available from level1
;           err(3):  distribution function standard deviations not
;                    available
;           err(4):  less than one data point on average in each
;                    segment.
;      
;
;  USAGE:
;
;     IDL> ddeis_process_block, block_data, mode_info, control,
;                               z_ele, z_ion, err
;  
;  HISTORY:
;
;     VERSION 1.00 -- 12/18/97 -- Written, John C. Dorelli
;     VERSION 1.01 -- 1/21/98 -- added calculation of time ordered 
;                                   data indices 'z_ele.idx' and
;                                   'z_ion.idx'
;     VERSION 1.02 -- 1/25/98 -- fixed a bug in s/c potential correction
;                                   (control.potential was ignored)
;
;=============================================================================

PRO ddeis_block_process, block_data, mode_info, control, z_ele, z_ion, err

err = intarr(10)

; perform count corrections

IF control.correct THEN BEGIN
    correct_interpolate, block_data, mode_info
ENDIF ELSE BEGIN
    correct_interpolate, block_data, mode_info, /nocorrect
ENDELSE

; compute angles and distribution function

calculate_angles, block_data, mode_info
calculate_f, block_data, mode_info
sigma_cc_sigma_f, block_data, mode_info

; define some constants

m_ele = 9.11d-28
m_ion = 1.67d-24

; compute dimensions and species indices of output arrays 

nscan = mode_info.scan_length
ndet = 12

idx_ele = where(mode_info.energy(*, 0) LT 0)
idx_ion = where(mode_info.energy(*, 0) GT 0)

IF idx_ele(0) EQ -1 THEN BEGIN
    print, 'DDEIS_BLOCK_PROCESS:  No electron measurements found....'
    err(0) = 1
    return
ENDIF

IF idx_ion(0) EQ -1 THEN BEGIN
    print, 'DDEIS_BLOCK_PROCESS:  No ion measurements found....'
    err(1) = 1
    return
ENDIF

n_ele = n_elements(idx_ele)
n_ion = n_elements(idx_ion)

; define output structures for block

z_ele_block = {electrons_block, $
               time: dblarr(n_ele, ndet), $
               theta: fltarr(n_ele, ndet), $
               phi: fltarr(n_ele, ndet), $
               thetab: fltarr(n_ele, ndet), $
               phib: fltarr(n_ele, ndet), $
               alpha: fltarr(n_ele, ndet), $
               energy: fltarr(n_ele, ndet), $
               energy_old: fltarr(n_ele, ndet), $
               z: dblarr(n_ele, ndet), $
               sig: dblarr(n_ele, ndet), $
               dm: bytarr(n_ele, ndet), $
               idx: intarr(n_ele), $
               n: 0}

z_ion_block = {ions_block, $
               time: dblarr(n_ion, ndet), $
               theta: fltarr(n_ion, ndet), $
               phi: fltarr(n_ion, ndet), $
               thetab: fltarr(n_ion, ndet), $
               phib: fltarr(n_ion, ndet), $
               alpha: fltarr(n_ion, ndet), $
               energy: fltarr(n_ion, ndet), $
               energy_old: fltarr(n_ion, ndet), $
               z: dblarr(n_ion, ndet), $
               sig: dblarr(n_ion, ndet), $
               dm: bytarr(n_ion, ndet), $
               idx: intarr(n_ion), $
               n: 0}

z_ele_block.idx = idx_ele
z_ion_block.idx = idx_ion

; compute spacecraft potential correction

species = intarr(nscan)
species(idx_ele) = -1
species(idx_ion) = 1
potential = block_data.sc_potential

IF control.potential EQ 0 THEN BEGIN
    potential = 0.
ENDIF ELSE IF  block_data.sc_potential_stat EQ 0 THEN BEGIN
    potential = 0.
    err(2) = 1
ENDIF

; fill in total data structure for the block; later, we will break
; this structure into an 'nseg'-dimensional array of structures

z_ele_block.n = n_ele
z_ion_block.n = n_ion

FOR i = 1, ndet DO BEGIN

    ;time tags for individual measurements

    z_ele_block.time(*, i-1)= double(block_data.time+ $
                                     mode_info.time_offset(idx_ele))
    z_ion_block.time(*, i-1)= double(block_data.time+ $
                                     mode_info.time_offset(idx_ion))

    ;particle theta and phi in fixed payload coordinates

    z_ele_block.theta(*, i-1) = block_data.theta_ele(i-1)
    z_ion_block.theta(*, i-1) = block_data.theta_ion(i-1)
    z_ele_block.phi(*, i-1) = block_data.phi_det(idx_ele, i-1)
    z_ion_block.phi(*, i-1) = block_data.phi_det(idx_ion, i-1)

    ;magnetic field thetab and phib in fixed payload coordinates

    z_ele_block.thetab(*, i-1) = block_data.thetab(idx_ele)
    z_ion_block.thetab(*, i-1) = block_data.thetab(idx_ion)
    z_ele_block.phib(*, i-1) = (block_data.phib(idx_ele)+ $
      mode_info.phi_offset(idx_ele) + block_data.phase_start) MOD (2.*!pi)
    z_ion_block.phib(*, i-1) = (block_data.phib(idx_ion)+ $
      mode_info.phi_offset(idx_ion) + block_data.phase_start) MOD (2.*!pi)

    ;spacecraft potential corrected energies and particle pitch-angles 
    ;in fixed payload coordinates

    z_ele_block.alpha(*, i-1) = block_data.alpha(idx_ele, i-1)
    z_ion_block.alpha(*, i-1) = block_data.alpha(idx_ion, i-1)

    IF((i+1) MOD 2 EQ 0)THEN BEGIN
        z_ele_block.energy(*, i-1) = $
          mode_info.energy(idx_ele, 0)*species(idx_ele)+ $
          potential*species(idx_ele)
        z_ele_block.energy_old(*, i-1) = $
          mode_info.energy(idx_ele, 0)*species(idx_ele)
        z_ion_block.energy(*, i-1) = $
          mode_info.energy(idx_ion, 0)*species(idx_ion)+ $
          potential*species(idx_ion)
        z_ion_block.energy_old(*, i-1) = $
          mode_info.energy(idx_ion, 0)*species(idx_ion)
    ENDIF ELSE BEGIN
        z_ele_block.energy(*, i-1) = $
          mode_info.energy(idx_ele, 1)*species(idx_ele)+ $
          potential*species(idx_ele)
        z_ele_block.energy_old(*, i-1) = $
          mode_info.energy(idx_ele, 1)*species(idx_ele)
        z_ion_block.energy(*, i-1) = $
          mode_info.energy(idx_ion, 1)*species(idx_ion)+ $
          potential*species(idx_ion)
        z_ion_block.energy_old(*, i-1) = $
          mode_info.energy(idx_ion, 1)*species(idx_ion)
    ENDELSE

    ; zdata and standard deviations

    CASE control.ztype OF 
        0: $
          BEGIN
            z_ele_block.z(*, i-1) = block_data.f(idx_ele, i-1)
            z_ion_block.z(*, i-1) = block_data.f(idx_ion, i-1)
            z_ele_block.sig(*, i-1) = block_data.sigma_f(idx_ele)
            z_ion_block.sig(*, i-1) = block_data.sigma_f(idx_ion)
          END
        1: $
          BEGIN
            z_ele_block.z(*, i-1) = block_data.corrected_counts(idx_ele, i-1)
            z_ion_block.z(*, i-1) = block_data.corrected_counts(idx_ion, i-1)
            IF control.correct THEN BEGIN
                z_ele_block.sig(*, i-1) = block_data.sigma_cc(idx_ele)
                z_ion_block.sig(*, i-1) = block_data.sigma_cc(idx_ion)
            ENDIF ELSE BEGIN
                craw = block_data.corrected_counts(idx_ele, i-1)
                z_ele_block.sig(*, i-1) = $
                  sqrt(float(long(craw-0.5*dn(craw))) + (dn(craw)^2)/12.)
                craw = block_data.corrected_counts(idx_ion, i-1)
                z_ion_block.sig(*, i-1) = $
                  sqrt(float(long(craw-0.5*dn(craw))) + (dn(craw)^2)/12.)
            ENDELSE
        END
     ENDCASE
     IF total(block_data.sigma_f) EQ 0 THEN err(3) = 1
   
ENDFOR

; update the data mask

cond_ele1 = (block_data.corrected_counts(idx_ele, *) GE control.filter(0) AND $
             block_data.corrected_counts(idx_ele, *) LE control.filter(1))
cond_ion1 = (block_data.corrected_counts(idx_ion, *) GE control.filter(0) AND $
             block_data.corrected_counts(idx_ion, *) LE control.filter(1))
cond_ele2 = (z_ele_block.energy GE control.filter(2) AND $
             z_ele_block.energy LE control.filter(3))
cond_ion2 = (z_ion_block.energy GE control.filter(2) AND $
             z_ion_block.energy LE control.filter(3))
cond_ele3 = (z_ele_block.alpha GE control.filter(4) AND $
             z_ele_block.alpha LE control.filter(5))
cond_ion3 = (z_ion_block.alpha GE control.filter(4) AND $
             z_ion_block.alpha LE control.filter(5))
cond_ele4 = block_data.datamask(idx_ele, *)
cond_ion4 = block_data.datamask(idx_ion, *)

idx_dm_ele = where(cond_ele1 AND cond_ele2 AND cond_ele3 AND cond_ele4)
idx_dm_ion = where(cond_ion1 AND cond_ion2 AND cond_ion3 AND cond_ion4)

z_ele_block.dm = 0b
z_ele_block.dm(idx_dm_ele) = 1b
z_ion_block.dm = 0b
z_ion_block.dm(idx_dm_ion) = 1b

; compute # of time data points in each segment

IF control.nseg NE 0 THEN BEGIN
    n_ele = long(z_ele_block.n/control.nseg)
    n_ion = long(z_ion_block.n/control.nseg)
ENDIF ELSE BEGIN
    print, 'DDEIS_BLOCK_PROCESS:  zero segments selected.  Setting number'
    print, 'of fragments equal to one....'
    n_ele = long(z_ele_block.n)
    n_ion = long(z_ion_block.n)
ENDELSE

IF n_ele EQ 0 OR n_ion EQ 0 THEN BEGIN
    print, 'DDEIS_BLOCK_PROCESS:  less than one data point on average in'
    print, 'each segment.'
    n_ele = long(z_ele_block.n)
    n_ion = long(z_ion_block.n)
    err(4) = 1
ENDIF

; define time-ordered structure segments

z_ele_seg = {$
             time: dblarr(n_ele, ndet), $
             theta: fltarr(n_ele, ndet), $
             phi: fltarr(n_ele, ndet), $
             thetab: fltarr(n_ele, ndet), $
             phib: fltarr(n_ele, ndet), $
             alpha: fltarr(n_ele, ndet), $
             energy: fltarr(n_ele, ndet), $
             energy_old: fltarr(n_ele, ndet), $
             z: dblarr(n_ele, ndet), $
             sig: dblarr(n_ele, ndet), $
             dm: bytarr(n_ele, ndet), $
             idx: intarr(n_ele), $
             n: 0}

z_ion_seg = {$
             time: dblarr(n_ion, ndet), $
             theta: fltarr(n_ion, ndet), $
             phi: fltarr(n_ion, ndet), $
             thetab: fltarr(n_ion, ndet), $
             phib: fltarr(n_ion, ndet), $
             alpha: fltarr(n_ion, ndet), $
             energy: fltarr(n_ion, ndet), $
             energy_old: fltarr(n_ion, ndet), $
             z: dblarr(n_ion, ndet), $
             sig: dblarr(n_ion, ndet), $
             dm: bytarr(n_ion, ndet), $
             idx: intarr(n_ion), $
             n: 0}

; create time-ordered structure arrays

z_ele = replicate(z_ele_seg, control.nseg)
z_ion = replicate(z_ion_seg, control.nseg)

FOR i = 1, control.nseg DO BEGIN

    z_ele(i-1).time = $
      z_ele_block.time(n_ele*(i-1):n_ele*i-1, *)
    z_ion(i-1).time = $
      z_ion_block.time(n_ion*(i-1):n_ion*i-1, *)

    ; particle theta and phi in fixed payload coordinates

    z_ele(i-1).theta = $
      z_ele_block.theta(n_ele*(i-1):n_ele*i-1, *)
    z_ele(i-1).phi = $
      z_ele_block.phi(n_ele*(i-1):n_ele*i-1, *)
    z_ion(i-1).theta = $
      z_ion_block.theta(n_ion*(i-1):n_ion*i-1, *)
    z_ion(i-1).phi = $
      z_ion_block.phi(n_ion*(i-1):n_ion*i-1, *)

    ; magnetic field thetab and phib in fixed payload coordinates

    z_ele(i-1).thetab = $
      z_ele_block.thetab(n_ele*(i-1):n_ele*i-1, *)
    z_ele(i-1).phib = $
      z_ele_block.phib(n_ele*(i-1):n_ele*i-1, *)
    z_ion(i-1).thetab = $
      z_ion_block.thetab(n_ion*(i-1):n_ion*i-1, *)
    z_ion(i-1).phib = $
      z_ion_block.phib(n_ion*(i-1):n_ion*i-1, *)

    ; particle pitch-angles and energies in fixed payload coordinates

    z_ele(i-1).alpha = $
      z_ele_block.alpha(n_ele*(i-1):n_ele*i-1, *)
    z_ele(i-1).energy = $
      z_ele_block.energy(n_ele*(i-1):n_ele*i-1, *)
    z_ele(i-1).energy_old = $
      z_ele_block.energy_old(n_ele*(i-1):n_ele*i-1, *)
    z_ion(i-1).alpha = $
      z_ion_block.alpha(n_ion*(i-1):n_ion*i-1, *)
    z_ion(i-1).energy = $
      z_ion_block.energy(n_ion*(i-1):n_ion*i-1, *)
    z_ion(i-1).energy_old = $
      z_ion_block.energy_old(n_ion*(i-1):n_ion*i-1, *)

    ; zdata and their standard deviations

    z_ele(i-1).z = $
      z_ele_block.z(n_ele*(i-1):n_ele*i-1, *)
    z_ele(i-1).sig = $
      z_ele_block.sig(n_ele*(i-1):n_ele*i-1, *)
    z_ion(i-1).z = $
      z_ion_block.z(n_ion*(i-1):n_ion*i-1, *)
    z_ion(i-1).sig = $
      z_ion_block.sig(n_ion*(i-1):n_ion*i-1, *)

    ; data mask

    z_ele(i-1).dm = $
      z_ele_block.dm(n_ele*(i-1):n_ele*i-1, *)
    z_ele(i-1).idx = $
      z_ele_block.idx(n_ele*(i-1):n_ele*i-1)
    z_ele(i-1).n = n_ele
    z_ion(i-1).dm = $
      z_ion_block.dm(n_ion*(i-1):n_ion*i-1, *)
    z_ion(i-1).idx = $
      z_ion_block.idx(n_ion*(i-1):n_ion*i-1)
    z_ion(i-1).n = n_ion

    ; time ordered data indices

    z_ele(i-1).idx = z_ele_block.idx(n_ele*(i-1):n_ele*i-1)
    z_ion(i-1).idx = z_ion_block.idx(n_ion*(i-1):n_ion*i-1)

ENDFOR

END
