;==============================================================================
;
;  NAME:
;
;     ddeis_epa_avg
;
;  PURPOSE:
;
;     This function averages the Hydra-DDEIS data in 
;     energy-pitch-angle bins.
;
;  ARGUMENTS:
;
;     input (STRUCTURE):
;
;        structure containing one segment of processed Hydra-DDEIS
;        data.  This structure is an element of a structure array
;        output by the routine 'ddeis_block_process.'
;
;     control (STRUCTURE):
;
;        a structure which controls the behavior of 'ddeis_epa_avg.'
;        A description of the fields of this structure follows:
;
;           enmin  (FLOAT):
;
;              minimum energy to use in the binning (eV)
;
;           enmax  (FLOAT):
;
;              maximum energy to use in the binning (eV)
;
;           de  (FLOAT)
;
;              fractional energy bin size (delta_e/e)
;
;           da (FLOAT)
;
;              pitch-angle bin size (degrees)
;
;           sigma (INTEGER)
;
;              1:  compute zdata  standard deviations,
;                  'sigz_meas', and those expected from propagation of known
;                  systematic errors, 'sigz_prop.'
;
;              0:  no standard deviation calculations
;
;     output (STRUCTURE)
;
;        an output structure containing average properties of each
;        energy-pitch-angle bin.  A description of the fields of this
;        structure follows:
;
;           nraw (INTEGER)
;
;              number of valid measurements
;
;           traw: dblarr(nraw)
;
;              times of valid measurements (in seconds)
;      
;           eraw: fltarr(nraw)
;
;              valid energy measurements (eV)
;
;           araw: fltarr(nraw)
;
;              valid pitch-angle measurements (eV)
;
;           zraw: dblarr(nraw)
;
;              valid z measurements (either counts or distribution
;              function in CGS units)
;
;           sigraw: dblarr(nraw)
;
;              valid standard deviations (propagated from known
;              systematic errors) 
;         
;           bine: intarr(nraw)
;
;              energy bin indices 
;
;           bina: intarr(nraw)
;
;              pitch-angle bin indicies
;  
;           nbe (INTEGER)
;
;              number of energy bins
;           
;           nba (INTEGER)
; 
;              number of pitch-angle bins
;
;           ebar:  fltarr(nbe)
;
;              centers of energy bins (eV)
;
;           abar:  fltarr(nba)
;
;              centers of pitch-angle bins (radians)
;
;           tavg:  dblarr(nba, nbe)
;
;              times of centers of bins (seconds)
;
;           zavg:  dblarr(nba, nbe)
;
;              average of zdata in each bin (CGS units for
;              distribution function data)
;
;              NOTE:  bins for which zavg was not computed have 
;                     zavg = -1d20
;
;           sigz_meas:  dblarr(nba, nbe)
;
;              measured standard deviation in each bin
;
;              NOTE:  bins for which sigz_meas was not computed have
;                     sigz_meas = -1d20
;
;           sigz_prop:  dblarr(nba, nbe)
;
;              standard deviation in each bin expected from
;              propagation of known systematic errors
;
;
;              NOTE:  bins for which sigz_prop was not computed have
;                     sigz_prop = -1d20

;           nbz:  intarr(nba, nbe)
;
;              number of data points in each bin
;
;           err:  intarr(10)
;
;              an array of error flags.  A value of 0 indicates normal
;              exit.  Values of 1 indicate the following for each
;              element:
;
;                 err(0):  not enough measurements in any bin to 
;                          compute average zdata
;                 err(1):  not enough measurements in any bin to
;                          compute measured standard deviations
;                 err(2):  not enough measurements in any bin to
;                          compute propagated standard deviations
;
;  USAGE:
;
;     IDL> ddeis_epa_avg, input, control, output
;
;  HISTORY:
;
;     VERSION 1.00 -- 12/19/97 -- Written, John C. Dorelli
;     VERSION 1.01 -- 1/16/98 -- corrected bug in storage of raw
;                                   energies
;                                corrected bugs in calculation of
;                                   energy and pitch-angle bins
;                       
;
;==============================================================================

PRO ddeis_epa_avg, input, control, output

; store raw data

yraw = reform(input.energy, 12*input.n)
xraw = reform(input.alpha, 12*input.n)
zraw = reform(input.z, 12*input.n)
sigraw = reform(input.sig, 12*input.n)
traw = reform(input.time, 12*input.n)

data_ok = where(input.dm EQ 1 AND yraw GT control.enmin)

nraw = n_elements(data_ok)
xraw = xraw(data_ok)
yraw = yraw(data_ok)
zraw = zraw(data_ok)
sigraw = sigraw(data_ok)
traw = traw(data_ok)

; compute bin sizes and numbers of bins

da = control.da*!pi/180.
dlogE = alog10((2.+control.de)/(2.-control.de))*1d0

nbe = long((alog10(control.enmax)-alog10(control.enmin))/dlogE)+1
nba = long(!pi/da)+1
IF da EQ !pi THEN nba = 1

; define output structure

output = {$, 
          nraw: nraw, $
          traw: traw, $
          eraw: yraw, $
          araw: xraw, $
          zraw: zraw, $
          sigraw: sigraw, $
          bine: intarr(nraw), $
          bina: intarr(nraw), $
          nbe: nbe, $
          nba: nba, $
          ebar: fltarr(nbe), $
          abar: fltarr(nba), $
          tavg: dblarr(nba, nbe), $
          zavg: dblarr(nba, nbe), $
          nbz: dblarr(nba, nbe), $
          sigz_meas: dblarr(nba, nbe), $
          sigz_prop: dblarr(nba, nbe), $
          err: replicate(0, 10)}

output.eraw = yraw
output.araw = xraw
output.zraw = zraw
output.nraw = n_elements(zraw)

; compute centers of bins

eb0 = control.enmin
epsleft = alog10(eb0)+findgen(nbe)*dlogE
epsbar = epsleft+0.5*dlogE
output.ebar = 10^epsbar

ab0 = 0.
aleft = findgen(nba)*da
output.abar = aleft+0.5*da

; compute bin indices

blogE = long((alog10(yraw)-alog10(eb0))/dlogE)+1
ba = long((xraw-ab0)/da)+1
output.bine = blogE
output.bina = ba

; compute bin averages of zdata

FOR i = 1, nraw DO BEGIN
    k = ba(i-1)
    l = blogE(i-1)
    output.nbz(k-1, l-1) = output.nbz(k-1, l-1)+1
    output.zavg(k-1, l-1) = double(output.zavg(k-1, l-1) + zraw(i-1))
    output.tavg(k-1, l-1) = double(output.tavg(k-1, l-1) +traw(i-1))
ENDFOR
zavgok = where(output.nbz GT 0)
zavgnok = where(output.nbz EQ 0)
IF zavgok(0) EQ -1 THEN BEGIN
    print, 'DDEIS_EPA_AVG:  not enough measurements in each bin to'
    print, 'compute averages....'
    output.err(0) = 1
    return
ENDIF ELSE BEGIN
    output.zavg(zavgok) = output.zavg(zavgok)/output.nbz(zavgok)
    output.zavg(zavgnok) = -1d20
    output.tavg(zavgok) = output.tavg(zavgok)/output.nbz(zavgok)
    output.tavg(zavgnok) = -1d20
ENDELSE

; compute standard deviations for the zdata

IF control.sigma THEN BEGIN

    ; measured standard deviation

    res = dblarr(nba, nbe)
    FOR i = 1, nraw DO BEGIN
        k = ba(i-1)
        l = blogE(i-1)
        res(k-1, l-1) = double((zraw(i-1)-output.zavg(k-1, l-1))^2+$
                               res(k-1, l-1))
    ENDFOR
    sigok = where(output.nbz GT 1)
    signok = where(output.nbz LE 1)
    IF sigok(0) EQ -1 THEN BEGIN
        print, 'DDEIS_EPA_AVG:  not enough measurements in each bin to'
        print, 'compute measured standard deviations....'
        output.err(1) = 1
        return
    ENDIF ELSE BEGIN
        output.sigz_meas(sigok) = sqrt(res(sigok)/(output.nbz(sigok)-1.))/$
          sqrt(output.nbz(sigok))
        output.sigz_meas(signok) = -1d20
    ENDELSE

    ; standard deviation propagated from known systematic errors

    res = dblarr(nba, nbe)
    FOR i = 1, nraw DO BEGIN
        k = ba(i-1)
        l = blogE(i-1)
        res(k-1, l-1) = res(k-1, l-1) + sigraw(i-1)^2
    ENDFOR
    sigok = where(output.nbz GT 0)
    signok = where(output.nbz EQ 0)
    IF sigok(0) EQ -1 THEN BEGIN
        print, 'DDEIS_EPA_AVG:  not enough measurements in each bin'
        print, 'to compute propagated standard deviations....'
        output.err(2) = 1
        return
    ENDIF ELSE BEGIN
        output.sigz_prop(sigok) = sqrt(double(res(sigok)/output.nbz(sigok)))
        output.sigz_prop(signok) = -1d20
    ENDELSE
ENDIF

END

    

