;$Id: fitf_read.pro,v 1.1.1.1 2002/04/10 15:24:55 jbf Exp $
function fitf_read, date, time, species, nocorrect=nocorrect, parent=parent, $
                    nopotential=nopotential, de, da, index, nseg, nindex, $
                    energy, f, sigmaf, nbenergy, readtime, $
                    error = error, _extra = e
;+
; NAME:
;       fitf_read
;
; PURPOSE:
;
;       This function calculates a cut of Hydra-DDEIS data as a
;       function of energy in a specified pitch-angle range.
;
; CATEGORY:
;       
;       ddeis level 1 library
;
; CALLING SEQUENCE:
;       
;       result = fitf_read(state.args.path,state.args.t0, state.select, $
;                   state.mode(state.select).den, $
;                   state.mode(state.select).dalph,$
;                   state.mode(state.select).asel, $
;                   state.mode(state.select).hres, $
;                   state.mode(state.select).seg_idx, $
;                   energy,f,sigmaf, nbenergy, datatime, $
;                   next = state.mode(state.select).next, $
;                   prev = state.mode(state.select).prev, $
;                   nocorrect = (state.mode(state.select).correct EQ 0), $
;                   nopotential = (state.mode(state.select).potential EQ 0), $
;                   parent = state.base_id, error = error)
; INPUTS:
;
;     date (INTEGER): year, month and date in the format yyyymmdd
;                     (e.g., 19960529 is May 29, 1996)
;        
;     time (STRING):  hours, minutes and seconds in the format 'hh:mm:ss' 
;                     (e.g., '04:30:00')
;
;     species (INTEGER): 0 for electrons 1 for ions
;
;     de (FLOAT): energy bin size (delta_E/E)
;  
;     da (FLOAT): pitch-angle bin size (degrees)
;
;     index (INTEGER): bin index to be used in the cut
;
;     nseg (integer): number of segments to split up one 13.8 second
;                     block into
;
;     nindex (integer): the index of the block desired (see nseg above)
;     
;
; KEYWORD PARAMETERS:
;
;     nocorrect: set this keyword to disable count correction
;
;     nopotential: set this keyword to disable s/c potential
;                   subtraction
;    
;     parent: the base address of a parent widget if it exhists.
;    
;     error: return the error array from ddeis_block_process
;
;     next: get the next block of data
;
;     previous: get the previous block of data
;
; OUTPUTS:
;
;     result (integer): 0 for failure, 1 for success.
;
;     energy (FLOAT): 1-D array of energies (eV)
;
;     f (DOUBLE): 1-D array of distribution function data (CGS units)
;
;     sigmaf (double): 1-D array of standard deviations of the f
;                      points
;
;     nbenergy (integer): The number of samples averaged together at
;                         each energy.
;
;     readtime (strarr(2)): The min and max time of the data read in.
;
;
; SIDE EFFECTS:
;       
;       A new level1 file is open if necessary.
;
; PROCEDURE:
;       
;       Read one block of data, perform appropriate averaging, return
;       cut of the data for the specified parameter range.
;
; EXAMPLE:
;
;       See fitf_plot.pro for a good example of using this function.
;
; Written by:   Eric E. Dors, 1 March 1998.
;
; MODIFICATION HISTORY:
;
;       Mon Mar 16 11:44:47 1998, Eric Dors <eed@cis>
;
;		Problem in reform statement for sigmaf, if
;		out.sigz_prop(index-1, idx) has onlye one element, it
;		is recast to a scalar double instead of a dblarr(1).
;		To fix this put the argument to reform in "[]".
;
;-

; prepare input to 'ddeis_block_process'
IF (NOT keyword_set(nocorrect)) THEN BEGIN
    correct = 1 
ENDIF ELSE BEGIN
    correct = 0
    print, 'Count correction disabled!!'
    IF (n_elements(parent) EQ 1) THEN BEGIN
        ret = widget_message('Count correction disabled!!', $
                             dialog_parent = parent)
    ENDIF ELSE BEGIN
        ret = widget_message('Count correction disabled!!')
    ENDELSE
ENDELSE

IF (NOT keyword_set(nopotential)) THEN BEGIN
    potential = 1 
ENDIF ELSE BEGIN
    potential = 0
    print, 'S/C potential subtraction disabled!!'
    IF (n_elements(parent) EQ 1) THEN BEGIN
        ret = widget_message('S/C potential subtraction disabled!!', $
                             dialog_parent = parent)
    ENDIF ELSE BEGIN
        ret = widget_message('S/C potential subtraction disabled!!')
    ENDELSE
ENDELSE

control1 = {$
             correct: correct, $
             potential: potential, $
             filter: [0., 10000., 0.1, 20000, 0., !pi], $
             ztype: 0, $
             nseg: nseg}

; prepare input to 'ddeis_epa_avg'

control2 = {$
             enmin: 0.1, $
             enmax: 20000., $
             de: de, $
             da: da, $
             sigma: 1}

; process one block of data

ddeis_block_get, date, time, block_data, mode_info, file_flag, _extra = e

IF NOT file_flag THEN BEGIN
    print, 'FITF_READ:  level1 file not found....'
    return, 0
ENDIF
ddeis_block_process, block_data, mode_info, control1, z_ele, z_ion, error

; perform energy-pitch-angle bin average
IF (species eq 0) THEN BEGIN
    z = z_ele(nindex)
ENDIF ELSE BEGIN
    z = z_ion(nindex)
ENDELSE

;calculate time range of data read
hours   = min(z.time)/3600.
minutes = (hours - floor(hours))*60.
seconds = (minutes - floor(minutes))*60.
milisec = (seconds - floor(seconds))*1000.

timemin = string(format='(i2.2,a1,i2.2,a1,i2.2,a,i3.3)',$
                 hours,':',minutes,':',seconds,'.',milisec) 

hours   = max(z.time)/3600.
minutes = (hours - floor(hours))*60.
seconds = (minutes - floor(minutes))*60.
milisec = (seconds - floor(seconds))*1000.

timemax = string(format='(i2.2,a1,i2.2,a1,i2.2,a,i3.3)',$
                 hours,':',minutes,':',seconds,'.',milisec) 

readtime = [timemin, timemax]

ddeis_epa_avg, z, control2, out

; create 1-D cut arrays

energy = out.ebar
nba = n_elements(out.abar)
IF index LT 1 OR index GT nba THEN BEGIN
    print, 'FITF_READ:  pitch-angle bin index out of range;'
    print, 'setting ''index'' equal to one (first pitch-angle bin).'
    index = 1
ENDIF
f = reform(out.zavg(index-1, *))
nbenergy = reform(out.nbz(index-1, *))

idx = where(f NE -1d20, nidx)
IF (nidx EQ 0) THEN return, 0
energy = energy(idx)
f = f(idx)
nbenergy = nbenergy(idx)

sigmaf = reform([out.sigz_prop(index-1, idx)])
;sigma = out.sigz_meas(index-1, idx)
idx = where(sigmaf EQ -1d20, n_idx)
IF (n_idx NE 0) THEN BEGIN
    ;assume this will never happen, but flag in case it does'
    print, 'unavailable standard deviation, need to fix this'
    stop
ENDIF

return, 1
END
