pro hydra_plot_clr, zmat_in, xarr_in, yarr_in, dx=dx, dy=dy, $
                    zlog=zlog, zrange=zrange, $
                    nodata=nodata, _extra=e, special_colors=special, $
                    sample=sample, xfilled=xfilled

; INPUTS:
; zmat is the data, dblarr( n_elements(xarr), n_elements(yarr) )
; xarr are the x values for each z column (assumed to be X = dx * i + X(0), i an integer)
; yarr are the y values for each z row (same assumption, but can be logarythmic)
; /ylog is handled, but /xlog will give incorrect results.

; nodata specifies that a point should be plotted in grey

; special_colors = fltarr(2,n) can be used to insert special colors,
;                              e.g. saturation.  It is an array of pairs,
;                              each pair is (value,color).

; xfilled  keyword lets the routine know that the zmatrix has been
;     filled and it does not need to fill data.


common papco_color
common papco_color_names

common hydra_plot_clr_common, xsized_screen

; copy inputs to local variables
zmat= zmat_in
xarr= xarr_in
yarr= yarr_in

if not keyword_set(nodata) then nodata=5.43e21

if not keyword_set(zrange) then begin
   zmin=min(zmat,max=zmax)
   zrange=[zmin,zmax]
endif else begin
   zmin= double(zrange(0))
   zmax= double(zrange(1))
endelse

zlog= keyword_set(zlog)

sample= keyword_set(sample)     ; sample data, don't average

; find colorbar range
cindex= where( papco_color(*,2) eq 1 ) ; search the active vector
if (cindex(0) eq -1) then begin
    print, 'Color system failure, consult jbf@space-theory.physics.uiowa.edu'
    print, 'Using color set 0'
    cindex=0
endif
color_range= PAPCO_get_Colorindices( cindex )
colorbase= color_range(0)
ncolors= color_range(1)-color_range(0)
colortop= color_range(1)

if (keyword_set(e)) then begin
    plot_param= tag_names( e )
    r= where( plot_param eq 'YLOG' )
    if r(0) ne -1 then ylog=e.ylog else ylog=0
endif else begin
    ylog=0
endelse

if not keyword_set(dx) then begin
    dxs= xarr(1:*)-xarr(0:*)
    r=where( dxs gt 0.0 )
    if r(0) eq -1 then return
    dxs= dxs(r)
    dx= ( dxs( sort( dxs ) ) )(0)
endif

if not keyword_set(dy) then begin
   if (ylog) then begin
       dys= alog10(yarr(1:*)/yarr(0:*))
       dy= ( dys( sort( dys ) ) )(0) ;min
   endif else begin
       dys= yarr(1:*)-yarr(0:*)
       dy=( dys( sort( dys ) ) )(0)
   endelse
endif

maxx= xarr_in(n_elements(xarr_in)-1)+dx
if (ylog) then maxy= yarr_in(n_elements(yarr_in)-1)*10^dy else $
  maxy= yarr_in(n_elements(yarr_in)-1)+dy

plot, [xarr_in(0), maxx, maxx, xarr_in(0)], [yarr_in(0),yarr_in(0),maxy,maxy],$
  /noerase, /nodata, _extra=e

xcrange= !x.crange
ycrange= !y.crange

; find which xbins appear on the plot
rplotx= where( xarr ge xcrange(0) and xarr+dx le xcrange(1), nplotx )
if nplotx lt n_elements( xarr ) then begin
    if nplotx gt 0 then begin
        xarr=xarr(rplotx)
        zmat=zmat(rplotx,*)
    endif else begin
        xarr=xarr(0)   ; we'll see that nothing is printed later
        zmat=zmat(0,*)
    endelse
endif

; calculate the number of x and y bins for image
if not keyword_set( xfilled ) then begin
    nx= long( ( xarr(n_elements(xarr)-1) - xarr(0) ) / dx + 0.5 ) + 1
    wherex= long((xarr-xarr(0))/dx+0.5)
endif else begin
    nx= n_elements( xarr )
    wherex= indgen( nx )
endelse

;if (ylog) then begin
;    ny= fix( ( yarr(n_elements(yarr)-1) / yarr(0) ) / dy + 0.5 ) + 1
;    wherey = fix(alog10(yarr/yarr(0))/dy+0.5)
;endif else begin
;    ny= fix( ( yarr(n_elements(yarr)-1) - yarr(0) ) / dy + 0.5 ) + 1
;    wherey = fix((yarr-yarr(0))/dy+0.5)
;endelse
ny= n_elements(yarr)
wherey= indgen(ny)

zimage= make_array( nx, ny, /float, value=nodata )

for j=0,n_elements(wherey)-1 do begin
   zimage(wherex,wherey(j))= zmat(*,j)
endfor

xmax= xarr( n_elements(xarr)-1 ) + dx
if (ylog) then ymax= yarr(n_elements(yarr)-1)*dy else ymax= yarr( n_elements(yarr)-1 ) + dy

if (ylog) then begin
    lld= convert_coord( xarr(0), 10^ycrange(0), /data, /to_device )
    urd= convert_coord( xmax, 10^ycrange(1), /data, /to_device )
endif else begin
    lld= convert_coord( xarr(0), yarr(0), /data, /to_device )
    urd= convert_coord( xmax, ymax, /data, /to_device )
endelse

common plot_Composer, widgetData ; from PaPCo, always use screen res.
isPaPCo= n_elements( widgetData ) gt 0

if isPaPCo then begin    
    if !d.name eq 'PS' then begin
        if n_elements(nbins_last) eq 0 then $
          nbins_last= long( widgetData.default_draw_size(0) * 0.675 + 0.5 )
        nbins= nbins_last
    endif else begin
        nbins_last= long( widgetData.default_draw_size(0) * 0.675 + 0.5 )
        nbins= long(urd(0)-lld(0))
    endelse
    xsized= long(urd(0)-lld(0))
endif else begin
    nbins= long(urd(0)-lld(0))
    xsized= long(urd(0)-lld(0))
endelse
ysized= long(urd(1)-lld(1))

if !d.name eq 'PS' then begin
    if !d.x_px_cm eq 1000 then begin
        print, '% hydra_plot_clr: reducing PS resolution to 300 dpi'
        if not isPaPCo then xsized = xsized * 300 / 2540
        ysized = ysized * 300 / 2540 
    endif
endif

print, nbins, ysized, long(urd(0)-lld(0)), '<<<<'

; average down the array if there aren't enough pixels to show each 
; measurement.
if (nbins lt nx) then begin
    if (sample) then begin
        print, '% hydra_plot_clr: sampling data'
        rrr= long( findgen( nbins ) / nbins * nx )
        zimage= zimage( rrr, * )
    endif else begin
        print, '% hydra_plot_clr: time-averaging '+strtrim(nx,2)+$
          ' time intervals down to '+strtrim(xsized,2)+'.'    
        zimage= hydra_average( zimage, nbins, /allow, nodata=nodata )
    endelse
    nx= nbins
endif

; transform data to color
if (zlog) then begin
    zsize= alog10( zmax/zmin )
    zcolor= make_array( nx, ny, /byte, value=colorbase )
    rok= where( zimage ge zmin )
    if rok(0) ne -1 then zcolor(rok)= alog10( zimage(rok)/zmin ) / $
      ( zsize / ncolors ) + colorbase < colortop
endif else begin
    zsize= zmax-zmin
    zcolor= make_array( nx, ny, /byte, value=colorbase )
    rcol= where( zimage ge zmin )
    zcolor(rcol)= ( zimage(rcol) - zmin ) / $
      ( zsize / ncolors ) + colorbase < colortop
    r= check_math()
    if r ne 0 then message, 'check_math()='+strtrim(r,2), /cont
endelse

; fill color nodata points grey
rnd=where( zimage eq nodata )
if rnd(0) ne -1 then zcolor( rnd ) = grey

; fill special color points with special colors
if n_elements(special) gt 0 then begin
    for i=0,n_elements(special(0,*))-1 do begin
        value= special(0,i)
        color= byte(special(1,i))
        rval= where( zimage eq value )
        if rval(0) ne -1 then zcolor(rval)=color
    endfor
endif

if (nplotx gt 0) then begin
    if !d.name eq 'PS' then begin
        lln= convert_coord( lld, /device, /to_normal )
        urn= convert_coord( urd, /device, /to_normal )
        xsizen= urn(0)-lln(0)
        ysizen= urn(1)-lln(1)
        tv, zcolor, lln(0), lln(1), xsize=xsizen, ysize=ysizen, /normal
    endif else begin
        tv, congrid(zcolor, xsized, ysized), lld(0), lld(1)+1 ; +1 is a kludge
    endelse
endif

plot, [xarr_in(0), maxx, maxx, xarr_in(0)], [yarr_in(0),yarr_in(0),maxy,maxy],$
  /noerase, /nodata, _extra=e

return

end
