;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_bfield, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hydra_bfield for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_bfield, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_bfield, panel, type, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           _EXTRA=extra_par, subtable=subtable



; The following common blocks are needed:

   common time, xut1,xut2 ;common time limits in T90
   common mjdt, mjdt1, mjdt2

   common coordinateSystems, plotted_x, plotted_y
                                        ;info on coords used by mouse

   common hydra_bfield_data, $
     bfield_type, $
     bfield_header, $
     bfield_lr_data, $
     bfield_mr_data, $
     bfield_hr_data, $
     bfield_ppar_data

   common papco_color_names

   panelset,panel			;sets the panel position viewport

   oldy=!y
   oldx=!x

   goodticks, xtickv, xminor
   xticks=n_elements(xtickv)-1

; set duplex font if big_gif is device ; duplex roman
   if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')

; the plot calls normally use the arrays t90 for the time axis and yray (ymat
; for color plots) for the y or z axis.
   dxut= xut2-xut1

   IF (bfield_type EQ 0) THEN BEGIN
      data = bfield_lr_data
      resstr= 'LR'
   ENDIF ELSE IF (bfield_type EQ 1) THEN BEGIN
      data = bfield_mr_data
      resstr= 'MR'
   ENDIF ELSE if (bfield_type eq 2) then BEGIN
       data = bfield_hr_data
       resstr= 'HR'
   Endif else begin
       data = bfield_ppar_data
       resstr= 'PPA-R'
   endelse

   if getenv('hydra_presentation_mode') eq '1' then begin
       str_type= ''
   endif else begin
       str_type= 'Hyd '+resstr+' Bfield!c'
   endelse

   options= type(2)

   units= bfield_header(bfield_type).units

   plot_name=' '
   case type(1) of
       0:begin
           ctit2=str_type+'!9!!!xB!9!!!x' & cytit=units
           t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
           ydat= data(*).bmag
           ylog=0
       end
       1:begin
           ctit2=str_type+'B!dX-gsm' & cytit=units
           t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
           ydat= data(*).bx
           ylog=0
       end
     2:begin
         ctit2=str_type+'B!dY-gsm' & cytit=units
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         ydat= data(*).by
         ylog=0
     END
     3:begin
         ctit2=str_type+'B!dZ-gsm' & cytit=units
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         ydat= data(*).bz
         ylog=0
         hydra_plot_kw= { reference:0., $
                          poly_above_ref:red, $
                          poly_below_ref:blue }
     END
     4:begin
         ctit2=str_type+'B!dX-pay' & cytit=units
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         ydat= data(*).bx_pay
         ylog=0
     end
     5:begin
         ctit2=str_type+'B!dY-pay' & cytit=units
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         ydat= data(*).by_pay
         ylog=0
     END
     6:begin
         ctit2=str_type+'B!dZ-pay' & cytit=units
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         ydat= data(*).bz_pay
         ylog=0
     END
     7:begin
         ctit2=str_type+'B!dtheta payload' & cytit='deg.'
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         bmag= data(*).bmag
         ydat= make_array( n_elements( bmag ), /float, value=1e31 )
         rv= where( bmag lt 1e30 )
         if rv(0) ne -1 then ydat(rv)= acos( data(rv).bz_pay/bmag(rv) ) / !dtor
         ylog=0
     end
     8:begin
         ctit2=str_type+'B!dphi payload' & cytit='deg.'
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         bmag= data(*).bmag
         ydat= make_array( n_elements( bmag ), /float, value=1e31 )
         rv= where( bmag lt 1e30 )
         if rv(0) ne -1 then begin
             btheta= acos( data(rv).bz_pay/bmag(rv) )
             bhatx = data(rv).bx_pay/bmag(rv)
             ydat(rv)= acos( ( bhatx/sin(btheta)) < 1.0 > (-1.0) ) / !dtor
             rneg= where( data(rv).by_pay lt 0. )
             if rneg(0) ne -1 then ydat(rv(rneg))= 360 - ydat(rv(rneg)) 
         endif
         ylog=0
     end
     9:begin
         ctit2=str_type+'Mag. Pressure' & cytit='cgs'
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         ydat= data(*).mag_pressure
         ylog=0
     END
     10:BEGIN
         case bfield_type of 
             0: n = 1
             1: n = 1
             2: n = 1
             3: n = 1
         ENDCASE
         ctit2=str_type+'!cAngle b/t adj. B' & cytit='deg'

         t90= ( data(*).time + data(n:*).time ) / 2 + $
           ( bfield_header(bfield_type).start_time - xut1 ) 

         bx1 = data(0:*).bx
         by1 = data(0:*).by
         bz1 = data(0:*).bz
         mag1 = sqrt(bx1^2 + by1^2 + bz1^2)

         bx2 = data(n:*).bx
         by2 = data(n:*).by
         bz2 = data(n:*).bz
         mag2 = sqrt(bx2^2 + by2^2 + bz2^2)

         gd = where( mag1 GT 0 AND mag2 GT 0 )
         t90 = t90(gd)
         ydat= acos( $
                    (bx2(gd)*bx1(gd)+by2(gd)*by1(gd)+bz2(gd)*bz1(gd)) / $
                     ( mag2(gd) * mag1(gd) ) ) / !dtor
         ylog=0
         
     END
     11:begin
         ctit2=str_type+'!cEle Cyclotron' & cytit='Hz'
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         ydat= data(*).bmag * 1e-5 * 4.8e-10 / $
           ( 2 * !pi * 9.1e-28 * 3e10 ) 
         rnv= where( data(*).bmag gt 1e30 )
         if rnv(0) ne -1 then ydat(rnv)=1e31
         ylog=0
     end
     12:begin
         ctit2=str_type+'!cIon Cyclotron' & cytit='Hz'
         t90= data(*).time + ( bfield_header(bfield_type).start_time - xut1 )
         ydat= data(*).bmag * 1e-5 * 4.8e-10 / $
           ( 2 * !pi * 9.1e-28 * 3e10 ) / 1836. 
         rnv= where( data(*).bmag gt 1e30 )
         if rnv(0) ne -1 then ydat(rnv)=1e31
         ylog=0
     end
         
  ENDCASE
  
  if ( options and long(8) ) gt 0 then begin ; filter neg descriminant
      r= where(( data(*).bflags and 2 ) eq 0 )
      if r(0) ne -1 then begin
          message, 'Filtering negative descriminant points.', /cont
          ydat= ydat(r)
          t90= t90(r)
      endif
  endif

  ctit=' '                      ; Do not use this variable!

  cytit= font+cytit
  ctit2= font+ctit2

; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images

  if n_elements(t90) le 1 then begin
      x= hydra_message( ['p_hydra_bfield: ', $
                         'No good data available'], /warn )
      plotted_x= !x
      plotted_y= !y
      return
  endif

; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF

  index=where((t90(1:*) ge 0) and (t90(1:*) le dxut),count)+1
  if count ge 2 then begin
      t90=t90(index)
      ydat=ydat(index)
  end

  if count gt 0 then begin
; insert breaks where there are data dropouts. (grid the data)
      dt= t90(1:*) - t90(0:*)
      r= where(dt gt 0)
      if r(0) ne -1 then dt=min(dt(r)) else dt=1
      nn= max(t90)/dt + 1
      ydat_grid= make_array( nn, /float, value=1e31 )
      t90_grid= make_array( nn, /float, value=0 )
      r= long( t90/dt )
      ydat_grid( r ) = ydat
      t90_grid( r ) = t90
      
      t90= t90_grid 
      ydat= ydat_grid
  endif

  y_kw= hydra_ytick_parm( ydat, fill=1e31, $
                          ylog=ylog, $
                          panel=panel(0), $
                          options= type(3) )
  
;  rt=where( t90 ge 0 and t90 le dxut ) 
;  if (rt(0) ne -1) then begin
;      if type(1) eq 11 or type(1) eq 12 then begin
;          hydra_ylim_1, panel(0), type, ydat(rt), yrange=yrange, ylog=ylog
;      endif else begin
;          hydra_ylim_new, panel(0), type, ydat(rt), yrange=yrange, ylog=ylog
;      endelse                   ;  This is horrible I hate myself.
;  endif else begin
;      yrange=[0,1]
;      ylog=0
;  endelse
  
; set up extra plot keywords, common ones first

  extra_plotPar_common={ xrange:[0,dxut], $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xtickv:xtickv, xminor:xminor, xticks:xticks, $
                         yticklen:0.010, ytitle:cytit, $
                         ythick:1.3, xthick:1.3, max_value:1e30 }

  extra_plotPar_common= create_struct( extra_plotPar_common, $
                                       y_kw )

; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

   if n_elements(extra_par) EQ 0 then $
     extra_par={color:1}

; add keyword structure set here with the one passed in

   extra_plotPar= create_struct( extra_plotPar_common, extra_par )

   axis_plotPar=create_struct( extra_plotPar_common, { color:1} )

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot

   if panel(0) eq 0 then $
     xtickformat='hydra_timeticks' $
   else $
     xtickformat='noticks'

; plot the axis in color=1, and only plot if if you're not overplotting
   if not keyword_set(overplot) then begin
       plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
         xstyle=1, ystyle=1, xtickformat=xtickformat       
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   endif

; now do the actual data plot
   if n_elements( hydra_plot_kw ) gt 0 then $
     hp_kw= create_struct( extra_plotPar, hydra_plot_kw ) $
   else $
     hp_kw= extra_plotPar
   hydra_plot, t90, ydat, _extra=hp_kw, xstyle=5, ystyle=5, $
     dx=dt, overplot=keyword_set(overplot)
   if type(1) eq 11 or type(1) eq 12 then begin ; plot harmonics
       r= where( ydat lt 1e28 )
       if r(0) eq -1 then n=1 else $
         n= fix( extra_plotPar.yrange(1) / min( ydat(r) ) ) < 15
       for i=2,n do begin
           hydra_plot, t90, ydat*i, _extra=extra_plotPar, xstyle=5, ystyle=5, $
             dx=dt, overplot=keyword_set(overplot)
       endfor
   endif

   if not keyword_set(overplot) then $
     oplot, !x.crange, [0,0], linestyle=1, color=dark_grey
   
   if not keyword_set(overplot) then begin
       plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
         xstyle=1, ystyle=1, xtickformat=xtickformat
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   endif

; store the coordinate information into common block
   plotted_x = !x
   plotted_y = !y

   !x=oldx
   !y=oldy

; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
  abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
  lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )

  if type(1) ne 11 and type(1) ne 12 then $
    right_side_label,panel,lfs+ctit2, _extra=extra_plotPar

  hydra_footprint, bfield_header(bfield_type)

end





