;***************************************************************************
;* 
;* PROCEDURE:     
;*      p_hydra_highres, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type hydra_highres for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale     
;*
;* INPUTS:       
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.   
;* 
;* CALLING SEQUENCE:
;*       p_hydra_highres, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_highres, panel, type, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           _EXTRA=extra_par, subtable=subtable

common hydhighresstuff, C_fln, hrkp, nhrkp, level1id, gsmcdf ;,bmag, bmagtime

; The following common blocks are needed:

   common time, xut1,xut2 ;common time limits in T90
   common mjdt, mjdt1, mjdt2

;   common hydra_yscale, yscl
   common yscale, yscl			;man/auto yscaling
   common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

   common hres_bdata, Cxut1, Cxut2, bx, by, bz, bmagtime

   common papco_color_names

   panelset,panel			;sets the panel position viewport

   oldy=!y
   oldx=!x

   goodticks, xtickv, xminor
   xticks=n_elements(xtickv)-1

; set duplex font if big_gif is device 
   if !d.name eq 'Z' then font='!5' else font='' ; duplex roman

; the plot calls normally use the arrays t90 for the time axis and yray (ymat 
; for color plots) for the y or z axis.
   dxut= xut2-xut1

   t90= hrkp(*).time - mjdt1.t     ; seconds since xut1

   load_hres_b=1
   if n_elements(Cxut1) gt 0 then $
     if Cxut1 eq xut1 and Cxut2 eq xut2 then load_hres_b=0
   
   plot_name=' '
   case type(0) of
     0:begin
         plot_name='B'
         ctit2='HR B magnitude' & cytit='nT'         
         print, '% p_hydra_highres: reading bfield from level1...'
         if (load_hres_b) then begin
             read_level1_highres_bfield, level1id, mjdt1.t, mjdt2.t, $
               bx, by, bz, bmagtime, gsm=gsmcdf
             Cxut1=xut1
             Cxut2=xut2
         endif
         bmag= sqrt( bx*bx+by*by+bz*bz )
         ydat= bmag + 0.01
         t90=bmagtime - mjdt1.t
         dt=0.125
         this_Par={ ylog:0 }    ;, min_value:0
     end
     1:begin
         ctit2='HR Electron !cDensity' & cytit='cm!E-3!n'
         ydat= hrkp(*).density + 0.01
         dt=1.15
         this_Par={ ylog:1 };, min_value:0
     end
     2:begin
         ctit2='HR T!ie!n !ctemperature' & cytit='deg. K'
         ydat= hrkp(*).averageE * 1.16e4 / ( 3./2 )
         r=where(ydat lt 0.0)
         if r(0) ne -1 then ydat(r)=1e31
         dt=1.15
         this_Par={ ylog:1 };, min_value:0
     end
     3:begin                    ; mean energy overplot
         plot_name='meane_overplot'
         ctit2=' ' & cytit=' '
         ydat= hrkp(*).averageE
         dt=1.15
         this_Par={ ylog:1 }
     end
     4:begin
         plot_name='Bz'
         ctit2='HR Bz (GSM)' & cytit='nT'         
         print, '% p_hydra_highres: reading bfield from level1...'
         if (load_hres_b) then begin
             read_level1_highres_bfield, level1id, mjdt1.t, mjdt2.t, $
               bx, by, bz, bmagtime, gsm=gsmcdf
             Cxut1=xut1
             Cxut2=xut2
         endif
         ydat= bz
         t90=bmagtime - mjdt1.t
         dt=0.125
         this_Par={ ylog:0 }    ;, min_value:0
     end
 endcase

; kludge to have overplot be red
;     if keyword_set(overplot) then begin
;         extra_Par.color=red
;     endif

; filter out invalid data (flagged with T90=0)
 r=where(t90 eq 0L)
 if r(0) ne -1 then ydat(r)=1e31

  ctit=' '                      ; Do not use this variable!

  cytit= font+cytit
  ctit2= font+ctit2

; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data. 
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images

; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF
  index=where((t90(1:*) ge 0) and (t90(1:*) le dxut),count)+1
  if count ge 2 then begin
      t90=t90(index)
      ydat=ydat(index)     
  end


  if n_elements(yscl) eq 0 then yscl=dblarr(10,4)

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin 
;         yscl(*,2)  is then ymax
;         yscl(*,3)  is ylog
; when scaling is set to automatic, the automatic y limits are loaded in.

  if (yscl(panel(0),0) eq 1) then begin
      print, '% p_hydra_highres: getting manual scale specifications...'
      yst= yscl(panel(0),1)
      yen= yscl(panel(0),2)
      this_Par.ylog= yscl(panel(0),3)
      IF (yst LE 0 AND this_Par.ylog EQ 1) THEN BEGIN
         print, '% p_hydra_highres: Negative ymin: swwitching from log to linear...'
         yscl(panel(0),3) =  0
         this_Par.ylog = yscl(panel(0),3)
      ENDIF

      print, 'yen, yst, ylog'
      print, '###', panel(0), yen, yst, this_Par.ylog
  endif else begin              ; use autoscale results
      rv=where(ydat lt 1e31)
      if rv(0) ne -1 then rt=where( t90(rv) ge 0 and t90(rv) le dxut ) $
      else rt=-1
      if (rt(0) ne -1) then begin
          hydra_ylim, ydat(rv(rt)), yst, yen, ylog, _extra=this_Par
          this_Par.ylog=ylog
          print, 'ylog='+strtrim(this_Par.ylog,2)
          if plot_name eq 'Bz' or plot_name eq 'B' then begin
              yst=-max([abs(yst),abs(yen)])
              yen=max([abs(yst),abs(yen)])
          endif
      endif else begin
          yst=0 & yen=1
      endelse     
      yscl(panel(0),1)= yst
      yscl(panel(0),2)= yen
      yscl(panel(0),3)= this_Par.ylog
  endelse

  print, '>>>', yst, yen

; set up extra plot keywords, common ones first

   extra_plotPar_common={yrange:[yst, yen], xrange:[0,dxut], $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $ 
                         xtickv:xtickv, xminor:xminor, xticks:xticks, $
                         yticklen:0.010, ytitle:cytit, $
                         ythick:1.3, xthick:1.3, max_value:1e30 }

; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

   if n_elements(extra_par) EQ 0 then $
     extra_par={color:1}

; color printing kludge
   if keyword_set(overplot) and plot_name ne 'meane_overplot' then begin
       extra_par={color:red}
   endif
; add keyword structure set here with the one passed in

   extra1_plotPar_common=create_struct(extra_plotPar_common, extra_par)

   extra1_plotPar_common=create_struct(extra1_plotPar_common, this_par)

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot 

   if panel(0) eq 0 then $
     extra_plotPar=create_struct({xtickformat:'hydra_timeticks'}, $
                                 extra1_plotPar_common) $
   else $
     extra_plotPar=create_struct({xtickformat:'noticks'}, $
                                 extra1_plotPar_common)

; calculate the average number of measurements per pixel
   hydra_get_device_space, x0, y0, xpixels, ypixels
   mpp= n_elements(ydat)/float(xpixels)
   

; plot the axis in color=1, and only plot if if you're not overplotting
   color1_plotpar=create_struct({color:1}, extra_plotPar)
   if not keyword_set(overplot) then begin
;     extend this structure by ctit if the panel is topmost
       if (panel(1) eq panel(0)+panel(2)) then $
         color1_plotpar=create_struct({title:ctit}, $
                                      color1_plotpar)
       plot,t90,ydat, _extra=color1_plotPar, /NODATA, $
         xstyle=3, ystyle=3
       hydra_events

   endif 

; now do the actual data plot
   if (mpp le 0.1) then begin
       hydra_xbin_plot, t90, t90+dt, ydat, $
         _extra=extra_plotPar, xstyle=7, ystyle=7
   endif else begin
       plot, t90, ydat, _extra=extra_plotPar, xstyle=7, ystyle=7;, $
;         nsum=(fix(mpp)>1)
   endelse

   if plot_name eq 'Bz' then begin
       oplot, [0,dxut], [0,0], linestyle=1
   endif

; store the coordinate information into common block
   plotted_x = !x
   plotted_y = !y

   !x=oldx
   !y=oldy

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015

  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
     extra_plotPar=create_struct({xtickformat:'hydra_juldate', $
                                  color:1}, $
                                  extra1_plotPar_common) 
     plot,t90,ydat, _extra=extra_plotPar, /NODATA, $
        xstyle=3, ystyle=3
     x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
  endif

; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
  abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
  lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
  right_side_label,panel,lfs+ctit2, _extra=extra_plotPar
  
end 





