;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_kp_extra, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hydra_kp_extra for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_kp_extra, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_kp_extra, panel, type, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           _EXTRA=extra_par, subtable=subtable

; The following common blocks are needed:

   common time, xut1,xut2 ;common time limits in T90
   common mjdt, mjdt1, mjdt2

   common coordinateSystems, plotted_x, plotted_y
                                        ;info on coords used by mouse

   common hydra_kp_extra_data, $
    kp_extra_header,  $
    kp_extra_data

   common papco_color_names

   panelset,panel			;sets the panel position viewport

   oldy=!y
   oldx=!x

   goodticks, xtickv, xminor
   xticks=n_elements(xtickv)-1
   
; set duplex font if big_gif is device ; duplex roman
   if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')
   
; the plot calls normally use the arrays t90 for the time axis and yray (ymat
; for color plots) for the y or z axis.
   dxut= xut2-xut1
   
   data= kp_extra_data
   rfill = -1.e+31
   errorbars = 0
   
; Define the data filter for the 'best' quantities 
; Note: 'rstudy' quantities have their own filter, applied below
   IF (n_elements(data(0).rstudy_den_stats) EQ 6) THEN BEGIN
      good_best = where( (data.fit_stat_ion EQ 0) OR $
                         (data.fit_stat_ion AND 4) OR $
                         (data.fit_stat_ion AND 8) OR $
                         (data.fit_stat_ion AND 16) )
      gfit_den = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                       AND data.rstudy_den_stats(2) NE rfill $
                       AND data.rstudy_den_stats(3) NE rfill)
      gfit_ux = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                      AND data.rstudy_u_stats(0,2) NE rfill $
                      AND data.rstudy_u_stats(0,3) NE rfill)
      gfit_uy = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                      AND data.rstudy_u_stats(1,2) NE rfill $
                      AND data.rstudy_u_stats(1,3) NE rfill)
      gfit_uz = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                      AND data.rstudy_u_stats(2,2) NE rfill $
                      AND data.rstudy_u_stats(2,3) NE rfill)
      gfit_tparl = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                         AND data.rstudy_tparl_stats(2) NE rfill $
                         AND data.rstudy_tparl_stats(3) NE rfill)
      gfit_tperp = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                         AND data.rstudy_tperp_stats(2) NE rfill $
                         AND data.rstudy_tperp_stats(3) NE rfill)
      gfit_gyrotropy =where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                            AND data.rstudy_gyrotropy_stats(2) NE rfill $
                            AND data.rstudy_gyrotropy_stats(3) NE rfill)
   ENDIF ELSE BEGIN 
      good_best = where( (data.fit_stat_ion EQ 0) OR $
                         (data.fit_stat_ion AND 4) )
      gfit_den = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                       AND data.rstudy_den_stats(6) EQ 0)
      gfit_ux = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                      AND data.rstudy_u_stats(0,6) EQ 0)
      gfit_uy = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                      AND data.rstudy_u_stats(1,6) EQ 0)
      gfit_uz = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                      AND data.rstudy_u_stats(2,6) EQ 0)
      gfit_tparl = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                         AND data.rstudy_tparl_stats(6) EQ 0)
      gfit_tperp = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                         AND data.rstudy_tperp_stats(6) EQ 0)
      gfit_gyrotropy =where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                            AND data.rstudy_gyrotropy_stats(6) EQ 0)
   ENDELSE

; Species
   hyd_str = 'TPP H!u+!n!c'
   hep_str = 'TPP He!u+!n!c'
   hepp_str = 'TPP He!u++!n!c'
   oxy_str = 'TPP O!u+!n!c'

   u_mag_str = '!10#!xU!10#!x'
   u_x_str = 'U!dx!n'
   u_y_str = 'U!dy!n'
   u_z_str = 'U!dz!n'
   u_perp_mag_str = '!10#!xU!u!9x!x!n!10#!x'
   u_perp_x_str = 'U!u!9x!x!n!dx!n'
   u_perp_y_str = 'U!u!9x!x!n!dy!n'
   u_perp_z_str = 'U!u!9x!x!n!dz!n'
   u_parl_str = 'U!u!9#!x!n'

   kt_str = 'kT'
   kt_parl_str = 'kT!d!9#!x!n'
   kt_perp_str = 'kT!d!9x!x!n'

   p_parl_str = 'P!d!9#!x!n'
   p_perp_str = 'P!d!9x!x!n'
   
   plot_name=' '
   case type(0) of
      0:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+'n' & cytit='cm!E-3!n'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+'n' & cytit='cm!E-3!n'
               factor = sqrt(4./1.)
            END
            2:BEGIN
               ctit2 = hepp_str+'n' & cytit='cm!E-3!n' 
               factor = sqrt(4./2.)
            END
            3:BEGIN
               ctit2 = oxy_str+'n' & cytit='cm!E-3!n'
               factor = sqrt(16./1.)
            END 
         ENDCASE 
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).density_ion*factor
         dt=13.8
         ylog=1 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_den).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_den).density_ion*factor - $
             data(gfit_den).rstudy_den_stats(2)*factor > 0
            erry2 = data(gfit_den).density_ion*factor + $
             data(gfit_den).rstudy_den_stats(3)*factor > 0
         ENDIF
      END 
      1:BEGIN 
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+'n (Rstudy)' & cytit='cm!E-3!n'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+'n (Rstudy)' & cytit='cm!E-3!n'
               factor = sqrt(4./1.)
            END
            2:BEGIN
               ctit2 = hepp_str+'n (Rstudy)' & cytit='cm!E-3!n' 
               factor = sqrt(4./2.)
            END
            3:BEGIN
               ctit2 = oxy_str+'n (Rstudy)' & cytit='cm!E-3!n'
               factor = sqrt(16./1.)
            END 
         ENDCASE
         t90= data(gfit_den).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(gfit_den).rstudy_den_stats(1)*factor
         dt=13.8
         ylog=1 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_den).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = ( data(gfit_den).rstudy_den_stats(1) - $
                      data(gfit_den).rstudy_den_stats(2) )*factor > 0
            erry2 = ( data(gfit_den).rstudy_den_stats(1) + $
                      data(gfit_den).rstudy_den_stats(3) )*factor > 0
         ENDIF
      END
      2:BEGIN 
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+'kT' & cytit='eV'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+'kT' & cytit='eV'
               factor = 1.
            END
            2:BEGIN
               ctit2 = hepp_str+'kT' & cytit='eV'
               factor = 2.
            END
            3:BEGIN
               ctit2 = oxy_str+'kT' & cytit='eV'
               factor = 1.
            END 
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).mean_energy_ion*factor
         dt=13.8
         ylog=1 
      end
      3:begin
         ctit2='TPP !cvar_bhat_diag' & cytit=' '
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).var_bhat_diag
         dt=13.8
         ylog=1 
      END
      4:begin
         ctit2='TPP !clshell' & cytit='R!dE!n'
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).lshell
         dt=13.8
         ylog=1 
      END
      5:begin
         ctit2='TPP !cradius' & cytit='R!dE!n'
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).radius
         dt=13.8
         ylog=1 
      END
      6:begin
         ctit2= hyd_str+'CC!dmax!n' & cytit=' '
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).max_cc(0)
         dt=13.8
         ylog=1 
      END
      7:begin
         ctit2= hyd_str+'Energy of CC!dmax!n' & cytit='eV'
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).max_cc(1)
         dt=13.8
         ylog=1 
      END
      8:begin
         ctit2= hyd_str+'E!dmax!n Patched' & cytit='eV'
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).max_energy
         dt=13.8
         ylog=1 
      END
      9:begin
         ctit2= hyd_str+'E!dmin!n Patched' & cytit='eV'
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).min_energy
         dt=13.8
         ylog=1 
      END
      10:begin
         ctit2= hyd_str+'Angle b/t Parl e.v. and B' & cytit='rad'
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).angle_b
         dt=13.8
         ylog=1 
      END
      11:begin
         ctit2= hyd_str+'V_alf' & cytit='km/s'
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).v_alf
         dt=13.8
         ylog=0 
      END
      12:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_mag_str & cytit='km/s'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_mag_str & cytit='km/s'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_mag_str & cytit='km/s'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_mag_str & cytit='km/s'
               factor = sqrt(1./16.)
            END 
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_pay(3)*factor
         dt=13.8
         ylog=1 
      END
      13:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_parl_str & cytit='km/s'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_parl_str & cytit='km/s'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_parl_str & cytit='km/s'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_parl_str & cytit='km/s'
               factor = sqrt(1./16.)
            END 
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_parl*factor
         dt=13.8
         ylog=0 
      END
      14:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_perp_x_str & cytit='km/s [PAY]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_perp_x_str & cytit='km/s [PAY]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_perp_x_str & cytit='km/s [PAY]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_perp_x_str & cytit='km/s [PAY]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_perp(0)*factor
         dt=13.8
         ylog=0 
      END
      15:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_perp_y_str & cytit='km/s [PAY]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_perp_y_str & cytit='km/s [PAY]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_perp_y_str & cytit='km/s [PAY]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_perp_y_str & cytit='km/s [PAY]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_perp(1)*factor
         dt=13.8
         ylog=0 
      END
      16:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_perp_z_str & cytit='km/s [PAY]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_perp_z_str & cytit='km/s [PAY]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_perp_z_str & cytit='km/s [PAY]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_perp_z_str & cytit='km/s [PAY]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_perp(2)*factor
         dt=13.8
         ylog=0 
      END
      17:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_perp_mag_str & cytit='km/s'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_perp_mag_str & cytit='km/s'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_perp_mag_str & cytit='km/s'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_perp_mag_str & cytit='km/s'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_perp(3)*factor
         dt=13.8
         ylog=1 
      END
      18:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_x_str & cytit='km/s [PAY]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_x_str & cytit='km/s [PAY]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_x_str & cytit='km/s [PAY]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_x_str & cytit='km/s [PAY]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_pay(0)*factor
         dt=13.8
         ylog=0 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_ux).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_ux).u_pay(0)*factor - $
             data(gfit_ux).rstudy_u_stats(0,2)*factor
            erry2 = data(gfit_ux).u_pay(0)*factor + $
             data(gfit_ux).rstudy_u_stats(0,3)*factor
         ENDIF
      END
      19:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_x_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_x_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_x_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_x_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(gfit_ux).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(gfit_ux).rstudy_u_stats(0,1)*factor
         dt=13.8
         ylog=0 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_ux).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_ux).rstudy_u_stats(0,1)*factor - $
             data(gfit_ux).rstudy_u_stats(0,2)*factor
            erry2 = data(gfit_ux).rstudy_u_stats(0,1)*factor + $
             data(gfit_ux).rstudy_u_stats(0,3)*factor
         ENDIF
      END
      20:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_y_str & cytit='km/s [PAY]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_y_str & cytit='km/s [PAY]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_y_str & cytit='km/s [PAY]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_y_str & cytit='km/s [PAY]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_pay(1)*factor
         dt=13.8
         ylog=0 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_uy).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_uy).u_pay(1)*factor - $
             data(gfit_uy).rstudy_u_stats(1,2)*factor
            erry2 = data(gfit_uy).u_pay(1)*factor + $
             data(gfit_uy).rstudy_u_stats(1,3)*factor
         ENDIF
      END
      21:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_y_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_y_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_y_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_y_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(gfit_uy).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(gfit_uy).rstudy_u_stats(1,1)*factor
         dt=13.8
         ylog=0 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_uy).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_uy).rstudy_u_stats(1,1)*factor - $
             data(gfit_uy).rstudy_u_stats(1,2)*factor
            erry2 = data(gfit_uy).rstudy_u_stats(1,1)*factor + $
             data(gfit_uy).rstudy_u_stats(1,3)*factor
         ENDIF
      END
      22:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_z_str & cytit='km/s [PAY]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_z_str & cytit='km/s [PAY]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_z_str & cytit='km/s [PAY]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_z_str & cytit='km/s [PAY]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_pay(2)*factor
         dt=13.8
         ylog=0 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_uz).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_uz).u_pay(2)*factor - $
             data(gfit_uz).rstudy_u_stats(2,2)*factor
            erry2 = data(gfit_uz).u_pay(2)*factor + $
             data(gfit_uz).rstudy_u_stats(2,3)*factor
         ENDIF
      END
      23:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_z_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_z_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_z_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_z_str+' (Rstudy)' & cytit='km/s [PAY]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(gfit_uz).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(gfit_uz).rstudy_u_stats(2,1)*factor
         dt=13.8
         ylog=0 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_uz).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_uz).rstudy_u_stats(2,1)*factor - $
             data(gfit_uz).rstudy_u_stats(2,2)*factor
            erry2 = data(gfit_uz).rstudy_u_stats(2,1)*factor + $
             data(gfit_uz).rstudy_u_stats(2,3)*factor
         ENDIF
      END
      24:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_x_str & cytit='km/s [GSM]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_x_str & cytit='km/s [GSM]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_x_str & cytit='km/s [GSM]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_x_str & cytit='km/s [GSM]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_gsm(0)*factor
         dt=13.8
         ylog=0 
      END
      25:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_y_str & cytit='km/s [GSM]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_y_str & cytit='km/s [GSM]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_y_str & cytit='km/s [GSM]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_y_str & cytit='km/s [GSM]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_gsm(1)*factor
         dt=13.8
         ylog=0 
      END
      26:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+u_z_str & cytit='km/s [GSM]'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+u_z_str & cytit='km/s [GSM]'
               factor = sqrt(1./4.)
            END
            2:BEGIN
               ctit2 = hepp_str+u_z_str & cytit='km/s [GSM]'
               factor = sqrt(2./4.)
            END
            3:BEGIN
               ctit2 = oxy_str+u_z_str & cytit='km/s [GSM]'
               factor = sqrt(1./16.)
            END
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).u_gsm(2)*factor
         dt=13.8
         ylog=0 
      END
      27:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+kt_parl_str & cytit='eV'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+kt_parl_str & cytit='eV'
               factor = 1.
            END
            2:BEGIN
               ctit2 = hepp_str+kt_parl_str & cytit='eV'
               factor = 2.
            END
            3:BEGIN
               ctit2 = oxy_str+kt_parl_str & cytit='eV'
               factor = 1.
            END 
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).ktparl*factor
         dt=13.8
         ylog=1 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_tparl).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_tparl).ktparl*factor - $
             data(gfit_tparl).rstudy_tparl_stats(2)*factor
            erry2 = data(gfit_tparl).ktparl*factor + $
             data(gfit_tparl).rstudy_tparl_stats(3)*factor
         ENDIF
      END
      28:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+kt_parl_str+' (Rstudy)' & cytit='eV'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+kt_parl_str+' (Rstudy)' & cytit='eV'
               factor = 1.
            END
            2:BEGIN
               ctit2 = hepp_str+kt_parl_str+' (Rstudy)' & cytit='eV'
               factor = 2.
            END
            3:BEGIN
               ctit2 = oxy_str+kt_parl_str+' (Rstudy)' & cytit='eV'
               factor = 1.
            END 
         ENDCASE
         t90= data(gfit_tparl).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(gfit_tparl).rstudy_tparl_stats(1)*factor
         dt=13.8
         ylog=1 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_tparl).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_tparl).rstudy_tparl_stats(1)*factor - $
             data(gfit_tparl).rstudy_tparl_stats(2)*factor
            erry2 = data(gfit_tparl).rstudy_tparl_stats(1)*factor + $
             data(gfit_tparl).rstudy_tparl_stats(3)*factor
         ENDIF
      END
      29:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+kt_perp_str & cytit='eV'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+kt_perp_str & cytit='eV'
               factor = 1.
            END
            2:BEGIN
               ctit2 = hepp_str+kt_perp_str & cytit='eV'
               factor = 2.
            END
            3:BEGIN
               ctit2 = oxy_str+kt_perp_str & cytit='eV'
               factor = 1.
            END 
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).ktperp*factor
         dt=13.8
         ylog=1 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_tperp).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_tperp).ktperp*factor - $
             data(gfit_tperp).rstudy_tperp_stats(2)*factor
            erry2 = data(gfit_tperp).ktperp*factor + $
             data(gfit_tperp).rstudy_tperp_stats(3)*factor
         ENDIF
      END
      30:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+kt_perp_str+' (Rstudy)' & cytit='eV'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+kt_perp_str+' (Rstudy)' & cytit='eV'
               factor = 1.
            END
            2:BEGIN
               ctit2 = hepp_str+kt_perp_str+' (Rstudy)' & cytit='eV'
               factor = 2.
            END
            3:BEGIN
               ctit2 = oxy_str+kt_perp_str+' (Rstudy)' & cytit='eV'
               factor = 1.
            END 
         ENDCASE
         t90= data(gfit_tperp).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(gfit_tperp).rstudy_tperp_stats(1)*factor
         dt=13.8
         ylog=1 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_tperp).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_tperp).rstudy_tperp_stats(1)*factor - $
             data(gfit_tperp).rstudy_tperp_stats(2)*factor
            erry2 = data(gfit_tperp).rstudy_tperp_stats(1)*factor + $
             data(gfit_tperp).rstudy_tperp_stats(3)*factor
         ENDIF
      END
      31:BEGIN
         CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+p_parl_str & cytit='cgs'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+p_parl_str & cytit='cgs'
               factor = sqrt(4.*1.)
            END
            2:BEGIN
               ctit2 = hepp_str+p_parl_str & cytit='cgs'
               factor = sqrt(4.*2.)
            END
            3:BEGIN
               ctit2 = oxy_str+p_parl_str & cytit='cgs'
               factor = sqrt(16.*1.)
            END 
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).pres_parl*factor
         gd = where(ydat GT 0)
         IF (gd(0) NE -1) THEN t90 = t90(gd)
         IF (gd(0) NE -1) THEN ydat = ydat(gd)
         dt=13.8
         ylog=1 
      END
      32:BEGIN
        CASE type(2) OF
            0:BEGIN
               ctit2 = hyd_str+p_perp_str & cytit='cgs'
               factor = 1.
            END
            1:BEGIN
               ctit2 = hep_str+p_perp_str & cytit='cgs'
               factor = sqrt(4.*1.)
            END
            2:BEGIN
               ctit2 = hepp_str+p_perp_str & cytit='cgs'
               factor = sqrt(4.*2.)
            END
            3:BEGIN
               ctit2 = oxy_str+p_perp_str & cytit='cgs'
               factor = sqrt(16.*1.)
            END 
         ENDCASE
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).pres_perp*factor
         gd = where(ydat GT 0)
         IF (gd(0) NE -1) THEN t90 = t90(gd)
         IF (gd(0) NE -1) THEN ydat = ydat(gd)
         dt=13.8
         ylog=1 
      END
      33:begin
         ctit2=hyd_str+'Gyrotropy' & cytit=' '
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).gyrotropy
         dt=13.8
         ylog=0 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_gyrotropy).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_gyrotropy).gyrotropy - $
             data(gfit_gyrotropy).rstudy_gyrotropy_stats(2)
            erry2 = data(gfit_gyrotropy).gyrotropy + $
             data(gfit_gyrotropy).rstudy_gyrotropy_stats(3)
         ENDIF
      END 
      34:begin
         ctit2=hyd_str+'Gyrotropy (Rstudy)' & cytit=' '
         t90= data(gfit_gyrotropy).time_ion + $
          ( kp_extra_header.start_time - xut1 )
         ydat= data(gfit_gyrotropy).rstudy_gyrotropy_stats(1)
         dt=13.8
         ylog=0 
         IF (type(1) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_gyrotropy).time_ion + $
             (kp_extra_header.start_time - xut1)+dt/2.
            erry1 = data(gfit_gyrotropy).rstudy_gyrotropy_stats(1) - $
             data(gfit_gyrotropy).rstudy_gyrotropy_stats(2)
            erry2 = data(gfit_gyrotropy).rstudy_gyrotropy_stats(1) + $
             data(gfit_gyrotropy).rstudy_gyrotropy_stats(3)
         ENDIF
      END
      35:begin
         ctit2=hyd_str+'rchi2 All' & cytit=' '
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).rchi2_all
         dt=13.8
         ylog=1 
      END
      36:begin
         ctit2=hyd_str+'rchi2 Patch' & cytit=' '
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).rchi2_patch
         dt=13.8
         ylog=1 
      END
      37:begin
         ctit2=hyd_str+'npatch' & cytit=' '
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).npatch
         dt=13.8
         ylog=1 
      END
      38:begin
         ctit2=hyd_str+'nrealizations' & cytit=' '
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).nrealizations
         dt=13.8
         ylog=1 
      END
      39:begin
         ctit2=hyd_str+'Anisotropy' & cytit=' '
         t90= data(good_best).time_ion + ( kp_extra_header.start_time - xut1 )
         ydat= data(good_best).ktparl/data(good_best).ktperp
         dt=13.8
         ylog=1 
      END
   ENDCASE 
   
   ctit=' '                     ; Do not use this variable!
   
   cytit= font+cytit
   ctit2= font+ctit2
   
; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)
   
; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images
   
; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF
   
   index=where((t90 ge 0) and (t90 le dxut),count)
   if count ge 2 then begin
      t90=t90(index)
      ydat=ydat(index)
   end
   
; insert breaks where there are data dropouts. (grid the data)
   dt= t90(1:*) - t90(0:*)
   r= where(dt gt 0)
   if r(0) ne -1 then dt=min(dt(r)) else dt=1
   nn= max(t90)/dt + 1
   ydat_grid= make_array( nn, /float, value=1e31 )
   t90_grid= make_array( nn, /float, value=0 )
   r= long( t90/dt )
   ydat_grid( r ) = ydat
   t90_grid( r ) = t90
   
   t90= t90_grid
   ydat= ydat_grid
   
   
; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin
;         yscl(*,2)  is then ymax
;         yscl(*,3)  is ylog
; when scaling is set to automatic, the automatic y limits are loaded in.
   
   y_kw= hydra_ytick_parm( ydat, panel=panel(0), ylog=ylog, fill=1e31 )
   
; set up extra plot keywords, common ones first
   
   extra_plotPar_common={xrange:[0,dxut], $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xtickv:xtickv, xminor:xminor, xticks:xticks, $
                         yticklen:0.010, ytitle:cytit, $
                         ythick:1.3, xthick:1.3, max_value:1e30 }
   
   extra_plotPar_common= $
     create_struct( extra_plotPar_common, y_kw )
   
; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.
   
   if n_elements(extra_par) EQ 0 then $
    extra_par={color:1}
   
; add keyword structure set here with the one passed in
   
   extra_plotPar= create_struct( extra_plotPar_common, extra_par )
   
   axis_plotPar= create_struct( extra_plotPar_common, {color:1} )
   
; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot
   
   if panel(0) eq 0 then BEGIN
      xtickformat='hydra_timeticks'
   ENDIF ELSE BEGIN 
      xtickformat='noticks'
   ENDELSE

; plot the axis in color=1, and only plot if if you're not overplotting
   IF NOT keyword_set(overplot) THEN BEGIN 
      plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
       xstyle=1, ystyle=1, xtickformat=xtickformat
      if panel(0) eq 0 then $
       x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   ENDIF 
   
; now do the actual data plot
   hydra_plot, t90, ydat, _extra=extra_plotPar, xstyle=5, ystyle=5, $
    overplot= keyword_set(overplot)
   
   IF (errorbars EQ 1) THEN BEGIN
      FOR ii=0,n_elements(errx)-1 DO BEGIN
         oplot,[errx(ii),errx(ii)],[erry1(ii),erry2(ii)],line=0, $
          color=extra_plotPar.color
         oplot,[errx(ii)-dt/2.,errx(ii)+dt/2.],[erry1(ii),erry1(ii)],line=0, $
          color=extra_plotPar.color
         oplot,[errx(ii)-dt/2.,errx(ii)+dt/2.],[erry2(ii),erry2(ii)],line=0, $
          color=extra_plotPar.color
      ENDFOR
   ENDIF
   
   if not keyword_set(overplot) then begin
       plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
         xstyle=1, ystyle=1, xtickformat=xtickformat
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   endif

; store the coordinate information into common block
   plotted_x = !x
   plotted_y = !y
   
   !x=oldx
   !y=oldy
   
; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015
   
   if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
      r= where( (xtickv+xut1) mod 86400 eq 0, count )
      if r(0) ne -1 then begin
         xtickv= xtickv(r)
         xticks= count
      endif else begin
         xtickv= xtickv([0,n_elements(xtickv)-1])
         xticks= 1
      endelse
      
      axis_plotPar.xtickv= xtickv
      axis_plotPar.xticks= xticks
      
      plot, t90, ydat, /NODATA, $
       _extra=axis_plotPar, xtickformat='hydra_juldate', $
       xstyle=1, ystyle=5
      
   endif
   
; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
   abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
   lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
   right_side_label,panel,lfs+ctit2, _extra=extra_plotPar
   
   hydra_footprint, kp_extra_header
   
   return
END 





