;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_moments, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hydra_moments for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_moments, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_moments, panel, type, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           _EXTRA=extra_par, subtable=subtable



; The following common blocks are needed:

   common time, xut1,xut2 ;common time limits in T90
   common mjdt, mjdt1, mjdt2

;   common hydra_yscale, yscl
   common yscale, yscl			;man/auto yscaling
   common coordinateSystems, plotted_x, plotted_y
                                        ;info on coords used by mouse

   common hydra_moments_data, $
    moments_species, $
    moments_header, $
    moments_ele_data, $
    moments_ion_data

   COMMON hydra_moments_labels, $
    moments_labels

   common papco_color_names

   panelset,panel			;sets the panel position viewport

   oldy=!y
   oldx=!x

   goodticks, xtickv, xminor
   xticks=n_elements(xtickv)-1

; set duplex font if big_gif is device ; duplex roman
   if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')

; the plot calls normally use the arrays t90 for the time axis and yray (ymat
; for color plots) for the y or z axis.
   dxut= xut2-xut1


   IF (moments_species EQ 0) THEN BEGIN
       data = moments_ele_data      
       ispecies = -1
       panel_title='MOM Ele!c'
       spec_super='!r!ue'
   ENDIF ELSE BEGIN       
       data = moments_ion_data
       ispecies = 1
       panel_title='MOM Ion!c'
       spec_super='!r!ui'
   ENDELSE

   if getenv( 'hydra_presentation_mode' ) eq '1' then panel_title=''
   if type(1) eq 56 then panel_title=''
   
   str_species= font + panel_title

; Define the labels
   IF (n_elements(moments_labels) EQ 0) THEN $
      hydra_define_structure,/mom_lab

   pl= moments_labels(*,2)
   parl='!b!9#!x'
   perp='!b!9x!x'

   pl(56)='!sG!r/!s!d23ms!n'+spec_super ; High Res. Angyrotropy (Signal)
   pl(22)='!sG!r/!s!dMOM!n'+spec_super ; Angyrotropy
   pl(0)='n!s'+spec_super
   pl(1)='kT!s'+spec_super
   pl(18)='kT!s'+parl+spec_super
   pl(19)='kT!s'+perp+spec_super
   pl(20)='P!s'+parl+spec_super
   pl(21)='P!s'+perp+spec_super
   pl(24)='V!s!dA!n'+spec_super
   pl(25)='M!s!dA!n'+spec_super
   pl(26)='M!s!dA!n'+parl+spec_super
   pl(27)='M!s!dA!n'+perp+spec_super
   pl(29)='Q!s'+spec_super
   pl(30)='Q!s'+parl+spec_super
   pl(31)='!10#!xQ!s'+perp+'!n!10#!x'+spec_super
   pl(3)= 'U!s!dx-payload'+spec_super
   pl(4)= 'U!s!dy-payload'+spec_super
   pl(5)= 'U!s!dz-payload'+spec_super
   pl(6)= 'U!s!dx-GSM'+spec_super
   pl(7)= 'U!s!dy-GSM'+spec_super
   pl(8)= 'U!s!dz-GSM'+spec_super
   pl(9)= 'U!s'+spec_super
   pl(10)= 'U!s'+parl+spec_super
   pl(11)= 'U!s'+perp+'!n!dx-payload'+spec_super
   pl(12)= 'U!s'+perp+'!n!dy-payload'+spec_super
   pl(13)= 'U!s'+perp+'!n!dz-payload'+spec_super
   pl(14)= 'U!s'+perp+'!n!dx-GSM'+spec_super
   pl(15)= 'U!s'+perp+'!n!dy-GSM'+spec_super
   pl(16)= 'U!s'+perp+'!n!dz-GSM'+spec_super
   pl(17)= '!9!!!xU!s'+perp+'!n!9!!!x'+spec_super
   pl(23)= 'A!s'+spec_super
   pl(38)= 'B!dz-GSM'

   if getenv( 'hydra_presentation_mode' ) eq '1' then begin
       ctit2 = str_species+pl(type(1))
   endif else begin       
       ctit2 = str_species+moments_labels(type(1),2)
   endelse
   cytit = moments_labels(type(1),3)

;decom   IF (type(2)) THEN color_code = 1 ELSE color_code = 0
   color_code = 0

 plot_name=' '
 eplot = 0
 oplot_status = 0
 vector = ' '
 case type(1) of
     0:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).density 
         dt=13.8
         ylog=1
         species=ispecies

         oplot_status = 1
         oplot_y = 20
     end
     1:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).mean_energy
         dt=13.8
         ylog=1
         species=ispecies
     end
     2:begin                    ; mean energy overplot for spectrogram?
         plot_name='meane_overplot'
         ctit2=' ' & cytit=' '
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).mean_energy
         dt=13.8
         ylog=1
         species=ispecies
     END
     3:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_pay(0)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     4:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_pay(1)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     5:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_pay(2)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     6:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_gsm(0)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     7:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_gsm(1)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     8:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_gsm(2)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     9:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_pay(3)
         dt=13.8
         ylog=1
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     10:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_parl
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     11:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_perp_pay(0)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     12:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_perp_pay(1)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     13:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_perp_pay(2)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     14:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_perp_gsm(0)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     15:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_perp_gsm(1)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     16:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_perp_gsm(2)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     17:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_perp_pay(3)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
     END
     18:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).ktparl
         dt=13.8
         ylog=1
         species=ispecies
     END
     19:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).ktperp
         dt=13.8
         ylog=1
         species=ispecies
     END
     20:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).pres_parl
         dt=13.8
         ylog=1
         species=ispecies
     END
     21:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).pres_perp
         dt=13.8
         ylog=1
         species=ispecies
     END
     22:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).gyrotropy
         dt=13.8
         ylog=1
         species=ispecies
         oplot_status = 1
         oplot_y = 1
     END
     23:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).anisotropy
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 1
         hydra_plot_kw= { reference:1., $
                          poly_above_ref:red, $
                          poly_below_ref:blue }
     END
     24:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).v_alf
         dt=13.8
         ylog=1
         species=ispecies
     END
     25:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).m_tot
         dt=13.8
         ylog=1
         species=ispecies
         oplot_status = 1
         oplot_y = 1
         hydra_plot_kw= { reference:0., $
                          poly_above_ref:yellow, $
                          poly_below_ref:cyan }         
     END
     26:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).m_parl
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 1
         hydra_plot_kw= { reference:0., $
                          poly_above_ref:yellow, $
                          poly_below_ref:cyan }         
     END
     27:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).m_perp
         dt=13.8
         ylog=1
         species=ispecies
         oplot_status = 1
         oplot_y = 1
         hydra_plot_kw= { reference:0., $
                          poly_above_ref:yellow, $
                          poly_below_ref:cyan }         
     END
     28:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).sc_potential
         dt=13.8
         ylog=0
         species=ispecies
     END
     29:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).q_pay(3)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'heatflux'
     END
     30:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).q_parl
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'heatflux'
         hydra_plot_kw= { reference:0., $
                          poly_above_ref:red, $
                          poly_below_ref:blue }
     END
     31:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).q_perp_pay(3)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'heatflux'
     END
     32:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).qparl_over_bmag
         dt=13.8
         ylog=0
         species=ispecies
         hydra_plot_kw= { reference:0., $
                          poly_above_ref:red, $
                          poly_below_ref:blue }
     END
     33:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).b_pay(0)
         ydat_sig = data(*).b_pay_sig(0)
         eplot = 1
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
     END
     34:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).b_pay(1)
         ydat_sig = data(*).b_pay_sig(1)
         eplot = 1
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
     END
     35:begin
        t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
        ydat= data(*).b_pay(2)
        ydat_sig = data(*).b_pay_sig(2)
        eplot = 1
        dt=13.8
        ylog=0
        species=ispecies
         oplot_status = 1
         oplot_y = 0
     END
     36:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).b_gsm(0)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
     END
     37:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).b_gsm(1)
         dt=13.8
         species=ispecies
         oplot_status = 1
         oplot_y = 0
     END
     38:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).b_gsm(2)
         dt=13.8
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         hydra_plot_kw= { reference:0., $
                          poly_above_ref:red, $
                          poly_below_ref:blue }
     END
     39:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).b_pay(3)
         ydat_sig = data(*).b_pay_sig(3)
         eplot = 1
         ylog = 0
         dt=13.8
         species=ispecies
         oplot_status = 1
         oplot_y = 0
     END
     40:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).angle_bavg_bproxy*180./!pi
         ydat_sig = data(*).angle_bavg_bproxy_sig*180./!pi
         eplot = 1
         ylog = 0
         dt=13.8
         species=ispecies
     END
     41:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).cosangle_q_bavg
         ydat_sig = data(*).cosangle_q_bavg_sig
         eplot = 1
         ylog = 0
         dt=13.8
         species=ispecies
     END
     42:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).cosangle_u_bavg
         ydat_sig = data(*).cosangle_u_bavg_sig
         eplot = 1
         ylog = 0
         dt=13.8
         species=ispecies
     END
     43:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).b_pay_angle_cone*180./!pi
         dt=13.8
         species=ispecies
     END

; don't change the diagonalization status from the 44th position
     44:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).diagonalization_status
         dt=13.8
         ylog = 0
         species=ispecies
     END
     45:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= -data(*).u_pay(3)
         dt=13.8
         ylog = 0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         ctit2 = ' '
         vector = 'flow'
      END
      46:BEGIN
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= -data(*).q_pay(3)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         ctit2 = ' '
         vector = 'heatflux'
      END
     47:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_gse(0)
         dt=13.8
         ylog = 0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
      END
     48:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_gse(1)
         dt=13.8
         ylog = 0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
      END
     49:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).u_gse(2)
         dt=13.8
         ylog = 0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'flow'
      END
     50:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).q_gsm(0)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'heatflux'
     END
     51:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).q_gsm(1)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'heatflux'
     END
     52:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).q_gsm(2)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'heatflux'
     END
     53:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).q_gse(0)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'heatflux'
     END
     54:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).q_gse(1)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'heatflux'
     END
     55:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).q_gse(2)
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         vector = 'heatflux'
     END 
     56:begin                   ; "hr angyrotropy" panel.  Added by jbf.
         convert_t90_to_Date, xut1, year, day_oy, hour, minute, second, $ ;y2k
           month, cmonth, day_om
         if year lt 100 then year=year+1900
         datestr= strtrim( year, 2 ) + string( month, form='(i2.2)' ) + $
           string( day_om, form='(i2.2)' )
         comp_out_2=1
         itimeout=1
         itimerangeout=1
         
         dir= getenv( 'HD_DATA_SETS' )
         restore, dir+'angyrotropy_new.idlsav', /verbose
         angyro= angyrotropy.angyrotropy
         baseline= angyrotropy.baseline

         t90= convert_date_to_t90( year=year, month=month, day=day_om ) - $
           xut1 + angyrotropy.time

         index_real= angyrotropy.idx_real 

         dt1= 13.8
         ydat= angyro - baseline ; Signal
         
         oplot_status = 1
         oplot_y = -1e31
         species= ispecies

     END 
     57:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).qnu_parl
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
     END
     58:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).eflux_parl
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         hydra_plot_kw= { reference:0., $
                          poly_above_ref:red, $
                          poly_below_ref:blue }
     END
     59:begin
         t90= data(*).time + ( moments_header(moments_species).start_time - xut1 )
         ydat= data(*).eflux_parl_over_b
         dt=13.8
         ylog=0
         species=ispecies
         oplot_status = 1
         oplot_y = 0
         hydra_plot_kw= { reference:0., $
                          poly_above_ref:red, $
                          poly_below_ref:blue }
     END
 ENDCASE 
 
; If color-coding is desired, find the color_types!
  IF (color_code) THEN BEGIN 
     IF (vector EQ 'flow') THEN BEGIN
        hydra_calc_b_alignment, data.u_pay, data.b_pay, data.b_pay_angle_cone,$
         color_type
     ENDIF ELSE IF (vector EQ 'heatflux') THEN BEGIN 
        hydra_calc_b_alignment, data.q_pay, data.b_pay, data.b_pay_angle_cone,$
         color_type
     ENDIF ELSE BEGIN 
       color_code = 0
     ENDELSE
  ENDIF 

; Filter out the 'bad' points (according to status)!!  
  if ((type(2) and 1) eq 1) then begin  ; Patch invalid
      message, 'Filtering out the "bad" moments points (status le 0 or 5th digit of status=3, or b_status ne 1)',/cont
      patch_flag = long(data.moments_status/10000.) MOD 10 ; indicates the fifth digit of data.moments_status
      notgd = where(data.moments_status le 0 or patch_flag eq 3 or data.b_status ne 1)
                                ; if the fifth digit is equal to 3, it
                                ; means no patch, discards this block
   
      IF (notgd(0) NE -1) THEN ydat(notgd) = 1e31
  endif

; Filter out invalid data (according to fill value)
  r = where(ydat EQ -1.e+31)
  if r(0) ne -1 then ydat(r)= 1e31
  
  cytit= font+cytit
  ctit2= font+ctit2

; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images

; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF

  if type(1) ne 56 then begin   ; 56 is angyrotropy
      index=where((t90 ge 0) and (t90 le dxut),count)
      if count ge 2 then begin
          t90=t90(index)
          ydat=ydat(index)
          IF (eplot) THEN ydat_sig = ydat_sig(index)
          IF (color_code) THEN color_type = color_type(index)
      endif else goto, nodata
  endif

  if ( type(2) and 2 ) eq 2 then begin
      hd_make_timetags, ttag, t90+xut1, 13.8, /t90
      hyd_access, ttag, 'return_current_quality', rc_index
      index= where( rc_index gt 0.1 and rc_index lt 10.0 )
      if index(0) ne -1 then begin
          t90= t90(index)
          ydat= ydat(index)
          IF (eplot) THEN ydat_sig = ydat_sig(index)
          IF (color_code) THEN color_type = color_type(index)
      endif      
  endif

; insert breaks where there are data dropouts. (grid the data)
  if type(1) ne 56 then begin   ; 56 is angyrotropy
      dt= t90(1:*) - t90(0:*)
      r= where(dt gt 0)
      if r(0) ne -1 then dt=min(dt(r)) else dt=1
      nn= max(t90)/dt + 1
      ydat_grid= make_array( nn, /float, value=1e31 )
      IF (eplot) THEN ydat_sig_grid = make_array( nn, /float, value=1e31 )
      IF (color_code) THEN color_type_grid = make_array( nn, /float, value=1e31 )
      t90_grid= make_array( nn, /float, value=0 )
      r= long( t90/dt )
      ydat_grid( r ) = ydat
      IF (eplot) THEN ydat_sig_grid( r )= ydat_sig
      IF (color_code) THEN color_type_grid( r ) = color_type
      t90_grid( r ) = t90
  
      t90= t90_grid
      ydat= ydat_grid
      IF (eplot) THEN ydat_sig = ydat_sig_grid
      IF (color_code) THEN color_type = color_type_grid
  endif

  nodata:
; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin
;         yscl(*,2)  is then ymax
;         yscl(*,3)  is ylog
; when scaling is set to automatic, the automatic y limits are loaded in.

;  hydra_ylim_new, panel(0), type, ydat, yrange=yrange, ylog=ylog
  r= where( ydat lt 1e30 )
  y_kw= hydra_ytick_parm( ydat(r), panel=panel(0), ylog=ylog )

; set up extra plot keywords, common ones first
  
  extra_plotPar_common={ xrange:[0,dxut], $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xtickv:xtickv, xminor:xminor, xticks:xticks, $
                         yticklen:0.010, ytitle:cytit, $
                         ythick:1.3, xthick:1.3, max_value:1e30 }
  extra_plotPar_common= create_struct( extra_plotPar_common, y_kw )

; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

   if n_elements(extra_par) EQ 0 then $
     extra_par={color:1}

; add keyword structure set here with the one passed in

   extra_plotPar= create_struct( extra_plotPar_common, extra_par )

   axis_plotPar= create_struct( extra_plotPar_common, {color:1} )
   
; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot

   if panel(0) eq 0 then begin
       xtickformat= 'hydra_timeticks'
   endif else begin
       xtickformat= 'noticks'
   ENDELSE

; plot the axis in color=1, and only plot if if you're not overplotting
   if not keyword_set(overplot) then begin
       plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
         xstyle=1, ystyle=1, xtickformat=xtickformat
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03

       IF (color_code) THEN BEGIN
          ry = yrange
          FOR ii=0,n_elements(t90)-2 DO BEGIN
             IF (color_type(ii) NE 1e31) THEN $
              polyfill, [t90(ii),t90(ii),t90(ii)+13.8,t90(ii)+13.8], $
              [ry(0),ry(1),ry(1),ry(0)], $
              color=long(color_type(ii))
          ENDFOR 
       ENDIF 

       IF (oplot_status) THEN BEGIN 
           if total( abs( !y.crange ) ) lt 1e30 then $ ; check for !inf
             oplot, !x.crange, [oplot_y,oplot_y], line=1, color=dark_grey
       ENDIF
       
   endif

   if n_elements( hydra_plot_kw ) gt 0 then begin
       hp_kw= create_struct( hydra_plot_kw, extra_plotPar )
   endif else begin
       hp_kw= extra_plotPar
   endelse

; now do the actual data plot
   if type(1) ge 25 and type(1) le 27 then begin
       hydra_plot, t90, ydat, _extra=hp_kw, xstyle=5, ystyle=5, $
         overplot= keyword_set( overplot ), dx=dt1       
       hydra_plot, t90, ydat, _extra=extra_plotPar, $
         reference=1., poly_above_ref=red, $
         xstyle=5, ystyle=5, $
         overplot= keyword_set( overplot ), dx=dt1 
       hydra_plot, t90, ydat, _extra=extra_plotPar, $
         reference=-1., poly_below_ref=blue, $
         xstyle=5, ystyle=5, $
         overplot= keyword_set( overplot ), dx=dt1 
   endif else if type(1) eq 56 then begin ; JBF -- sorry Pam!  ; 56 is angyrotropy

;===============================================================
; This doesn't seem to work right  PPQ 
;       if n_elements( index_real ) gt 0 then begin
;           if !y.type(0) eq 1 then y0= !y.crange(0) else y0=0.
;           for i=0,n_elements(index_real)-1 do begin
;               y1= ydat(index_real(i))
;               polyfill, $
;                 ( [0,dt1,dt1,0,0]+t90(index_real(i))) > !x.crange(0) < !x.crange(1), $
;                 [y0,y0,y1,y1,y0], $
;                 color=get_color_index('red')
;           endfor
;       endif
;===============================================================
       hydra_plot, t90, ydat, _extra=hp_kw, xstyle=5, ystyle=5, $
         overplot= keyword_set( overplot ), dx=dt1       
   endif else begin
       hydra_plot, t90, ydat, _extra=hp_kw, xstyle=5, ystyle=5, $
         overplot= keyword_set( overplot )
   endelse
   
   IF (eplot) THEN BEGIN
       FOR jkl=0,n_elements(t90)-1 DO BEGIN 
           oplot, [t90(jkl)+dt/2.,t90(jkl)+dt/2.],[ydat(jkl)-ydat_sig(jkl),ydat(jkl)+ydat_sig(jkl)],line=0,color=extra_plotPar.color
       ENDFOR
   ENDIF
   
   if not keyword_set(overplot) then begin
       plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
         xstyle=1, ystyle=1, xtickformat=xtickformat
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   endif

; store the coordinate information into common block
   plotted_x = !x
   plotted_y = !y

   !x=oldx
   !y=oldy

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015

  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
     r= where( (xtickv+xut1) mod 86400 eq 0, count )
     if r(0) ne -1 then begin
         xtickv= xtickv(r)
         xticks= count
     endif else begin
         xtickv= xtickv([0,n_elements(xtickv)-1])
         xticks= 1
     endelse

     axis_plotPar.xtickv= xtickv
     axis_plotPar.xticks= xticks

     plot, t90, ydat, /NODATA, $
       _extra=axis_plotPar, xtickformat='hydra_juldate', $
       xstyle=1, ystyle=5

  endif

; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
  abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
  lfs= strmid( abunchoflfs, 0, 4*(plots_atthisposition<10) )
  right_side_label,panel,lfs+ctit2, _extra=extra_plotPar

  hydra_footprint, moments_header(moments_species)

  footprint= [ ctit2, $
               'File version: '+$
               strtrim(moments_header(moments_species).data_version,2) ]
  hyd_ident, footprint, concise=''
end





