pro x_ephemeris_label,labels,yposdata

;plot x-axis ephemeris labels on right side, fashioned after x_side_label.pro

  chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)*!P.CHARSIZE

  FOR ii=0,n_elements(yposdata)-1 DO BEGIN
     result = convert_coord(.5,yposdata(ii),/data,/to_normal)
     xyouts,!p.position(2)+chrs(0)*5,result(1),$
        labels(ii),/normal,color=1,charsize=!P.CHARSIZE,alignment=0

  ENDFOR


end
;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_other, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hydra_other for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_other, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_other, panel, type, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           _EXTRA=extra_par, subtable=subtable



; The following common blocks are needed:

   common time, xut1,xut2 ;common time limits in T90
   common mjdt, mjdt1, mjdt2

;   common hydra_yscale, yscl
   common yscale, yscl			;man/auto yscaling
   common coordinateSystems, plotted_x, plotted_y
                                        ;info on coords used by mouse

   common hydra_other_data, other_header, other_data, hydra_values

   common papco_color_names

   panelset,panel			;sets the panel position viewport

   oldy=!y
   oldx=!x

   goodticks, xtickv, xminor
   xticks=n_elements(xtickv)-1

   r_earth =  6378.1

; set duplex font if big_gif is device
   if !d.name eq 'Z' then font='!5' else font='' ; duplex roman

; the plot calls normally use the arrays t90 for the time axis and yray (ymat
; for color plots) for the y or z axis.
   dxut= xut2-xut1

   type_name=''
   
   case type(0) of
      0:begin
         ctit2='ILT' & cytit='deg'
         t90= other_data(*).time + ( other_header.start_time - xut1 )
         ydat= other_data(*).ilt
         dt=60
         this_Par={ ylog:0 }
      END
      1:begin
          ctit2='MAG LAT' & cytit='deg'
          t90= other_data(*).time + ( other_header.start_time - xut1 )
          ydat= other_data(*).maglat
          dt=60
          this_Par={ ylog:0 }
     end          
      2:begin
         ctit2='MLT' & cytit='hr'
         t90= other_data(*).time + ( other_header.start_time - xut1 )
         ydat= other_data(*).mlt
         dt=60
         this_Par={ ylog:0 }
      END
      3:begin
         ctit2='RADIUS' & cytit='R_E'
         t90= other_data(*).time + ( other_header.start_time - xut1 )
         ydat= other_data(*).radius/r_earth
         dt=60
         this_Par={ ylog:0 }
      END
      4:begin
         ctit2='L_SHELL' & cytit=' '
         t90= other_data(*).time + ( other_header.start_time - xut1 )
         ydat= other_data(*).lshell
         dt=60
         this_Par={ ylog:0 }
      END
      5:begin
         ctit2='Ephem Box' & cytit=' '
         t90= other_data(*).time + ( other_header.start_time - xut1 )
         ydat= other_data(*).lshell
         dt=60
         this_Par={ ylog:0 }
      END
      6:begin
         ctit2='X_GSM' & cytit='R_E'
         t90= other_data(*).time + ( other_header.start_time - xut1 )
         ydat= other_data(*).xgsm/r_earth
         dt=60
         this_Par={ ylog:0 }
      END
      7:begin
         ctit2='Y_GSM' & cytit='R_E'
         t90= other_data(*).time + ( other_header.start_time - xut1 )
         ydat= other_data(*).ygsm/r_earth
         dt=60
         this_Par={ ylog:0 }
      END
      8:begin
         ctit2='Z_GSM' & cytit='R_E'
         t90= other_data(*).time + ( other_header.start_time - xut1 )
         ydat= other_data(*).zgsm/r_earth
         dt=13.8
         this_Par={ ylog:0 }
      END
      9:begin
          ctit2='S/C Potential!cBest' & cytit='V'
          t90= hydra_values(*).time + ( other_header.start_time - xut1 )
          ydat= hydra_values(*).sc_pot(0)
          dt=13.8
          this_Par= { ylog:1 }
      end
      10:begin
          ctit2='S/C Potential!cEFI Berkeley' & cytit='V'
          t90= hydra_values(*).time + ( other_header.start_time - xut1 )
          ydat= hydra_values(*).sc_pot(1)
          dt=13.8
          this_Par= { ylog:1 }
      end
      11:begin
          ctit2='S/C Potential!cHydra' & cytit='V'
          t90= hydra_values(*).time + ( other_header.start_time - xut1 )
          ydat= hydra_values(*).sc_pot(2)
          dt=13.8
          this_Par= { ylog:1 }
      end
      12:begin
          ctit2='S/C Potential!cGSFC' & cytit='V'
          t90= hydra_values(*).time + ( other_header.start_time - xut1 )
          ydat= hydra_values(*).sc_pot(3)
          dt=13.8
          this_Par= { ylog:1 }
      end
      13:begin
          ctit2='DDEIS Background!cEle' & cytit='Counts'
          t90= hydra_values(*).time + ( other_header.start_time - xut1 )
          ydat= hydra_values(*).back_ele
          dt=13.8
          this_Par= { ylog:0 }
      end
      14:begin
          ctit2='DDEIS Background!cIon' & cytit='Counts'
          t90= hydra_values(*).time + ( other_header.start_time - xut1 )
          ydat= hydra_values(*).back_ion
          dt=13.8
          this_Par= { ylog:0 }
      end
      15:begin
          ctit2='PPA Background' & cytit='Counts/Sec'
          t90= hydra_values(*).time + ( other_header.start_time - xut1 )
          ydat= hydra_values(*).back_ppa
          dt=13.8
          this_Par= { ylog:0 }
      end
      16:begin
          ctit2='PPA Back / DD Back' & cytit=''
          t90= hydra_values(*).time + ( other_header.start_time - xut1 )
          ydat= make_array( n_elements( hydra_values ), /float, value=1e31 )
          rv= where( hydra_values(*).back_ele gt 0.05 )
          if rv(0) ne -1 then $
            ydat(rv)= hydra_values(rv).back_ppa / hydra_values(rv).back_ele
          dt=13.8
          this_Par= { ylog:1 }
      end
      17: begin
          ctit2='Hydra Status' & cytit=''
          t90= hydra_values(*).time + ( other_header.start_time - xut1 )
          ydat= hydra_values(*).psi_status * 0 ; we'll overplot the data
          dt=13.8
          this_Par= { ylog:0 }
          type_name='hydra_status'
      end
  endcase
  
   ctit=' '                     ; Do not use this variable!

   cytit= font+cytit
   ctit2= font+ctit2

; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images

; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF

   index=where((t90(1:*) ge 0) and (t90(1:*) le dxut),count)+1
   if count ge 2 then begin
      t90=t90(index)
      ydat=ydat(index)
   end

; insert breaks where there are data dropouts. (grid the data)
   if n_elements(dt) eq 0 then dt= t90(1:*) - t90(0:*)
   r= where(dt gt 0)
   if r(0) ne -1 then dt=min(dt(r)) else dt=1
   nn= max(t90)/dt + 1
   ydat_grid= make_array( nn, /float, value=1e31 )
   t90_grid= make_array( nn, /float, value=0 )
   r= long( t90/dt )
   ydat_grid( r ) = ydat
   t90_grid( r ) = t90

   t90= t90_grid
   ydat= ydat_grid


; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin
;         yscl(*,2)  is then ymax
;         yscl(*,3)  is ylog
; when scaling is set to automatic, the automatic y limits are loaded in.

   if n_elements(yscl) eq 0 then yscl=dblarr(10,4)

   if (yscl(panel(0),0) eq 1) then begin
      print, '% p_hydra_other: getting manual scale specifications...'
      yst= yscl(panel(0),1)
      yen= yscl(panel(0),2)
      this_Par.ylog= yscl(panel(0),3)

      IF (yst LE 0 AND this_Par.ylog EQ 1) THEN BEGIN
         print, '% p_hydra_other: Negative ymin: swwitching from log to linear...'
         yscl(panel(0),3) =  0
         this_Par.ylog = yscl(panel(0),3)
      ENDIF

   endif else begin             ; use autoscale results
      rv=where(ydat lt 1e31)
      if rv(0) ne -1 then rt=where( t90(rv) ge 0 and t90(rv) le dxut ) $
      else rt=-1
      if (rt(0) ne -1) then begin
          hydra_ylim, ydat(rv(rt)), yst, yen, ylog, _extra=this_Par
          this_Par.ylog=ylog
      endif else begin
          yst=0 & yen=1
          if this_Par.ylog then begin 
              yst=10^yst & yen=10^yen
          endif              
      endelse      
      yscl(panel(0),1)= yst
      yscl(panel(0),2)= yen
      yscl(panel(0),3)= this_Par.ylog
   endelse

; set up extra plot keywords, common ones first

   extra_plotPar_common={yrange:[yst, yen], xrange:[0,dxut], $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xtickv:xtickv, xminor:xminor, xticks:xticks, $
                         yticklen:0.010, ytitle:cytit, $
                         ythick:1.3, xthick:1.3, max_value:1e30 }

; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

   if n_elements(extra_par) EQ 0 then $
    extra_par={color:1}

; add keyword structure set here with the one passed in

   extra_plotPar= create_struct(extra_plotPar_common, $
                                create_struct( this_par, extra_par ) )

   axis_plotPar= create_struct(extra_plotPar_common, $
                               create_struct( this_par, {color:1} ) )

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot

   if panel(0) eq 0 then begin
       xtickformat= 'hydra_timeticks'
   endif else begin
       xtickformat= 'noticks'
   endelse


; plot the axis in color=1, and only plot if if you're not overplotting
   if not keyword_set(overplot) then begin
       IF (type(0) NE 5) THEN BEGIN
           plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
             xstyle=1, ystyle=1, xtickformat=xtickformat
       ENDIF ELSE BEGIN         ; ephem box
           plot,t90,ydat, _extra=axis_plotPar, /nodata, $
             xstyle=1, ystyle=5, xtickformat=xtickformat
       ENDELSE
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   endif

; hydra events "underplots" shaded regions, highliting specific events.
;       hydra_events

; now do the actual data plot
   IF (type(0) NE 5) THEN BEGIN
      hydra_plot, t90, ydat, _extra=extra_plotPar, xstyle=5, ystyle=5, $
        overplot= keyword_set( overplot )
   ENDIF ELSE BEGIN
      plot, t90, ydat, _extra=extra_plotPar, xstyle=5,ystyle=5, $
       max_value=1e31, /nodata
      
      yposcoeffs = [.7,.6,.5,.4,.3]
      ypos = !y.crange(0) + yposcoeffs*(!y.crange(1)-!y.crange(0))
      
;      print, 'crange: ',!y.crange(1),!y.crange(0)
;      print, 'ypos: ',ypos
;      print, 'maxy,miny: ',max(ydat),min(ydat)
;      print, 'minx,maxx: ',min(t90),max(t90)
;      print, 'xtickv: ',xtickv
      
      labels = strarr(n_elements(ypos))
      
      irow = 0
      labels(irow) = 'UT'
      FOR ii=0,n_elements(xtickv)-1 DO BEGIN
         rtime = (xtickv(ii) + xut1) mod 86400. ; Seconds since midnight
         hours = long(rtime/3600.)
         mins = long( (rtime MOD 3600)/3600. * 60. + .5)
         strout = string(hours,'(i2.2)')+':'+string(mins,'(i2.2)')
         xyouts,xtickv(ii),ypos(irow),strout,alignment=.5, color=1
      ENDFOR

      irow = 1
      labels(irow) = 'ILT'
      v = other_data.ilt
      x = other_data.time + ( other_header.start_time - xut1 ) ; Must be in t90 fmt
      u = xtickv
      result = interpol(v,x,u)

; Decomissioned, sign assignment performed now in r_hydra_other.pro
;decom; Apply the sign of maglat to ilt...
;decom      v = other_data.maglat
;decom      result2 = interpol(v,x,u)
;decom      sign_mask =  result2/abs(result2)
;decom      FOR ii=0,n_elements(result)-1 DO BEGIN
;decom          xyouts,xtickv(ii),ypos(irow),string(result(ii)*sign_mask(ii),'(f6.2)'),alignment=.5, color=1
;decom      ENDFOR
      FOR ii=0,n_elements(result)-1 DO BEGIN
          xyouts,xtickv(ii),ypos(irow),string(result(ii),'(f6.2)'), $
            alignment=.5, color=1
      ENDFOR

      irow = 2
      labels(irow) = 'MLT'
      v = other_data.mlt
      x = other_data.time + ( other_header.start_time - xut1 )
      u = xtickv
      result = interpol(v,x,u)
      FOR ii=0,n_elements(result)-1 DO BEGIN
         rtime = result(ii)*3600.
         hours = long(rtime/3600.)
         mins = long( (rtime MOD 3600)/3600. * 60. + .5)
         strout = string(hours,'(i2.2)')+':'+string(mins,'(i2.2)')
         xyouts,xtickv(ii),ypos(irow),strout,alignment=.5, color=1
      ENDFOR

      irow = 3
      labels(irow) = 'RADIUS'
      v = other_data.radius/r_earth
      x = other_data.time + ( other_header.start_time - xut1 )
      u = xtickv
      result = interpol(v,x,u)
      FOR ii=0,n_elements(result)-1 DO BEGIN
          xyouts,xtickv(ii),ypos(irow),string(result(ii),'(f6.2)'),alignment=.5, color=1
      ENDFOR

      irow = 4
      labels(irow) = 'L_SHELL'
      v = other_data.lshell
      x = other_data.time + ( other_header.start_time - xut1 )
      u = xtickv
      result = interpol(v,x,u)
      FOR ii=0,n_elements(result)-1 DO BEGIN
          xyouts,xtickv(ii),ypos(irow),string(result(ii),'(f6.1)'),$
            alignment=.5, color=1
      ENDFOR
      
      x_ephemeris_label,labels,ypos

  ENDELSE

  if type_name eq 'hydra_status' then begin
      t90= hydra_values(*).time + ( other_header.start_time - xut1 )
      psi_status= hydra_values(*).psi_status
      mode= hydra_values(*).mode
      r= where( t90 eq 0 )
      if r(0) ne -1 then begin
          psi_status(r)= 9999
          mode(r) = 9999
      endif
      r= where( t90 ge 0 and t90 lt dxut )
      if r(0) eq -1 then r= indgen(2)
      t90= t90(r)
      psi_status= psi_status(r)
      mode= mode(r)

      r= where( psi_status eq 0 )
      data= t90*0 -1
      if r(0) ne -1 then begin
          data(r)=0.1
          oplot, t90, data, min_value=0, color=get_color_index('black'), $
            thick=3
      endif
      data= t90*0 -1
      r= where( psi_status eq 1 )
      if r(0) ne -1 then begin
          data(r)=0.1
          oplot, t90, data, min_value=0, color=get_color_index('yellow'), $
            thick=3
      endif
      xyouts, 0, 0.105, '   PSI Status', color=get_color_index('grey')

      modes= mode
      modes= modes( uniq( modes ))
      modes= modes(sort(modes))
      u= modes(uniq(modes))
      r= where( u ne 0 and u ne 9999 )
      u= u(r)
      colors=['red','green','blue','magenta','yellow','cyan']
      mn=strarr(50)
      mn(10)='1x2x'
      mn(22)='3x'
      mn(23)='4x'
      mn(24)='1x'
      mn(25)='5x'
      mn(40)='6x'
            
      for i=0,n_elements(u)-1<8 do begin          
          r= where( mode eq u(i) )
          data= t90*0.-1
          data(r)=0.5
          oplot, t90, data, $
            color=get_color_index(colors(i)), thick=3, min_value=0.
          textpos= (dxut/n_elements(u))*(i+0.5)
          oplot, [t90(r(0)),t90(r(0)),textpos],[0.5,0.54,0.65], color=1
          xyouts, textpos, 0.65, mn(u(i)), color=1
      endfor
  end

; store the coordinate information into common block
  plotted_x = !x
  plotted_y = !y

  !x=oldx
  !y=oldy

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015

  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
      r= where( (xtickv+xut1) mod 86400 eq 0, count )
      if r(0) ne -1 then begin
          xtickv= xtickv(r)
          xticks= count
      endif else begin
          xtickv= xtickv([0,n_elements(xtickv)-1])
          xticks= 1
      endelse

      axis_plotpar.xticks= xticks
      axis_plotpar.xtickv= xtickv

      plot, t90, ydat, /NODATA, $
        _extra= axis_plotpar, $
        xtickformat='hydra_juldate', $
        xstyle=1, ystyle=1

  endif

; don't print the name of the plot if it is over a colorplot.
  common plot_composer, widgetData
  panelKind= widgetData.plotInfos(*).panelkind
  idx= where( widgetData.plotsDrawn(*).panelvector(0) eq panel(0) and $
              widgetData.plotInfos(*).overplot eq 0 and $
              widgetData.plotInfos(*).panelkind ne '' )  
  idx= idx(0)
  if idx eq -1 then idx=0
  colorplot= widgetData.plotsDrawn(idx).zscl_min ne $
    widgetData.plotsDrawn(idx).zscl_max
  
  if not colorplot then begin
; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
      abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
      lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
      right_side_label,panel,lfs+ctit2, _extra=extra_plotPar
  endif
  
  if not overplot then hydra_footprint, other_header
end





