pro perform_hydra_walen, ion=ion

common walen_data, datestr, starttime, stoptime, $
  diff_method, zoom, wdata, inds, ref_index, wfit

common get_error, get_err_no, get_err_msg

myname = 'PERFORM_HYDRA_WALEN: '
;========================================================

IF keyword_set(ion) THEN BEGIN
    density = wdata.density_ion
    density_sigma = wdata.density_sigma_ion
    u = wdata.u_ion
    u_sigma = wdata.u_sigma_ion
ENDIF ELSE BEGIN
    density = wdata.density_ele
    density_sigma = wdata.density_sigma_ele
    u = wdata.u_ele
    u_sigma = wdata.u_sigma_ele
ENDELSE 

; Pressure is the total pressure for both ions and electrons
pres_parl = wdata.pres_parl
pres_perp = wdata.pres_perp

; Magnetic field
b = wdata.b_pay
b_sigma = wdata.b_pay_sigma

; Form the index pairs for the Walen differences
if (diff_method eq 1) then begin ; Asymptotic jump, only one diff
    ndiff = 1
    ind_pair = lonarr(1,2)
    ind_pair(0,0:1) = [inds(0),inds(n_elements(inds)-1)]
endif else begin
    ndiff = n_elements(inds)-1
    ind_pair = lonarr(ndiff+1,2)
    ind_pair(*,0) = inds
    ind_pair(*,1) = ref_index
    gd = where(ind_pair(*,0) ne ref_index)
    ind_pair = ind_pair(gd,0:1)
endelse

compstr = ['X','Y','Z']

ndeltas = ndiff*3               ; 3 components too
y = dblarr(ndeltas)
x = dblarr(ndeltas)
sigy = dblarr(ndeltas)
sigx = dblarr(ndeltas)
labelstr = strarr(ndeltas)

npts = -1

nT2G = 1.d-9*1.d+4
mass = 1.67d-24

for idiff = 0, ndiff-1 do begin

    index1 = ind_pair(idiff,0)
    index2 = ind_pair(idiff,1)
    
; Sonnerup et al., 1981 method *_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*

; Moments quantities
    
    rho1 = mass*density(index1) ; cgs
    rho1_sigma = mass*density_sigma(index1) ; cgs
    rho2 = mass*density(index2) ; cgs
    rho2_sigma = mass*density_sigma(index2) ; cgs
    umom1 = u(0:2,index1)       ; km/s
    umom2 = u(0:2,index2)       ; km/s
    umom1_sigma = u_sigma(0:2,index1) ; km/s
    umom2_sigma = u_sigma(0:2,index2) ; km/s
    pparl1 = pres_parl(index1)  ; cgs
    pparl2 = pres_parl(index2)  ; cgs
    pperp1 = pres_perp(index1)  ; cgs
    pperp2 = pres_perp(index2)  ; cgs
    
; Field quantities
    
    b1 = reform(b(0:2,index1))  ; cgs
    b2 = reform(b(0:2,index2))  ; cgs
    b1_sigma = reform(b_sigma(0:2,index1)) ; cgs
    b2_sigma = reform(b_sigma(0:2,index2)) ; cgs
    
    bmag1 = sqrt(total(b1(0:2)^2,/double)) ; cgs
    bmag2 = sqrt(total(b2(0:2)^2,/double)) ; cgs
    
    FOR icomp=0,2 DO BEGIN
        npts = npts + 1
        y(npts) = umom2(icomp)-umom1(icomp)
        sigy(npts) = sqrt(umom1_sigma(icomp)^2 + umom2_sigma(icomp)^2)
        x(npts) = (b2(icomp)/rho2 - b1(icomp)/rho1)
        sigx(npts) = sqrt( (1.d0/rho1*b1_sigma(icomp))^2 + $
                           (1.d0/rho2*b2_sigma(icomp))^2 + $
                           (b2(icomp)/(rho2^2)*rho2_sigma)^2 + $
                           (b1(icomp)/(rho1^2)*rho1_sigma)^2 )
        labelstr(npts) = compstr(icomp)+string(index1,'(i1)')+string(index2,'(i1)')
        
;        print, $
;          b1(icomp),b2(icomp), $
;          rho1,rho2, $
;          b1(icomp)/rho1,b2(icomp)/rho2, $
;          x(npts),sigx(npts), $
;          format='(8(e8.1,1x))'
        
    ENDFOR
;*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*_*
    
ENDFOR

; Renormalize so that we're finding a slope ~ 1 fit
con = 1.d-5                     ; because y is in km/s (above)
bmag = sqrt(total(b(0:2,inds)^2,1,/double))
arg = mass*density(inds)* $
  (1.d0/(4.d0*!pi)-(pres_parl(inds)-pres_perp(inds))/bmag^2)
gd = where(arg ge 0.d0)
if (gd(0) ne -1) then begin
    ngd = n_elements(gd)
    slp_avg = total(con*sqrt(arg(gd)),/double)/float(ngd)
    if (ngd eq 1) then slp_sig = 0.d0 else $
      slp_sig = sqrt(total((con*sqrt(arg(gd))-slp_avg)^2,/double)/float(ngd-1))
endif else begin
    message, 'Theoretical Slope Not Defined'
endelse

x = x*slp_avg
sigx = sigx*slp_avg
slope_theory = 1.
slope_theory_sigma = slp_sig/slp_avg

nparms = 1
gen_fit_release,x=x,sigma_x=sigx,y=y,sigma_y=sigy,$
  nparms=nparms, $
  errorpercent=.001, $
  slope=slope, $
  intercept=intercept, $
  chisq=chisq, $
  rchisq=rchisq, $
  ks_c68=c68_ks, $
  ks_c95=c95_ks, $
  /force_kappa_search, $
  status=stat, $
  kappa_data=kappa_data

IF (stat NE 1) THEN BEGIN
    get_err_no = 2
    get_err_msg = myname+'gen_fit_release FAILURE'
    return
ENDIF 

IF (slope LT 0) THEN sign = -1 ELSE sign = 1
slope_theory = slope_theory*sign

slopefit68 = [slope-sign*c68_ks(0),slope,slope+sign*c68_ks(1)]
slopefit95 = [slope-sign*c95_ks(0),slope,slope+sign*c95_ks(1)]
slopetheory = [slope_theory-sign*slope_theory_sigma,slope_theory,slope_theory+sign*slope_theory_sigma]
;===========================================================

wfit = { species:0, $
         sf68:fltarr(3), $
         sf95:fltarr(3), $
         st:fltarr(3), $
         r2:0.0, $
         f1:fltarr(100), $      ; 100 because of search space limits
         kappa:fltarr(100), $   ; 100 because of search space limits
         kappamin_index:0L, $
         nfit:0L, $
         xfit:fltarr(500), $    ; 500 is arbitrary
         yfit:fltarr(500), $
         sigyfit:fltarr(500), $
         sigxfit:fltarr(500), $
         labelfit:strarr(500) }

wfit.nfit = n_elements(x)
wfit.f1 = kappa_data.f1
wfit.kappa = kappa_data.kappa
wfit.kappamin_index = kappa_data.kappamin_index
wfit.xfit(0:wfit.nfit-1) = x
wfit.yfit(0:wfit.nfit-1) = y
wfit.sigyfit(0:wfit.nfit-1) = sigy
wfit.sigxfit(0:wfit.nfit-1) = sigx
wfit.labelfit(0:wfit.nfit-1) = labelstr
wfit.sf68 = slopefit68
wfit.sf95 = slopefit95
wfit.st = slopetheory
wfit.r2 = rchisq
if keyword_set(ion) then wfit.species = 1 else wfit.species = -1

return

END
