pro read_hydra_spec_sv, date_str, start_time, end_time, $
                        options=options, $
                        up=up, data_up=data_up, $
                        dn=dn, data_dn=data_dn, $
                        perp=perp, data_perp=data_perp, $
                        all=all, data_all=data_all, $
                        read_all= read_all, file=file

; Read Survey Spectrogram 
; Reads in one day of survey data.
;    date_str 'YYYYMMDD'
;    start_time, end_time in seconds since midnight
;    options is a long bit array:
;       1  spacecraft potential shift
;
; HISTORY:
;    v1.0  June 30, 97    Jeremy Faden    First rev coded and released.
;    v1.3  July 1, 97     Jeremy Faden    Fixed time tag bug -- whoops.
;    v1.4  July 23, 97    JBF             Time tags changed to double,
;                                         float is too dangerous
;                                         hydra_findfile function used
;    v1.5  Sept 2, 97     jbf             more versions checked, and
;                                            min(curr,this) is returned.
;    See RCS information.....
;    conforms with new papco information

  my_name= 'read_hydra_spec_sv'

; handle keywords.
  read_all= keyword_set( read_all )
  if n_elements( options ) eq 0 then options=0L

  common get_error, get_err_no, get_err_msg
  
  if n_elements( file ) eq 0 then file=''

  if strtrim(file,2) eq '' then begin
      f= hydra_findfile( date_str, /survey, version=version )
  endif else begin
      if strpos( file, '.cdf' ) ne -1 then $
        f= file
      message, 'Opening file '+f, /cont
      cdf= cdf_open(f)
      catch, err
      if err eq 0 then $
        cdf_attget, cdf, 'version', 0, version $
      else $
        version=0.1
      catch, /cancel
      cdf_close, cdf      
  endelse

  if version eq 0.0 then begin
      return                    ; hydra_findfile has set the error.
  endif

  if version ge 1.0 then begin
      if version lt 1.20 then begin
          get_err_msg='Hydra Survey file versions less than 1.20 no longer!c'+$
            'supported.  To update your data files, please visit!c '+$
            'Hydra''s data site. '
          get_err_no=1
          return
      endif
      
      if version lt 1.30 and read_all then begin
          get_err_msg= 'Hydra Survey file versions less than 1.30 have !c' + $
            'skew stored in a format that is incompatible with this PaPCo !c'+$
            'modules.  To update your data files, please visit!c '+$
            'Hydra''s data site. '
          get_err_no=1
          return
      endif
  endif
  
  if f eq '' then return

  CDF= cdf_open(f)

  new_scpot= cdf_varexist( cdf, 'SC_POT_PHYS' )
  
  if not new_scpot then begin
      message, 'File contains old space craft potential variable.', /cont
      message, 'This data may contain a 2V offset.', /cont
  endif

  v20format= cdf_varexist( cdf, 'SIGMA_ALIGNED' )
  
  if v20format eq 0 then begin
      message, 'Survey Version less than 2.0.  '+$
        'Skew plots are substandard fidelity.', /cont
  endif      
      
  get_err_no=0

  header_checksum= fix( ( systime(1) ) mod 7919 ) 

  if n_elements( all ) eq 0 then begin
      header= { spec_header_2, $
                $                          
                start_time:start_time, $ ; T90
                end_time:end_time, $ ; T90
                $
                data:'', $      ; a place to put the name of the data variable name
                $               
                nen:fix(0), $
                mode_energy:fltarr(111,50), $
                reload_flag:0, $
                message_flag:0, $
                pitch_range:fltarr(2), $
                time_resolution:float(0), $
                options:options, $
                include_detectors:replicate(1,12), $
                $             
                spec_type_id:0, $
                spec_type_name:' ', $                 
                $
                data_version:float(0), $ ; ddcal version or survey file version
                file1:' ', $    ; input file names
                file2:' ', $
                reader_version:float(0), $
                $                     
                calibration_version:float(0), $
                rgains_version:float(0), $
                igains_version:float(0), $
                glint_mask_version:float(0), $
                background_version:float(0), $
                mfe_calibration_version:float(0), $
                sc_potential_version:float(0), $
                intersweep_version:float(0), $
                $
                header_checksum:header_checksum $
              }
  endif else begin
      header= all
  endelse

  header.file1= f
  header.data_version= version
  header.reader_version= hydra_code_version( /spec )

  hyd_cdf_varget, cdf, 'b_sec', b_sec
  header.time_resolution=b_sec

  cdf_control, cdf, var='MODE', get_var_info=r
  max_rec_count= r.maxrec
  
  hyd_cdf_varget, cdf, 'MODE', mode, rec_count=max_rec_count
  rvalid= where( mode gt 0 )

  if rvalid(0) eq -1 then begin
      get_err_msg=my_name+': survey file contains no data.'
      print, get_err_msg
      get_err_no=1
      return
  endif

  hyd_cdf_varget, cdf, 'TIME', time, rec_count=max_rec_count
  rec_start= ( where( time(rvalid) ge start_time ) )(0)
  if rec_start(0) eq -1 then begin
      get_err_msg= my_name+': No data found after start_time '
      get_err_no= 1
      print, get_err_msg
      return
  endif

  rec_start= rvalid( rec_start )

  rec_end= ( where( time(rvalid) le end_time, count ) )(count-1)
  if rec_end(0) eq -1 then begin
      get_err_msg= my_name+': No data found before end_time '
      get_err_no= 1
      print, get_err_msg
      return
  endif
  rec_end= rvalid( rec_end ) 

  rec_count= ( rec_end - rec_start + 1 ) > 2

  data_values1= { hydra_spec_data, $
                  time:double(0), $ ; seconds since start_time
                  mode:byte(0), $ ; telemetry mode specifies energies
                  spectrum:fltarr(111), $
                  sigma:fltarr(111), $
                  species:byte(0), $ ; 1=electrons 2=ions 3=both
                  sc_pot:float(0), $
                  sc_pot_qual:byte(0), $
                  psi_status:byte(0), $
                  header_checksum:header_checksum $
                }
 
  
  data_values= replicate( data_values1, rec_count )

  hyd_cdf_varget, cdf, 'TIME', time, rec_start=rec_start, rec_count=rec_count
  data_values(*).time= time(*)

; fill holes in time array where there are data dropouts.
  rt= where( data_values(*).time eq 0.0 )
  hyd_cdf_varget, cdf, 'b_sec', bsec
  for i=0,n_elements(rt)-1 do begin
      if rt(i) gt 0 then data_values(rt(i)).time= $
        data_values(rt(i)-1).time+bsec
  endfor
      
  hyd_cdf_varget, cdf, 'MODE_ENERGY', rec_count=50, mode_energy 
  nen= n_elements( mode_energy(*,0) )
  
  r55= indgen(nen) + 55 - (nen/2)

  modes= where( total( abs( mode_energy ), 1 ) gt 0 )
  for i = 0, n_elements( modes )-1 do begin      
      imode= modes(i)
      header.mode_energy(r55,imode)= mode_energy(*,imode)
  endfor

  if new_scpot then begin
      message, 'using "physical potential"', /cont
      hyd_cdf_varget, cdf, 'SC_POT_PHYS', sc_pot, $
        rec_start=rec_start, rec_count=rec_count
  endif else begin
      hyd_cdf_varget, cdf, 'SC_POT', rec_start=rec_start, rec_count=rec_count, $
        sc_pot
      message, 'adding 2V to what is probably the engineering potential', /cont
      sc_pot= sc_pot+2
  endelse
  data_values(*).sc_pot= sc_pot(*)
  
  data_values(*).sc_pot_qual= 1 ; Goddard

  hyd_cdf_varget, cdf, 'MODE', rec_start=rec_start, rec_count=rec_count, $
    mode
  data_values(*).mode= mode(*)

  hyd_cdf_varget, cdf, 'STATUS', rec_start=rec_start, rec_count=rec_count, $
    status
  data_values(*).psi_status= status(*) and 1

  hyd_cdf_varget, cdf, 'GLINT_MASK_VERSION', glint_mask_version
  header.glint_mask_version= $
    min( [ glint_mask_version, header.glint_mask_version ] )
  
  hyd_cdf_varget, cdf, 'CALIBRATION_VERSION', calibration_version
  header.calibration_version= $
    min( [ calibration_version, header.calibration_version ] )

  hyd_cdf_varget, cdf, 'RGAINS_VERSION', rgains_version
  header.rgains_version= $
    min( [ rgains_version, header.rgains_version ] )

  hyd_cdf_varget, cdf, 'IGAINS_VERSION', igains_version
  header.igains_version= $
    min( [ igains_version, header.igains_version ] )

  hyd_cdf_varget, cdf, 'ISWEEP_VERSION', intersweep_version ; rec_start=modei-1
  header.intersweep_version= $
    min( [ intersweep_version, header.intersweep_version ] )

  hyd_cdf_varget, cdf, 'SC_POTENTIAL_VERSION', sc_potential_version
  header.sc_potential_version= $
    min( [ sc_potential_version, header.sc_potential_version ] )
  
  if read_all then begin
      n=3 
      data_up= data_values
      data_perp= data_values
      data_dn= data_values
  endif else n=0
  data_all= temporary( data_values )
  
  for i=0,n do begin
      case i of
          0: zvar_name='SPEC'
          1: zvar_name='SPEC_ALIGNED'
          2: zvar_name='SPEC_PERP'
          3: zvar_name='SPEC_OPPOSED'
      endcase
      hyd_cdf_varget, cdf, zvar_name, rec_start=rec_start, rec_count=rec_count, $
        ispec_data
      r0= where( ispec_data eq -32768 )
      spec_data= make_array( size= size( ispec_data ), /float, value=1e-20 )
      spec_data= 10 ^ ( (float( temporary(ispec_data) )) / 1000 )  
      if r0(0) ne -1 then spec_data(r0)= 0.0
      case i of 
          0: data_all(*).spectrum(r55)= spec_data(*,*)
          1: data_up(*).spectrum(r55)= spec_data(*,*)
          2: data_perp(*).spectrum(r55)= spec_data(*,*)
          3: data_dn(*).spectrum(r55)= spec_data(*,*)
      endcase
  endfor

  if read_all then begin
      if v20format then begin
          for i=1,3 do begin
              case i of
                  1: zvar_name='SIGMA_ALIGNED'
                  2: zvar_name='SIGMA_PERP'
                  3: zvar_name='SIGMA_OPPOSED'
              endcase
              hyd_cdf_varget, cdf, zvar_name, $
                rec_start=rec_start, rec_count=rec_count, $
                ispec_data
              r0= where( ispec_data eq -32768 )
              spec_data= make_array( size= size( ispec_data ), $
                                     /float, value=1e-20 )
              spec_data= 10 ^ ( (float( temporary(ispec_data) )) / 1000 )  
              if r0(0) ne -1 then spec_data(r0)= 0.0
              case i of 
                  1: data_up(*).sigma(r55)= spec_data(*,*)
                  2: data_perp(*).sigma(r55)= spec_data(*,*)
                  3: data_dn(*).sigma(r55)= spec_data(*,*)
              endcase
          endfor    
      endif else begin
          hyd_cdf_varget, cdf, 'SPEC_SKEW', $
            rec_start=rec_start, rec_count=rec_count, ispec_data
          r0= where( ispec_data eq -32768 )
          spec_data= make_array( size= size( ispec_data ), /float, value=1e-20)
          spec_data= 10 ^ ( (float( temporary(ispec_data) )) / 1000 )  
          if r0(0) ne -1 then spec_data(r0)= 0.0
          data_up(*).sigma(r55)= spec_data(*,*)
          data_dn(*).sigma(r55)= 0.0 ; this is true only if we combine them!
      endelse

      up= header
      up.pitch_range= [0,30]
      perp= header
      perp.pitch_range= [75,105]
      dn= header
      dn.pitch_range= [150,180]
  endif
  all= header   
  all.pitch_range= [0,180]

  cdf_close, cdf
  
  return
end


