;+
; NAME: cw_mpress_button
;
;
;
; PURPOSE: like widget_button, except that it tallies multiple presses
;    (eg. double clicks) and sends this number along with the event.
;
; CATEGORY: compound widget
;
;
;
; CALLING SEQUENCE: b= cw_mpress_button(base)
;
;
; 
; INPUTS: base, the parent widget base.
;
;
;
; OPTIONAL INPUTS: none.
;
;
;	
; KEYWORD PARAMETERS: 
;   wait=<float>         time to wait btw clicks, default is 0.25 sec.
;   see widget_button
;
;
; OUTPUTS:  returns event structure
;                   { WIDGET_MPRESS_BUTTON, $
;                     id:event.id, $
;                     top:event.top, $
;                     handler:event.handler, $
;                     select:0, $
;                     npress:widgetData.npress $
;                   }
;
;
; OPTIONAL OUTPUTS: none
;
;
;
; COMMON BLOCKS: none, internal state is stored in first child.
;
;
;
; SIDE EFFECTS: none
;
;
;
; RESTRICTIONS: wait should be no less than 0.1 or so, depending on
;    machine speed.  Don't use "(" in button value, since number of
;    tallys is printed.
;
;
; PROCEDURE: get event
;            if event is button event, then tally the event and 
;                set timer event.
;            if event is timer event, send mpress button event structure 
;               on to caller's event handler. 
;
; EXAMPLE:
;    pro test_it_event, event
;       help, /str, event
;    end

;    pro test_it
;       w= widget_base(title='test it')
;       x= cw_mpress_button(w,value='Press Me')
;       widget_control, w, /realize
;       widget_center, w
;       xmanager, 'test_it', w
;    end
;
;
; MODIFICATION HISTORY:
;   created, October 23, 1998 by Jeremy Faden
;-

;pro test_it_event, event
;   help, /str, event
;end

;pro test_it
;   w= widget_base(title='test it')
;   x= cw_mpress_button(w,value='Press Me')
;   widget_control, w, /realize
;   widget_center, w
;   xmanager, 'test_it', w
;end
   
function cw_mpress_button_event, event
   
   event_type= tag_names( event, /structure_name )

   child= widget_info( event.handler, /child )
   widget_control, child, get_uvalue=widgetData
   
   case event_type of
       'WIDGET_TIMER': begin
           if widgetData.show_tally then begin
               widget_control, widgetData.button_id,get_value=s
               s= str_sep( s, '(' )
               widget_control, widgetData.button_id,set_value=s(0)
           endif
           event1= { WIDGET_MPRESS_BUTTON, $
                     id:event.handler, $
                     top:event.top, $
                     handler:event.handler, $
                     select:0, $
                     npress:widgetData.npress $
                   }
           widgetData.npress= 0
       end       
       'WIDGET_BUTTON': begin
           widgetData.npress= widgetData.npress+1
           widget_control, event.handler, timer= widgetData.wait
           if widgetData.show_tally then begin
               widget_control, widgetData.button_id,get_value=s
               s= str_sep( s, '(' )
               widget_control, widgetData.button_id,set_value=s(0)+'('+$
                 strtrim(widgetData.npress,2)+')'
           endif
           event1=0
       end
   endcase
   
   widget_control, child, set_uvalue=widgetData
   return, event1   
end

function cw_mpress_button, base, $
                           wait=wait, $
                           show_tally=show_tally, $
                           xoffset=xoffset, yoffset=yoffset, $
                           uvalue=uvalue, $
                           _extra=e

   if n_elements( wait ) eq 0 then wait=0.25 ; sec
   if n_elements( xoffset ) eq 0 then xoffset=0.
   if n_elements( yoffset ) eq 0 then yoffset=0.   

   if n_elements( uvalue ) eq 0 then begin
       b= widget_base(base,event_func='cw_mpress_button_event',$
                      xoffset=xoffset, yoffset=yoffset )
   endif else begin
       b= widget_base(base,event_func='cw_mpress_button_event',$
                      xoffset=xoffset, yoffset=yoffset, uvalue=uvalue )
   endelse       

   child= widget_base(b)
  
   widgetData= { button_id: 0L, $
                 npress:0, $
                 wait:0.25, $
                 show_tally:keyword_set(show_tally) $
               }
   
   widgetData.button_id= widget_button( b, _extra=e )
   
   widget_control, child, set_uvalue= widgetData

   return, b
end
   
