function hyd_get_plot_coordinates, X_in, Y_in, _extra=plot_par, $
                                   xstyle=xstyle, ystyle=ystyle, $
                                   ps_dpi=ps_dpi, $
                                   overplot=overplot

;+
; NAME: hyd_get_plot_coordinates
;
;
;
; PURPOSE:
;    get the plot coordinates of the next plot, but preserve 
;    the plot state.  Return information about the plot, such as 
;    coordinates in normal space and pixel space.  Also provide 
;    translation of normal space coordinates into PS pixel coordinates
;    at arbitrary resolution.  A structure with various information is
;    returned.
;
; CATEGORY:
;    color plotting
;
; CALLING SEQUENCE:
;   hyd_get_plot_coordinates, x, y, _extra=e
; 
; OPTIONAL INPUTS:
;   x, y
;	
; KEYWORD PARAMETERS:
;   ps_dpi=x   specify the output resolution of the ps device.
;   /overplot  use existing plot coordinates.
;
; OUTPUTS:
;   Returns a structure with information about the plot.  Such as:
;     device pixel matrix size, x_crange, y_crange
;
; SIDE EFFECTS:
;   None.
;
; RESTRICTIONS:
;
; PROCEDURE:
;   Save plot state, make plot with ystyle and xstyle bit 4 (no draw) 
;   turned on, read in plot statistics, restore plot state.
;
; EXAMPLE:
;   qq= hyd_get_plot_coordinates()
;   help, /str, qq
;
;   set_plot, 'x' 
;   qq= hyd_get_plot_coordinates() 
;   tv= bytarr(qq.x_size_pixel,qq.y_size_pixel) 
;   tv(*)= get_color_index('white') 
;   r= indgen(n_elements(tv(*,0)/2)) * 2 
;   tv(r,*)= get_color_index('black') 
;   tv, tv, qq.x_pos_normal, qq.y_pos_normal, $
;     xsize=qq.x_size_normal, ysize=qq.y_size_normal, /normal
;   
;   set_plot, 'ps' 
;   qq= hyd_get_plot_coordinates(ps_dpi=150) 
;   tv= bytarr(qq.x_size_pixel,qq.y_size_pixel) 
;   tv(*)= get_color_index('white') 
;   r= indgen(n_elements(tv(*,0)/2)) * 2 
;   tv(r,*)= get_color_index('black') 
;   tv, tv, qq.x_pos_normal, qq.y_pos_normal, $
;     xsize=qq.x_size_normal, ysize=qq.y_size_normal, /normal
;   device, /close


; MODIFICATION HISTORY:
;    written, jbf, April 20, 2000
;-

  on_error, 2

;  check inputs 
  if n_elements( ps_dpi ) eq 0 then ps_dpi=75
  overplot= keyword_set( overplot )
  if n_elements( xstyle ) eq 0 then xstyle=!x.style
  if n_elements( ystyle ) eq 0 then ystyle=!y.style
  
  output= { xlog:0L, xrange:[0.,0.], $
            x_pos_inch:0., x_size_inch:0., $
            x_pos_pixel:0L, x_size_pixel:0L, $
            x_pos_normal:0., x_size_normal:0., $
            ylog:0L, yrange:[0.,0.], $
            y_pos_inch:0., y_size_inch:0., $
            y_pos_pixel:0L, y_size_pixel:0L, $            
            y_pos_normal:0., y_size_normal:0.  $            
          }
  
  if n_elements( X_in ) gt 0 then X=X_in else x= fltarr(2)
  if n_elements( Y_in ) gt 0 then Y=Y_in else y= fltarr(2)

; set the x and y ranges
  if not overplot then begin    
      old_x= !x
      old_y= !y
      old_z= !z
      old_p= !p

      plot, X, Y, /nodata, _extra=plot_par, $
        xstyle= xstyle or 4, $
        ystyle= ystyle or 4      
  endif

  xrange=!x.crange & yrange=!y.crange ; ranges in data space
  xlog=!x.type & ylog=!y.type

  if (xlog) then xrange=10^xrange
  if (ylog) then yrange=10^yrange

  output.xrange= xrange
  output.xlog= xlog
  output.yrange= yrange
  output.ylog= ylog
  
; get the device coordinate system
  range0= convert_coord(xrange(0),yrange(0),/data,/to_device)
  range1= convert_coord(xrange(1),yrange(1),/data,/to_device)

  output.x_pos_inch= range0(0) / !d.x_px_cm / 2.54
  output.y_pos_inch= range0(1) / !d.y_px_cm / 2.54

  output.x_size_inch= ( range1(0)-range0(0) ) / !d.x_px_cm / 2.54
  output.y_size_inch= ( range1(1)-range0(1) ) / !d.y_px_cm / 2.54

  nrange0= convert_coord(xrange(0),yrange(0),/data,/to_normal)
  nrange1= convert_coord(xrange(1),yrange(1),/data,/to_normal)

  output.x_pos_normal= nrange0(0)
  output.y_pos_normal= nrange0(1)

  output.x_size_normal= ( nrange1(0)-nrange0(0) )
  output.y_size_normal= ( nrange1(1)-nrange0(1) )
  
; find device size
  if !d.name eq 'PS' then begin
      output.x_size_pixel= output.x_size_inch * ps_dpi
      output.y_size_pixel= output.y_size_inch * ps_dpi
      output.x_pos_pixel= output.x_size_pixel * ps_dpi
      output.y_pos_pixel= output.x_size_pixel * ps_dpi
  endif else begin
      output.x_size_pixel= long( range1(0)-range0(0) + 1 + 0.5 )
      output.y_size_pixel= long( range1(1)-range0(1) + 1 + 0.5 )
      output.x_pos_pixel= long( range0(0) + 0.5 )
      output.y_pos_pixel= long( range0(1) + 0.5 )
  endelse

  if not overplot then begin
      !x= old_x
      !y= old_y
      !z= old_z
      !p= old_p
  endif
  
  return, output
end

