function hydra_average, X, navg, nodata=nodata, naverage, noavg=noavg, $
                        allow_smooth=allow_smooth

; averages array over first dimension (having n elements) down to
; navg elements.

;    x1 x2 x3 x4 x5 x6 x7 x8        <-- original 1D (or 2D array), nX elements
;    \      / \   / \      /        <-- (nX/navg) or (nX/navg+1)
;     \    /   \ /   \    /              averaged together
;       a1     a2      a3           <-- averaged array, navg elements
;
; INPUT:  X      is the array to average
;         navg   is the number of average bins
;         nodata specifies nodata value
;         noavg  don't divide by number of points, just sum
;
;         allow_smooth  (nX/navg+1) values go into each average, so
;                       some values in source array to go into
;                       two averages.  This is somewhat faster.
;
; OUTPUT: naverage   passes out the number of values in average (optional)
;         (returns)  the average array, or if /noavg, the sums, sum(Xi),
;                      to be divided by naverage
;
; ASSUMPTIONS and NOTES:
;  For 2d arrays, ANY value equal to 1e-20 is also treated as nodata.
;  It is also assumed that 1e-20 << X(i)*nsum .
;  Occasionally this routine triggers floating underflow.
;
; HISTORY:
;  Spring 1997   Written, Jeremy Faden, jbf@hydra.physics.uiowa.edu

   if n_elements(nodata) eq 0 then nodata= 1e-20
   allow_smooth= keyword_set(allow_smooth)

   sizeX= size(X)
   n= sizeX( 1 )                ; average over the first dimension
   nsum= float(n)/navg
   
   if (nsum eq 1.0) then begin  ; no averaging
       naverage= make_array( size=size(X), /int, value=1 )
       rnv= where( X eq nodata )
       if rnv(0) ne -1 then naverage(rnv)=0
       return, X
   endif 

   if (nsum lt 1.0 ) then begin ; use congrid to expand       
       print, '% hydra average: I don''t expand'
       return, X
   endif
   
   if ( sizeX(0) lt 1 or sizeX(0) gt 2) then begin
       print, '% hydra average: only 1- or 2-D arrays please...'
       return, -1
   endif else if sizeX(0) eq 1 then begin ; 1D average
       typeX= sizeX(2)
       saverage= make_array(navg,type=typeX,value=0)
       naverage= make_array(navg,/int)
   endif else begin             ; 2D average
       typeX= sizeX(3)
       ncol= sizeX(2)
       saverage= make_array(navg,ncol,type=typeX)
       naverage= make_array(navg,ncol,/int)
   endelse     
   
   if (allow_smooth) then begin
       print, '% hydra_average: smoothing allowed.'
       nsum1=ceil(nsum) 
   endif else nsum1=fix(nsum)-1

   ind= long(indgen(navg)*nsum)
   ind0= [ind,n]                ; initial indeces

   if sizeX(0) eq 1 then begin
       for i=0, nsum1 do begin
           rv= where(X(ind) ne nodata)
           if (rv(0) ne -1) then begin
               saverage(rv)=saverage(rv)+X(ind(rv))
               naverage(rv)=naverage(rv)+1
           endif
           ind=ind+1
       endfor

       if (not allow_smooth) then begin
           r= where( ind0(1:*)-ind(0:*) eq 1 ) 
           if r(0) ne -1 then begin
               ind= ind(r)
               rv= where(X(ind) ne nodata)
               if (rv(0) ne -1) then begin
                   saverage(r(rv))=saverage(r(rv))+X(ind(rv))
                   naverage(r(rv))=naverage(r(rv))+1
               endif
           endif
       endif
   endif else begin
       if nodata ne 1e-20 then begin
           rnodata=where(X eq nodata) 
           if rnodata(0) ne -1 then X(rnodata)=1e-20
       endif
           
       for i=0, nsum1-1 do begin
           rv= where(X(ind,*) ne 1e-20)
           if (rv(0) ne -1) then begin
               saverage= saverage+X(ind,*)
               naverage(rv)=naverage(rv)+1
               rz=where(saverage eq 1e-20)
               if rz(0) ne -1 then saverage(rz)=0.0
           endif
           ind=ind+1    
       endfor

       if (not allow_smooth) then begin ; average in the remaining guys
           r= where( ind0(1:*)-ind(0:*) eq 1 ) 
           if r(0) ne -1 then begin
               ind= ind(r)
               rv= where(X(ind,*) ne 1e-20)
               if (rv(0) ne -1) then begin
                   saverage(r,*)= saverage(r,*)+X(ind,*)
                   naverage(r(rv),*)=naverage(r(rv),*)+1
                   rz=where(saverage eq 1e-20) 
                   if rz(0) ne -1 then saverage(rz)=0.0
               endif
           endif
       endif

       if nodata ne 1e-20 then begin
           if rnodata(0) ne -1 then X(rnodata)=nodata
       endif

   endelse

   r0= where(naverage eq 0)
   if (r0(0) ne -1) then saverage(r0)= nodata
   
   if keyword_set( noavg ) then return, saverage
   
   rv= where(naverage gt 0)
   if (rv(0) ne -1) then saverage(rv)=saverage(rv)/naverage(rv)
   
   return, saverage
end

