function hyd_cdrom_findfile, date_str_in, $
                             survey=survey, $
                             level1=level1, $
                             ddcal=ddcal, $
                             moments_all=moments, $
                             ppal1=ppal1, $
                             version=version, count=count, $
                             no_report_index=no_report_index

;   a special findfile designed to use the hydra l1 cds.

;   /no_report_index means do not report where the file can be found.

    indir= getenv('CDROM')
    index_dir= getenv('HYDRA_CDROM_INDEX')

    if indir eq '' then begin
        message, 'CDROM location not specified.'
        return, ''
    endif

    date_str=strtrim( date_str_in, 2 )
    if strlen(date_str) eq 8 then date_str=strmid(date_str,2,6)

    if !version.os_family eq 'Windows' then begin
        dir_sep='\'             ; '
    endif else begin
        dir_sep='/'
    endelse
    
    if strmid( indir, strlen(indir)-1, 1 ) ne dir_sep then $
      indir=indir+dir_sep
    
    if strmid( index_dir, strlen(index_dir)-1, 1 ) ne dir_sep then $
      index_dir=index_dir+dir_sep

    if index_dir eq '' then $
      index_dir= indir+'index' + dir_sep
    
    case (1) of
        keyword_set(level1): begin
            index='l1.ind'
            file='dd'+date_str+'.cdf'
        end
        keyword_set(survey): begin
            index='sv.ind'
            file='sv'+date_str+'.cdf'
        end
        keyword_set(ddcal): begin
            index='l1cal.ind'
            file='dc'+date_str+'.cdf'
        end
        keyword_set(moments): begin
            index='mom.ind'
            file='me'+date_str+'.cdf'
        end
        keyword_set(ppal1): begin
            index='ppa.ind'
            file='pp'+date_str+'.cdf'
        end
    endcase
    
    count=0
    
    tryagain:
    volume=''
    on_ioerror, ioerror
    openr, lun, indir+'label', /get_lun
    volume=''
    readf, lun, volume
    close, lun
    goto, cont1
    
    ioerror:
    message, 'CDROM not loaded or read error.', /continue

    cont1:
    on_ioerror, no_index
    openr, lun, index_dir + index, /get_lun
    goto, cont2

    no_index:
    message, 'CDROM index not found.', /continue
    return, ''
    
    cont2:
    found=0
    s=''
    loc=['dummy']
    ver=[0.00]
    while not eof(lun) do begin
        readf, lun, s        
        s= str_sep(s,';') ; remove comments
        s= strtrim(s(0),2)
        if strmid(s,0,12) eq file then begin
            s= (str_sep(s,' '))(1)
            loc= [loc,s]
            n= strpos(s,'sv_v')
            v= float(strmid(s,n+4,5))
            ver= [ver,v]
            found=1
        endif
    endwhile
    close, lun    

    if not found then begin
        message, 'date not found in CDROM index.', /cont
        return, ''
    endif

    loc=loc(1:*)
    ver=ver(1:*)
    vollist=['']

    loaded=0
    for i=0,n_elements(loc)-1 do begin
        loc1= str_sep( loc(i),'/' )
        vollist= [vollist,'   '+loc1(0)+'  '+string(ver(i),format='(f5.2)')]
        if volume eq loc1(0) then begin
            loaded=1
            loaded_ver=ver(i)
            loaded_loc=loc1
        endif
    endfor
    vollist=vollist(1:*)

    if n_elements(no_report_index) gt 0 and no_report_index ge max(ver) then $
      return, ''

    if loaded then begin
        if loaded_ver lt max(ver) then begin
            message=[ 'Loaded cdrom does not contain the ', $
                      'best file version avaiable on CDROM.', $
                      'The following cdroms contain versions ', $
                      'more recent than '+$
                      strtrim(string(loaded_ver,form='(f5.2)'),2)+':', '']
            r= where( ver gt loaded_ver )
            message=[message,vollist(r),'']
            message=[message, 'Do you want to load a different volume?', $
                     'Yes, try new volume  ',$
                     'No, continue using this volume' ]
            r= widget_message( message, /question,title='Inferior Volume' )
            if r eq 'Yes' then goto, tryagain
        endif
    endif

    if not loaded then begin
        if volume eq '' then message= 'Hydra cdrom not loaded.' else $
          message= 'File is not on volume '+volume+'.'
        message=[ message, $
                  'Insert one of the following: ', '', $
                  vollist, '', $
                  'Yes, try new volume  ',$
                  'No, abort cdrom search.' ]
        r= widget_message( message, /question, title='Wrong Volume' )
        if r eq 'No' then return, '' $          
        else goto, tryagain
    endif
    
    loc= loaded_loc(1)          ; omit the label part.
    for i=2,n_elements(loaded_loc)-1 do $
      loc= loc + dir_sep + loaded_loc(i)
    
    ff= indir + loc + dir_sep + file
    r= findfile( ff, count=count )
    if count eq 0 then begin
        version=0.00
        return,''
    endif else begin
        n= strlen(loc)
        version= float(strmid(loc,n-4,4))
        count=1
        return, indir + loc + dir_sep + file        
    endelse
end


function hydra_net_findfile, date_str_in, version=version, $
                             survey=survey, $
                             level1=level1, $
                             ddcal=ddcal
                             
;  reads network index file and returns the name of the 
;  best product available.

   prod_code= keyword_set(survey)*1+keyword_set(level1)*2+$
     keyword_set(ddcal)*3
   survey= 1
   level1= 2
   ddcal=  3

   date_str= strtrim( date_str_in, 2 )
   if strlen( date_str ) eq 6 then date_str='19'+date_str

   wget_exe= getenv('wget_exe')
   if wget_exe eq '' then begin
       message, 'wget_exe environment variable not set', /cont
       version=0.
       return, ''
   endif
   if wget_exe eq '0' then begin
       message, 'wget turned off via wget_exe=0', /cont
       version=0.
       return, ''
   endif
   ftp_site= getenv( 'HYDRA_FTP_ARCHIVE' )
   if ftp_site eq '' then begin
       message, 'HYDRA_FTP_ARCHIVE environment variable not set', /cont
       version=0
       return, ''
   endif   
   
   n= strlen( ftp_site )
   if strmid( ftp_site, n-1, 1 ) ne '/' then ftp_site=ftp_site+'/'

   index_remote= ftp_site+'net_index_00'

   exe_str= wget_exe+' -T20 -q -O - '+index_remote
   message, 'reading index file...', /cont
   spawn, exe_str, lun
   
   case prod_code of 
       survey: prod_str= 'sv'
       level1: prod_str= 'dd'
       ddcal:  prod_str= 'ddcal'
       else: begin
           message, 'unrecognized file type.'
           return, ''
       end
   end

   bestfile=''
   bestver=0.0
   s=''
   for i=0, n_elements( lun )-1 do begin
       s=lun(i)
       parse= str_sep( s, ' ' )
       if parse(0) eq date_str then begin
           if parse(1) eq prod_str then begin
               ver= parse(2)
               if ver gt bestver then begin
                   bestver=ver
                   bestfile= parse(3)
               endif
           endif
       endif
   endfor
   
   if bestver gt 0.0 then begin
       version= bestver
       return, ftp_site+bestfile
   endif else begin
       message, 'file not found in index.', /cont
       version=0.0
       return, ''
   endelse

end



function hydra_findfile, date_str_in, $
                         CDHF=CDHF, $
                         ppal1=ppal1, $
                         footprint=footprint, $
                         moments_all=moments,moments_rc_ele=moments_rc_ele, $
                         moments_rc_ion=moments_rc_ion, $
                         moments_ion=moments_ion, $
                         lrkp=lrkp,hrkp=hrkp, $
                         spp_ions=spp_ions, $
                         kp_extra=kp_extra, kptest=kptest, tpp_ions=tpp_ion, $
                         hrkptemp=hrkptemp, attitude=attitude, $
                         level1=level1, survey=survey, ddcal=ddcal, $
                         version=version, silent=silent, $
                         path=path, directory=dir, tpphr=tpphr, $
                         getversion=getversion

; given date and data set name, return the fully-qualified name of the most
; current version of the data set.
;  IO:
;    date     (in)  Day to find, 'YYYYMMDD'
;    /ppal1   (in)  Find ppa level one file
;    /level1  (in)  Find level one file
;    /ddcal   (in)  Find level one calibration file
;    /survey  (in)  Find survey file
;    /lrkp    (in)  Find low-resolution key parameter file
;    /hrkp    (in)  Find high-resolution key parameter file
;    /kp_extra (in) Find key parameter extra file
;    /moments_all  (in) Find electron moments file
;    /moments_ion  (in) Find ion moments file
;    /moments_rc_ele (in) Find the subset moments file for electrons
;    /moments_rc_ion (in) Find the subset moments file for ions
;    /attitude (in) Find the level0 attitude file
;    /silent  (in)  Don't print messages.
;    path     (in)  use this searchpath instead of environment variables
;    version  (out) Version of the file found.
;   <returns> (out) Fully qualified name, or '' if none found.
;  Side Effects:
;    modifies the get_error common block with appropriate message.

  common get_error, get_err_no, get_err_msg

  myname= '% hydra_findfile'
  
  IF (keyword_set(getversion)) THEN getversion = getversion ELSE getversion=''

  if keyword_set( tpp_ion ) then kp_extra=1

  if !version.os_family eq 'Windows' then begin
      dir_sep='\'
      path_sep=';'
  endif else begin
      dir_sep='/'
      path_sep=':'
  endelse

  date_str= strtrim( date_str_in,2 )
  if strlen(date_str) le 6 then begin
      idate= long( date_str )
      if idate lt 10000 then idate= idate+20000000 else idate=idate+19000000
      date_str= strtrim( idate, 2 )
  endif

; -------------------------------------------------------------------
  if keyword_set( level1 ) then begin
      lookagain=0
      lookagain_level1:

      if not keyword_set(path) then path= getenv( 'HYDRA_LEVEL1_DATA_PATH' )
      if path eq '' then begin
          print, myname+$
            ': environment variable HYDRA_LEVEL1_DATA_PATH not set'
          version=0.0
          return, ''
      endif
      l1path= str_sep( path, path_sep )

;     search input path for highest version
      bestver=0.0 & bestfile='' & bestdir=''

      for i=0,n_elements(l1path)-1 do begin

          indir=strtrim(l1path(i),2)
          IF strmid(indir,strlen(indir)-1,1) NE dir_sep THEN $
            indir=indir+dir_sep

          if strmid(indir,0,5) eq 'CDROM' then begin
              ff= hyd_cdrom_findfile( date_str, /level1, $
                                      version=l1version, count=count, $
                                      no_report_index=bestver )
          endif else begin              
              ff=findfile( indir+date_str+'_hyd_dd_v?.??.cdf', count=count )
              if count eq 0 then begin
                  fz= findfile( indir+date_str+'_hyd_dd_v?.??.cdf.gz' )
                  if fz(0) ne '' then begin
                      message, 'decompressing '+fz(0), /cont
                      spawn, 'gunzip '+fz(0)
                      ff=findfile( indir+date_str+'_hyd_dd_v?.??.cdf', $
                                   count=count )
                  endif
              endif
          endelse
          
          if (count ne 0) then begin
              for i=0, count-1 do begin
                  n= strlen( ff(i) )
                  l1version=float(strmid(ff(i),n-8,4))
                  if l1version gt bestver then begin
                      bestver=l1version
                      bestfile=ff(i)
                      bestdir=indir
                  endif
              endfor
          endif 
      endfor

      if bestver eq 0.0 then begin
          if lookagain eq 0 then begin
              f= hydra_retrieve( date_str, /level1, errmsg=net_errmsg )
              lookagain=1
              goto, lookagain_level1
          endif
          get_err_msg= myname+'Hydra Level 1 file not found on path:'
          for i=0,n_elements(l1path)-1 do get_err_msg=get_err_msg+$
            '!c'+l1path(i)
          get_err_msg=get_err_msg+'!c'+net_errmsg
          if not keyword_set(silent) then print, get_err_msg
          get_err_no=1
          version=0
          return, ''
      endif else begin
          if not keyword_set(silent) then print, myname+': found '+$
            bestfile+'...'
          n=strlen(bestfile)
          version=bestver
          dir= bestdir
          get_err_no=0          ; tell papco all is well
          return, bestfile
      endelse
  endif


; -------------------------------------------------------------------
  if keyword_set( ddcal ) OR keyword_set( hrkp ) then begin
      lookagain=0
      lookagain_ddcal:      
      if not keyword_set(path) then $
        path= getenv( 'HYDRA_LEVEL1CAL_DATA_PATH' )
      if path eq '' then begin
          path= getenv( 'HYDRA_LEVEL1_DATA_PATH' )
          if path eq '' then begin
              print, myname+': environment variable HYDRA_LEVEL1_DATA_PATH '+ $
                'not set.'
              version=0.0
              return, ''
          endif else begin
              print, myname+$
                ': Warning: environment variable HYDRA_LEVEL1CAL_DATA_PATH '
              print, 'not set; using HYDRA_LEVEL1_PATH.'
          endelse
      endif
      l1path= str_sep( path, path_sep )

;     search input path for highest version
      bestver=0.0 & bestfile='' & bestdir=''

      for i=0,n_elements(l1path)-1 do begin

          indir=strtrim(l1path(i),2)
          IF strmid(indir,strlen(indir)-1,1) NE dir_sep THEN $
            indir=indir+dir_sep

          if strmid(indir,0,5) eq 'CDROM' then begin
              ff= hyd_cdrom_findfile( date_str, /ddcal, $
                                      version=l1version, count=count, $
                                      no_report_index=bestver )
          endif else begin
              ff=findfile( indir+date_str+'_hyd_ddcal_v?.??.cdf', count=count )
              if count eq 0 then begin
                  fz= findfile( indir+date_str+'_hyd_ddcal_v?.??.cdf.gz' )
                  if fz(0) ne '' then begin
                      message, 'decompressing '+fz(0), /cont
                      spawn, 'gunzip '+fz(0)
                      ff=findfile( indir+date_str+'_hyd_ddcal_v?.??.cdf', $
                                   count=count )
                  endif
              endif
          endelse
          
          if (count ne 0) then begin
              for i=0,count-1 do begin
                  n= strlen( ff(count-1) )
                  l1version=float(strmid(ff(i),n-8,4))
                  if l1version gt bestver then begin
                      bestver=l1version
                      bestfile=ff(i)
                      bestdir=indir
                  endif
              endfor
          endif else begin      ; second chance: look for l1 data file
              ff=findfile( indir+date_str+'_hyd_dd_v?.??.cdf', count=count )
              if count eq 0 then begin
                  fz= findfile( indir+date_str+'_hyd_dd_v?.??.cdf.gz' )
                  if fz(0) ne '' then begin
                      message, 'decompressing '+fz(0), /cont
                      spawn, 'gunzip '+fz(0)
                      ff=findfile( indir+date_str+'_hyd_dd_v?.??.cdf', $
                                   count=count )
                  endif
              endif
              if (count ne 0) then begin
                  for i=0,count-1 do begin
                      n=strlen(ff(i))
                      l1version=float(strmid(ff(i),n-8,4))
                      if l1version gt bestver then begin
                          bestver=l1version
                          bestfile=ff(i)
                          bestdir=indir
                      endif
                  endfor
              endif
          endelse
      endfor

      if bestver eq 0.0 or ( bestver ge 1.00 and bestver lt 1.60 ) then begin
          if lookagain eq 0 then begin
              f= hydra_retrieve( date_str, /ddcal, errmsg=net_errmsg )
              lookagain=1
              goto, lookagain_ddcal
          endif
          get_err_msg= 'Hydra Level 1 calibration file not found on path:'
          for i=0,n_elements(l1path)-1 do get_err_msg=get_err_msg+$
            '!c'+l1path(i)
          get_err_msg=get_err_msg+'!c'+net_errmsg
          if not keyword_set(silent) then print, get_err_msg
          get_err_no=1
          version=0
          return, ''
      endif else begin
          if not keyword_set(silent) then print, myname+': found '+$
            bestfile+'...'
          n=strlen(bestfile)
          version=bestver
          dir= bestdir
          get_err_no=0          ; tell papco all is well
          return, bestfile
      endelse
  endif

; -------------------------------------------------------------------
  if ( keyword_set( survey ) OR keyword_set( lrkp ) ) then BEGIN
      lookagain=0
      lookagain_survey:

      IF NOT keyword_set(path) THEN path1= getenv('HYDRA_SURVEY_DATA_PATH') $
      ELSE path1 = path

      path1= str_sep( path1, path_sep )

      bestver=0. & bestfile='' & bestdir=''

      for i= 0, n_elements( path1 )-1 do begin
          indir= strtrim( path1(i),2 )
          IF strmid(indir,strlen(indir)-1,1) NE dir_sep THEN $
            indir=indir+dir_sep
          
          if strmid(indir,0,5) eq 'CDROM' then begin
              ff= hyd_cdrom_findfile( date_str, /survey, $
                                      version=version, count=count, $
                                      no_report_index=bestver )
          endif else begin
              ff=findfile( indir+date_str+'_hyd_sv_v?.??.cdf', count=count )
              if count eq 0 then begin
                  fz= findfile( indir+date_str+'_hyd_sv_v?.??.cdf.gz' )
                  if fz(0) ne '' then begin
                      message, 'decompressing '+fz(0), /cont
                      spawn, 'gunzip '+fz(0)
                      ff=findfile( indir+date_str+'_hyd_sv_v?.??.cdf', $
                                   count=count )
                  endif
              endif
          endelse
          
          data_dir=indir
          if (count ne 0) then begin
              for i=0,count-1 do begin
                  n= strlen(ff(i))
                  version=float(strmid(ff(i),n-8,4))
                  if version gt bestver then begin
                      bestfile= ff(i)
                      bestdir= data_dir
                      bestver= version
                  endif
              endfor
          endif
      endfor
      
      if bestfile ne '' then begin
          if not keyword_set(silent) then print, myname+': found '+bestfile+'...'
          get_err_no=0
          dir= data_dir
          version=bestver
          return, bestfile
      endif else begin
          if lookagain eq 0 then begin
              f= hydra_retrieve( date_str, /survey, errmsg=net_errmsg )
              lookagain=1
              goto, lookagain_survey
          endif          
          get_err_msg= 'Hydra Survey file not found on path:'
          for i=0,n_elements(path1)-1 do get_err_msg=get_err_msg+$
            '!c'+path1(i)
          if not keyword_set(silent) then $
            print, myname+': Hydra Survey/LRKP file, '+ $
            data_dir+date_str+'_hyd_sv_v?.??.cdf, '+ $
            'not found.'
          get_err_no=1
          version=0
          return, ''
      endelse
  ENDIF
  
; --------------------------------------------------------------------
  if keyword_set( hrkptemp ) then BEGIN
      
      data_dir = '/home/ppq/bigdisk/key_parameters/hr/DDCAL/'
      
      IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep
      
       fstring = data_dir+'po_hrkp_hyd_'+date_str+'_v'

       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          fnstr = fstring+getversion+'.cdf'
          f= findfile( fnstr, count=count )
       endif else BEGIN
          fnstr = fstring+'??.cdf'
          f= findfile( fnstr, count=count )
       ENDELSE

       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-6, 2 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra HRKPtemp file,!c'+fnstr+'!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra HRKPtemp file, '+fnstr+' not found.'
          get_err_no=1
          version=0
          return, ''
       ENDELSE

    ENDIF

; --------------------------------------------------------------------
    if keyword_set( moments ) then BEGIN

       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_MOMENTS_DATA_PATH') $
       ELSE data_dir = path

       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep

       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          f= findfile( data_dir+date_str+'_hyd_mom_v'+getversion+'.cdf', $
                       count=count )
       endif else begin
          f= findfile( data_dir+date_str+'_hyd_mom_v?.??.cdf', count=count )
       ENDELSE

       if (count ne 0) then BEGIN

;          print, 'count: ',count

; Pick the highest version number
          version_max = 0.00
          FOR ifile=0,count-1 DO BEGIN
             nlen = strlen(f(ifile))
             version = float( strmid(f(ifile),nlen-8,4) )
;             print, 'version: ',version
             IF (version GT version_max) THEN BEGIN
                imax = ifile
                version_max = version
;                print, 'version_max: ',version_max
             ENDIF
          ENDFOR 
          f=f(imax)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-8, 4 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra Moments file,!c'+ $
           data_dir+date_str+'_hyd_mom_v?.??.cdf,!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra Moments file, '+ $
           data_dir+date_str+'_hyd_mom_v?.??.cdf, '+ $
           'not found.'
          get_err_no=1
          version=0
          return, ''
       endelse
    ENDIF

; --------------------------------------------------------------------
    if keyword_set( moments_ion ) then BEGIN

       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_MOMENTS_DATA_PATH') $
       ELSE data_dir = path

       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep

       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          f= findfile( data_dir+date_str+'_hyd_momi_v'+getversion+'.cdf', $
                       count=count )
       endif else begin
          f= findfile( data_dir+date_str+'_hyd_momi_v?.??.cdf', count=count )
       ENDELSE

       if (count ne 0) then BEGIN

;          print, 'count: ',count

; Pick the highest version number
          version_max = 0.00
          FOR ifile=0,count-1 DO BEGIN
             nlen = strlen(f(ifile))
             version = float( strmid(f(ifile),nlen-8,4) )
;             print, 'version: ',version
             IF (version GT version_max) THEN BEGIN
                imax = ifile
                version_max = version
;                print, 'version_max: ',version_max
             ENDIF
          ENDFOR 
          f=f(imax)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-8, 4 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra Ion Moments file,!c'+ $
           data_dir+date_str+'_hyd_momi_v?.??.cdf,!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra Ion Moments file, '+ $
           data_dir+date_str+'_hyd_momi_v?.??.cdf, '+ $
           'not found.'
          get_err_no=1
          version=0
          return, ''
       endelse
    ENDIF

; --------------------------------------------------------------------
    if keyword_set( moments_rc_ele ) then BEGIN

       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_MOMENTS_RC_DATA_PATH') $
       ELSE data_dir = path

       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep
       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          f= findfile( data_dir+date_str+'_hyd_momrc_v'+getversion+'.cdf', $
                       count=count )
       endif else begin
          f= findfile( data_dir+date_str+'_hyd_momrc_v?.??.cdf', count=count )
       ENDELSE

       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-8, 4 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra Moments file,!c'+ $
           data_dir+date_str+'_hyd_momrc_v?.??.cdf,!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra Moments file, '+ $
           data_dir+date_str+'_hyd_momrc_v?.??.cdf, '+ $
           'not found.'
          get_err_no=1
          version=0
          return, ''
       endelse
    ENDIF

; --------------------------------------------------------------------
    if keyword_set( moments_rc_ion ) then BEGIN

       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_MOMENTS_RC_DATA_PATH') $
       ELSE data_dir = path

       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep
       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          f= findfile( data_dir+date_str+'_hyd_momirc_v'+getversion+'.cdf', $
                       count=count )
       endif else begin
          f= findfile( data_dir+date_str+'_hyd_momirc_v?.??.cdf', count=count )
       ENDELSE

       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-8, 4 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra Moments file,!c'+ $
           data_dir+date_str+'_hyd_momirc_v?.??.cdf,!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra Moments file, '+ $
           data_dir+date_str+'_hyd_momirc_v?.??.cdf, '+ $
           'not found.'
          get_err_no=1
          version=0
          return, ''
       endelse
    ENDIF

; --------------------------------------------------------------------
    if keyword_set( kp_extra ) then BEGIN

;       data_dir = '/home/ppq/bigdisk/key_parameters/ion_kps/byblock/data/'

       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_TPPION_DATA_PATH') ELSE data_dir = path

       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep

       fstring = data_dir+date_str+'_hyd_extra_v'

       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          fnstr = fstring+getversion+'.cdf'
          f= findfile( fnstr, count=count )
       endif else BEGIN
          fnstr = fstring+'?.??.cdf'
          f= findfile( fnstr, count=count )
       ENDELSE

       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-8, 4 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra Ion Extra file,!c'+fnstr+'!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra Ion Extra file, '+fnstr+' not found.'
          get_err_no=1
          version=0
          return, ''
       ENDELSE

    ENDIF

; --------------------------------------------------------------------
    if keyword_set( tpphr ) then BEGIN

;       data_dir = '/home/ppq/bigdisk/key_parameters/ion_kps/ten_parameter_patch/highres/release/'

       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_TPPION_DATA_PATH') ELSE data_dir = path

       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep

       fstring = data_dir+date_str+'_hyd_tpphr_'+string(tpphr,'(i1)')+'_v'
       
       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          fnstr = fstring+getversion+'.cdf'
          f= findfile( fnstr, count=count )
       endif else BEGIN
          fnstr = fstring+'?.??.cdf'
          f= findfile( fnstr, count=count )
       ENDELSE
       
       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-8, 4 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra Ion TPPHR file,!c'+fnstr+'!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra Ion TPPHR, '+fnstr+' not found.'
          get_err_no=1
          version=0
          return, ''
       ENDELSE

    ENDIF
; --------------------------------------------------------------------
    if keyword_set( spp_ions ) then BEGIN

       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_SPPION_DATA_PATH') ELSE data_dir = path

;       print, data_dir

       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep

       fstring = data_dir+date_str+'_hyd_sppions_v'

       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          fnstr = fstring+getversion+'.cdf'
          f= findfile( fnstr, count=count )
       endif else BEGIN
          fnstr = fstring+'?.??.cdf'
          f= findfile( fnstr, count=count )
       ENDELSE

       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-8, 4 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra SPP Ion file,!c'+fnstr+'!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra SPP Ion file, '+fnstr+' not found.'
          get_err_no=1
          version=0
          return, ''
       ENDELSE

    ENDIF


    if keyword_set( attitude ) then BEGIN

       data_dir='/opt/hydra/lz?/'
       filename = 'po_at_def_'+date_str+'_v??.cdf'

       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep
       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          f= findfile( data_dir+'po_at_def_'+date_str+'_v'+getversion+'.cdf', $
                       count=count )
       endif else begin
          f= findfile( data_dir+filename, count=count )
       ENDELSE

       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-6, 2 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Polar Attitude file,!c'+ $
           data_dir+filename+',!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Polar Attitude file, '+ $
           data_dir+filename+' not found.'
          get_err_no=1
          version=0
          return, ''
       endelse
    ENDIF

    if keyword_set( kptest ) then BEGIN

;       data_dir = '/home/ppq/bigdisk/key_parameters/test_kplib/'

       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_KPTEST_DATA_PATH') $
       ELSE data_dir = path

       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep

       fstring = data_dir+date_str+'_hyd_kpall_v'

       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          fnstr = fstring+getversion+'.cdf'
          f= findfile( fnstr, count=count )
       endif else BEGIN
          fnstr = fstring+'?.??.cdf'
          f= findfile( fnstr, count=count )
       ENDELSE

       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-8, 4 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra KPtest file,!c'+fnstr+'!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra KPtest file, '+fnstr+' not found.'
          get_err_no=1
          version=0
          return, ''
       ENDELSE

    ENDIF

    if keyword_set( footprint ) then BEGIN
       
       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_FOOTPRINT_DATA_PATH') $
       ELSE data_dir = path
       
       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep
       
       f= findfile( data_dir+date_str+'_footprint.cdf', count=count )

       IF (count ne 0) THEN BEGIN 
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          get_err_no=0
          dir= data_dir
          return, f
       ENDIF ELSE BEGIN 
          get_err_msg= myname+': Hydra Footprint file,!c'+ $
           data_dir+date_str+'_footprint.cdf,!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra Footprint file, '+ $
           data_dir+date_str+'_footprint.cdf, '+ $
           'not found.'
          get_err_no=1
          return, ''
       ENDELSE

    ENDIF

    if keyword_set( CDHF ) then BEGIN

       IF NOT keyword_set(path) THEN data_dir= getenv('HYDRA_CDHFKP_DATA_PATH') $
       ELSE data_dir = path
       
       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep

       fstring = data_dir+'po_k0_hyd_'+date_str+'_v'

       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          fnstr = fstring+getversion+'.cdf'
          f= findfile( fnstr, count=count )
       endif else BEGIN
          fnstr = fstring+'??.cdf'
          f= findfile( fnstr, count=count )
       ENDELSE

       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-6, 2 ))
          get_err_no=0
          dir= data_dir
          return, f
       endif else begin
          get_err_msg= myname+': Hydra CDHF KP file,!c'+fnstr+'!c'+ $
           'not found.!c'
          if not keyword_set(silent) then $
           print, myname+': Hydra CDHF KP file, '+fnstr+' not found.'
          get_err_no=1
          version=0
          return, ''
       ENDELSE

    ENDIF

; --------------------------------------------------------------------
    if keyword_set( ppal1 ) then BEGIN

       IF NOT keyword_set(path) THEN $
         data_dir= getenv('HYDRA_PPAL1_DATA_PATH') ELSE data_dir = path
       
       IF strmid(data_dir,strlen(data_dir)-1,1) NE dir_sep THEN $
        data_dir=data_dir+dir_sep

       fstring = data_dir+date_str+'_hyd_ppa_v'

       if getversion ne '' then begin
          print, '% hydra_findfile: getting explicit version '+getversion
          fnstr = fstring+getversion+'.cdf'
          f= findfile( fnstr, count=count )
       endif else BEGIN
           fnstr = fstring+'?.??.cdf'
           f= findfile( fnstr, count=count )
       ENDELSE

       if (count ne 0) then begin
          f=f(count-1)
          if not keyword_set(silent) then print, myname+': found '+f+'...'
          n= strlen(f)
          version= float( strmid( f, n-8, 4 ))
          get_err_no=0
          dir= data_dir
          return, f
      endif else begin
          get_err_msg= myname+': Hydra PPA L1 file,!c'+fnstr+'!c'+ $
            'not found.!c'
          if not keyword_set(silent) then $
            print, myname+': Hydra PPA L1 file, '+fnstr+' not found.'
          get_err_no=1
          version=0
          return, ''
      ENDELSE
      
  ENDIF
  

    return, ''

 end


