function hpc_average, X, navg, nodata=nodata, naverage, noavg=noavg, $
                        allow_smooth=allow_smooth

; averages array over first dimension (having n elements) down to
; navg elements.

;    x1 x2 x3 x4 x5 x6 x7 x8        <-- original 1D (or 2D array), nX elements
;    \      / \   / \      /        <-- (nX/navg) or (nX/navg+1)
;     \    /   \ /   \    /              averaged together
;       a1     a2      a3           <-- averaged array, navg elements
;
; INPUT:  X      is the array to average
;         navg   is the number of average bins
;         nodata specifies nodata value
;         noavg  don't divide by number of points, just sum
;
;         allow_smooth  (nX/navg+1) values go into each average, so
;                       some values in source array to go into
;                       two averages.  This is somewhat faster.
;
; OUTPUT: naverage   passes out the number of values in average (optional)
;         (returns)  the average array, or if /noavg, the sums, sum(Xi),
;                      to be divided by naverage
;
; ASSUMPTIONS and NOTES:
;  For 2d arrays, ANY value equal to 1e-20 is also treated as nodata.
;  It is also assumed that 1e-20 << X(i)*nsum .
;  Occasionally this routine triggers floating underflow.
;
; HISTORY:
;  Spring 1997   Written, Jeremy Faden, jbf@hydra.physics.uiowa.edu

  rfill= 1e-20

   if n_elements(nodata) eq 0 then nodata= rfill
   allow_smooth= keyword_set(allow_smooth)

   sizeX= size(X)
   n= sizeX( 1 )                ; average over the first dimension
   nsum= float(n)/navg
   
   if (nsum eq 1.0) then begin  ; no averaging
       naverage= make_array( size=size(X), /int, value=1 )
       rnv= where( X eq nodata )
       if rnv(0) ne -1 then naverage(rnv)=0
       return, X
   endif 

   if (nsum lt 1.0 ) then begin ; use congrid to expand       
       print, '% hydra average: I don''t expand'
       return, X
   endif
   
   if ( sizeX(0) lt 1 or sizeX(0) gt 2) then begin
       print, '% hydra average: only 1- or 2-D arrays please...'
       return, -1
   endif else if sizeX(0) eq 1 then begin ; 1D average
       typeX= sizeX(2)
       saverage= make_array(navg,type=typeX,value=0)
       naverage= make_array(navg,/int)
   endif else begin             ; 2D average
       typeX= sizeX(3)
       ncol= sizeX(2)
       saverage= make_array(navg,ncol,type=typeX)
       naverage= make_array(navg,ncol,/int)
   endelse     
   
   if (allow_smooth) then begin
       print, '% hpc_average: smoothing allowed.'
       nsum1=ceil(nsum) 
   endif else nsum1=fix(nsum)-1

   ind= long(indgen(navg)*nsum)
   ind0= [ind,n]                ; initial indeces

   valid= ( X ne nodata )

   if sizeX(0) eq 1 then begin
       for i=0, nsum1 do begin
           saverage= saverage+X(ind)*valid(ind)
           naverage= naverage+valid(ind)
           ind=ind+1
       endfor

       if (not allow_smooth) then begin
           r= where( ind0(1:*)-ind(0:*) eq 1 ) 
           if r(0) ne -1 then begin
               ind= ind(r)
               saverage(r)=saverage(r)+X(r)*valid(r)
               naverage(r)=naverage(r)+valid(r)
           endif
       endif
   endif else begin
       if nodata ne rfill then begin
           rnodata=where(X eq nodata) 
           if rnodata(0) ne -1 then X(rnodata)=rfill
       endif
           
       for i=0, nsum1-1 do begin
           saverage= saverage+X(ind,*)*valid(ind,*)
           naverage=naverage+valid(ind,*)
           ind=ind+1    
       endfor

       if (not allow_smooth) then begin ; average in the remaining guys
           r= where( ind0(1:*)-ind(0:*) eq 1 ) 
           if r(0) ne -1 then begin
               ind= ind(r)
               rv= where(X(ind,*) ne rfill)
               saverage(r,*)= saverage(r,*)+X(ind,*)*valid(ind,*)
               naverage(r,*)=naverage(r,*)+valid(ind,*)
           endif
       endif

       if nodata ne rfill then if rnodata(0) ne -1 then X(rnodata)=nodata

   endelse

   r0= where(naverage eq 0)
   if (r0(0) ne -1) then saverage(r0)= nodata
   
   if keyword_set( noavg ) then return, saverage
   
   rv= where(naverage gt 0)
   if (rv(0) ne -1) then saverage(rv)=saverage(rv)/naverage(rv)
   
   return, saverage
end




pro hydra_plotcolor, zmat_in, $
                     xarr_in, xlog=xlog, $
                     yarr_in, ylog=ylog, $
                     dx=dx, dy=dy, $
                     dpi=dpi, $
                     ycenter=ycenter, xcenter=xcenter, $
                     resx=resx, resy=resy, $
                     zlog=zlog, zrange=zrange, $
                     _extra=e, special_colors=special, $
                     nodata=nodata, $
                     sample=sample, $
                     xfilled=xfilled, $ ; assume data in X is already filled.
                     ctable= ctable, $ ; color table range
                     overplot = overplot, $
                     diff_error2= diff_error, $ ; for making difference plots
                     min_value= min_value, $
                     max_value= max_value, $
                     draw_colorbar=colorbar, $
                     inset_colorbar= cbar_inset, $
                     cbar_xpos=cbar_xpos,$
                     cbar_title=zword, ztitle=ztitle, $                     
                     cbar_charsize=cbar_charsize, $
                     cbar_ticklen=cbar_ticklen, $
                     backcolor=backcolor, $
                     verbose=verbose
  
;+
; NAME: hydra_plotcolor
;
;
;
; PURPOSE: create a color image of a 2D array.
;
;
;
; CATEGORY: color plotting
;
;
;
; CALLING SEQUENCE: hydra_plotcolor, Z, X, Y
;
;
; 
; INPUTS: 
;     Z,  a 2D array (m,n) of values that are mapped to color space
;     X,  a 1D array of values that specify the beginning (along X
;       axis) of each column of Z.  If not specified, then findgen(m)
;       is used.  If a 1-D array is passed in, then /xcenter is
;       implicitly set.
;     Y,  a 1D array, to specify row positions.  If not specified,
;       then findgen(n) is used.  If a 1-D array is passed in, then /xcenter is
;       implicitly set.
;
;     if X is a 2D array (m,2) then each of the m pairs specify the
;       "right" and "left" sides of the bin. 
;     if Y is (n,2) then each pair specifies the bin "bottom" and "top." 
;     
;     if /xcenter specified, then the 1D array X are the bin centers,
;        and the bounds are found like they are if psym=10.
;     likewise for /ycenter.
;

; OPTIONAL INPUTS:
;  
; KEYWORD PARAMETERS:
;     xrange, yrange, /xlog, /ylog specify the x and Y axes.
;     zrange and /zlog specify the mapping from Z to color.
;     dx, dy       specify the x and y binsizes explicitly, the code need't
;                  calculate them
;     dpi          use this output resolution when averaging.  By
;                  default, 75dpi (CRT resolution) is used for all devices.
;
;     resx, resy   specify the required resolution in data coordinates,
;                  by default the device resolution is used.
;     special_colors=fltarr(2,m)  
;                  specify explicit mapping from Z value to color.  
;                  eg. special=[[0.,get_color_index('black')]]
;     nodata       value to be excluded from averages, and coded 
;                  grey on color image
;     /draw_colorbar draw colorbar next to plot
;     cbar_title   title for colorbar, printed above.
;     /inset_colorbar
;     min_value    data below this value is treated like fill data.
;     max_value    data above this value is treated like fill data.
;     /overplot    use existing plot coordinates to place the image.
;     ztitle       title for colorbar, printed on right hand side.
;     /sample      when pixel resolution is less then bin size,
;                    the data are averaged by default.  Sample results
;                    in Z values being sampled.
;     ctable       specify color table range in color index space.
;     backcolor=n  specify index of background color (default is 'grey')
;     /xcenter     specify that the x-values specify the bin centers.
;                    The bin boundries are set like they would for
;                    psym=10, using the midpoints as boundries.
;                    (Obsolete since a 1-D x array implicitly sets /xcenter)
;     /ycenter     works like /xcenter for the y-values.
;
; OUTPUTS:
;    none
;
;
; OPTIONAL OUTPUTS:
;    none
;
;
; COMMON BLOCKS:
;    none
;
;
; SIDE EFFECTS:
;    writes to plot device, affects plot coordinates as plot or
;      contour would.  !z.crange and !z.type (zlog) are set.
;
; RESTRICTIONS:
;    zlog, xlog, ylog: arrays assumed to be positive
;    !p.multi handled.
;    
;
; PROCEDURE:
;    plot of x,y is called with /nodata.
;    min and max of X and Y are located in plot coordinates.
;    greatest common divisor of the bin sizes is calculated, and the
;       gcd may be no smaller then xres and yres
;    data is so that all columns and rows fall into some integer 
;       number of evenly-sized bins, into the array "zimage"
;    if zimage x resolution is higher than pixel resolution, zimage x 
;       resolution is reduced to pixel resolution.
;    zimage is mapped to color index
;    tv is used to display image
;    plot axes are redrawn, in case tv image clobbered ticks.
;    colorbar is drawn.

; EXAMPLE:
;    X= findgen(30)-15 & Y=findgen(30)-15
;    Z= sqrt ( ( X#(Y*0+1) ) ^2 + ( (X*0+1)#Y ) ^2 )
;    hydra_plotcolor, Z, X, Y, zrange=[0,15], /xcenter, /ycenter, $
;       position=[15,10,80,95]/100., /colorbar
;    contour, z, x, y, /overplot, levels=indgen(8)*2
;
; MODIFICATION HISTORY:
;  Spring 1997   Written, Jeremy Faden, jbf@hydra.physics.uiowa.edu
;-

; Plots a colorplot of data.  Z data must be stored in a 2D array,
;    with columns correponding to intervals in x and rows corresponding
;    to intervals in y.
;
; INPUTS:
;
; zmat is the data, dblarr( n, m )
;
; xarr are the x values for each z column.
;    if xarr=fltarr(n,2) then pairs are start and end of column.
;    if xarr=fltarr(n)   then each is start of column with no gaps,
;                            and the last column is not shown.
; yarr are the y values for each z row
;    same format as xarr. fltarr(m,2) or fltarr(m)
;
; /ylog is handled, but /xlog will give incorrect results.
;
; dx, dy  explicitly specify the bin size for x and y, otherwise
;    the binsize is the greatest common divisor, within pixel
;    precision of the x and y values.
;
; /ycenter  "centers" the y bins on the yarr elements.  Bin partitions
;    are midway between adjacent elements.
; /xcenter  centers the x bins on the xarr elements.
; resx, resy  explicitly specify the required accuracy of the x and y
;    bins, otherwise the device pixel resolution sets these.
;
; nodata specifies that a point should be plotted in grey and omitted
;    from averaging.
;
; /sample  do not average zmat is device resolution is not sufficient,
;    instead display samples from zmat.
;
; special_colors = fltarr(2,n) can be used to insert special colors,
;                              e.g. saturation.  It is an array of pairs,
;                              each pair is (value,color).  Where
;                              value is found in zmat, use display color.
;
; ctable=[a,b]  specifies the range of colors to use for the colorbar. 
;    if not specified, then the following algorythm is used:
;        if n_elements(papco_color) gt 0 then use PAPCO color scheme.
;        otherwise, use [0,!d.n_colors]
;
; get_zimage  returns the binned array and does not plot it.
;
; min_value   disregard data lt min_value in autoranging, don't average it,
;             and plot as grey.  (ie. similar to nodata )
; max_value   same as min_value, data above max_value is invalid.
;
; Notes:
;    No averaging is done on columns (y).
;    Averaging will clobber special color values.
;
; HISTORY:
;    Written, Jeremy Faden, November 1997.  jbf@space-theory.physics.uiowa.edu

; set the windows device resolution

  COMMON papco_color, papco_color, r_papco, g_papco, b_papco 
  common hydra_setcolors1_common, hydra_colorsystem

  on_error, 2
  
  if getenv('win_px_cm') eq '' then begin
      case strlowcase(!version.os_family) of
          'unix': win_px_cm=40.
          'Windows': win_px_cm=37.9259
      endcase
  endif else begin
      win_px_cm= float( getenv( 'win_px_cm' ) )
  endelse
  
  if n_elements(dpi) eq 1 then begin
      target_px_cm= win_px_cm * ( dpi / 75. )
  endif else begin
      target_px_cm= win_px_cm 
  endelse
  
; check input correctness
  if n_elements(zmat_in) eq 0 then begin
      print, 'z matrix is undefined.'
      return
  end

  sz= size(zmat_in)
  sx= size(xarr_in)
  sy= size(yarr_in)
  if sx(0) eq 2 and sx(2) ne 2 then $
    message, 'error in X input array format', /cont
  if sx(0) ne 0 and sx(1) ne sz(1) then $
    message, "X and Z dimensions don't match", /cont
  if sy(0) eq 2 and sy(2) ne 2 then $
    message, 'error in Y input array format', /cont
  if sy(0) ne 0 and sy(1) ne sz(2) then $
    message, "Y and Z dimensions don't match", /cont
  if n_elements(Xarr_in) eq 1 then $
    message, "X is a scalar"
  if n_elements(yarr_in) eq 1 then $
    message, "Y is a scalar"

  if n_elements( xarr_in ) gt 0 then begin
      if min( xarr_in(1:*,0) - xarr_in(0:*,0) ) le 0. then $
        message, 'X is not monotonically increasing', /cont
  endif
  if n_elements( yarr_in ) gt 0 then begin
      if min( yarr_in(1:*,0) - yarr_in(0:*,0) ) le 0. then $
        message, 'Y is not monotonically increasing', /cont
  endif

; end--check input correctness.  We won't stop at this point, as 
;   is will inevitably catch some code the night before AGU.

  if n_elements(xarr_in) eq 0 then begin ; x and y not specified.
      m= n_elements(zmat_in(*,0))
      xarr_in= findgen(m)
      xarr_in= reform( [ xarr_in, xarr_in+1 ], m, 2)
  endif

  if n_elements(yarr_in) eq 0 then begin ; x and y not specified.
      n= n_elements(zmat_in(0,*))
      yarr_in= findgen(n)
      yarr_in= reform( [ yarr_in,yarr_in+1 ], n, 2)
  endif

; copy inputs to local variables
  zmat= double(zmat_in)
  xarr= double(xarr_in)
  yarr= double(yarr_in)

  if n_elements( nodata ) eq 0 then nodata=5.43e21
  r= where( finite(zmat) eq 0 )
  if r(0) ne -1 then zmat(r)=nodata

  if n_elements( min_value ) gt 0 then begin
      r= where( zmat lt min_value )
      if r(0) ne -1 then zmat(r)=nodata
  endif

  if n_elements( max_value ) gt 0 then begin
      r= where( zmat gt max_value )
      if r(0) ne -1 then zmat(r)=nodata
  endif

  if n_elements( zlog ) eq 0 then begin
      zlog= 0
  end

  if n_elements(zrange) eq 0 then begin
      if total( abs( !z.range ) ) gt 0. then begin
          zrange= !z.range
      endif else begin
          r= where( zmat ne nodata )
          if r(0) ne -1 then begin
              zmin=min(zmat(r),max=zmax)
              zrange=[zmin,zmax]
          endif else begin
              if zlog then begin 
                  zrange=[1,10.]
              endif else begin
                  zrange=[0.,1.]
              endelse
          endelse
      endelse
  endif
  zmin= double(zrange(0))
  zmax= double(zrange(1))

  xfilled= keyword_set( xfilled )
  if xfilled then begin
      dx= xarr(1)-xarr(0)
  endif

  sample= keyword_set(sample)   ; sample data, don't average

  diff= n_elements( diff_error ) gt 0 ; difference plot mode
  if diff then begin
      zmat_err= diff_error
  endif

; find colorbar range
  if n_elements(ctable) eq 0 then begin
      if n_elements( papco_color ) gt 0 then begin
          COMMON papco_color_names, black, red, green, yellow, $
            blue, magenta, cyan, $
            white, burgundy, olive, dark_green, teal, $
            royal_blue, violet, dark_grey, grey, $
            foreground, background
          cindex= where( papco_color(*,2) eq 1 ) ; search the active vector
          if (cindex(0) eq -1) then begin
              print, 'Color system failure, '
              print, 'consult jbf@space-theory.physics.uiowa.edu'
              print, 'Using color set 0'
              cindex=0
          endif
          forward_function papco_get_colorindeces
          color_range= PAPCO_get_Colorindices( cindex )
          !p.color= 1
          !p.background= 0
      endif else if n_elements( hydra_colorsystem ) gt 0 then begin
          color_range= hydra_colorsystem.colortable
          grey= hydra_colorsystem.grey
          !p.color= hydra_colorsystem.foreground
          !p.background= hydra_colorsystem.background
      endif else begin
          color_range=[0,!d.n_colors-1]
          grey=0
      endelse
  endif else begin
      color_range= ctable
      grey= get_color_index('grey')
  endelse
  colorbase= color_range(0)
  ncolors= color_range(1)-color_range(0)
  colortop= color_range(1)

; handle backcolor keyword
  if n_elements( backcolor ) eq 0 then backcolor= grey
  
  ylog= keyword_set( ylog )
  
  xlog= keyword_set( xlog )

; handle /xlog keyword.
  if keyword_set( xlog ) then begin
      xarr=alog10(xarr)
  endif

; reformat xarr to start, end time format. (if necessary.)
  s= size(xarr)
  if s(0) eq 1 then begin
      avg= ( xarr(1:*)+xarr(0:*) ) / 2
      a1= xarr(0) - ( avg(0)-xarr(0) )
      n= n_elements( xarr )-1
      an= xarr(n) + ( xarr(n)-avg(n-1) )
      xarr_new= make_array( s(1),2 )
      xarr_new(*,0)= [a1,avg]
      xarr_new(*,1)= [avg,an]
      xarr= xarr_new
  endif

; handle /ylog keyword.
  if keyword_set( ylog ) then begin
      yarr=alog10(yarr)
  endif

; reformat yarr to start, end time format. (if necessary.)
  s= size(yarr)
  if s(0) eq 1 then begin
      avg= ( yarr(1:*)+yarr(0:*) ) / 2
      a1= yarr(0) - ( avg(0)-yarr(0) )
      n= n_elements( yarr )-1
      an= yarr(n) + ( yarr(n)-avg(n-1) )
      yarr_new= make_array( s(1),2 )
      yarr_new(*,0)= [a1,avg]
      yarr_new(*,1)= [avg,an]
      yarr= yarr_new
  endif

  q=n_elements(xarr_in)-1 & r=n_elements(yarr_in)-1
  if not keyword_set( overplot ) then begin
      plot, xarr_in([0,q]), yarr_in([0,r]), $
        /nodata, _extra=e, ylog=ylog, xlog=xlog
      !p.multi(0)= !p.multi(0)+1 
  endif
  xcrange= !x.crange
  ycrange= !y.crange

; calculate pixel resolution 
  if ylog then ycrange10=10^ycrange else ycrange10=ycrange
  if xlog then xcrange10=10^xcrange else xcrange10=xcrange

  lld= convert_coord( xcrange10(0), ycrange10(0), /data, /to_device )
  urd= convert_coord( xcrange10(1), ycrange10(1), /data, /to_device )
  if !d.name eq 'Z' then begin
      xsized= ( long(urd(0)-lld(0)) )
      ysized= ( long(urd(1)-lld(1)) ) 
  endif else begin
      xsized= ( long(urd(0)-lld(0)) ) * win_px_cm / !d.x_px_cm ; normalize
      ysized= ( long(urd(1)-lld(1)) ) * win_px_cm / !d.y_px_cm ;  resolution
  endelse

  if n_elements( resx ) eq 0 then $
    resx= ( xcrange(1) - xcrange(0) ) / xsized
  if n_elements( resy ) eq 0 then $
    resy= ( ycrange(1) - ycrange(0) ) / ysized

; calculate bin size dx and dy
  if not keyword_set(dx) then begin
      dxs= xarr(*,1)-xarr(*,0)
      if alog10( max( abs(dxs/dxs(0)) ) ) gt 2 or $
        alog10( min( abs(dxs/dxs(0)) ) ) lt -2 then begin
          message, 'x bin sizes vary by more than two orders of magnitude.', /c
          message, 'try /xlog.'
      endif
      dx= gcd( dxs, errtot=resx )
  endif

  if not keyword_set(dy) then begin
      dys= yarr(*,1)-yarr(*,0)
      if alog10( max( abs(dys/dys(0)) ) ) gt 2 or $
        alog10( min( abs(dys/dys(0)) ) ) lt -2 then begin
          message, 'y bin sizes vary by more than three orders of magnitude.', /c
          message, 'try /ylog.'
      endif
      dy= gcd( dys, errtot=resy )
  endif

  if keyword_set( verbose ) then begin
      message, 'dx='+strtrim(dx), /cont
      message, 'dy='+strtrim(dy), /cont
  endif

; find which xbins appear on the plot
  rplotx= where( xarr(*,0) ge xcrange(0) and xarr(*,1) le xcrange(1), nplotx )
  if nplotx lt n_elements( xarr(*,0) ) then begin
      if nplotx gt 0 then begin
          xarr=xarr(rplotx,*)
          zmat=zmat(rplotx,*)
          if diff then zmat_err= zmat_err(rplotx,*)
      endif else begin
          xarr=xarr(0,*)        ; we'll see that nothing is printed later
          zmat=zmat(0,*)
          if diff then zmat_err= zmat_err(0,*)
      endelse
  endif

  rploty= where( yarr(*,0) ge ycrange(0) and yarr(*,1) le ycrange(1), nploty )
  if nploty lt n_elements( yarr(*,0) ) then begin
      if nploty gt 0 then begin
          yarr=yarr(rploty,*)
          zmat=zmat(*,rploty)
          if diff then zmat_err= zmat_err(*,rploty)
      endif else begin
          yarr=yarr(0,*)        ; we'll see that nothing is printed later
          zmat=zmat(*,0)
          if diff then zmat_err= zmat_err(*,rploty)
      endelse
  endif

; calculate the number of x and y bins for image
;   and form zimage -> zmat map
  if (xfilled) then begin
      nx= n_elements( xarr(*,0) )
      wherex1= lindgen( nx )
      wherex2= lindgen( nx )
  endif else begin
      nx= long( ( max(xarr) - min(xarr) ) / dx + 0.5 ) 
      wherex1= long((xarr(*,0)-xarr(0))/dx+0.5)
      wherex2= (long((xarr(*,1)-xarr(0))/dx+0.5)-1)>wherex1
  endelse

  ny= long( ( max(yarr) - min(yarr) ) / dy + 0.5 )
  wherey1 = long((yarr(*,0)-yarr(0))/dy+0.5)
  wherey2 = (long((yarr(*,1)-yarr(0))/dy+0.5)-1)>wherey1

  wherex= lonarr(nx)-1
  wherey= lonarr(ny)-1
  for i=0L,n_elements(wherex1)-1 do $
    wherex((wherex1(i)<(nx-1)):(wherex2(i)<(nx-1)))= i
  for i=0L,n_elements(wherey1)-1 do $
    wherey((wherey1(i)<(ny-1)):(wherey2(i)<(ny-1)))= i

  zimage= make_array( nx, ny, /float, value=nodata )
  if diff then zimage_err= make_array( nx, ny, /float, value=nodata )

  rx= where( wherex ge 0 )
  ry= where( wherey ge 0 )
  for j=0L,n_elements(ry)-1 do begin
      zimage(rx,ry(j))= zmat(wherex(rx),wherey(ry(j)))
      if diff then zimage_err(rx,ry(j))= zmat_err(wherex(rx),wherey(ry(j)))
  endfor

; find the location of tv image of zimage on data coords.
  if xlog then begin
      x1=10^xarr(0) 
      x2=10^xarr( n_elements(xarr)-1 )
  endif else begin
      x1=xarr(0)
      x2=xarr( n_elements(xarr)-1 )
  endelse
 
  if ylog then begin
      y1=10^yarr(0) 
      y2=10^yarr( n_elements(yarr)-1 )
  endif else begin
      y1=yarr(0)
      y2=yarr( n_elements(yarr)-1 )
  endelse

  lld= convert_coord( x1, y1, /data, /to_device )
  urd= convert_coord( x2, y2, /data, /to_device )

; find device size
  if !d.name eq 'Z' then begin
      xsized= ( long(urd(0)-lld(0)) ) 
      ysized= ( long(urd(1)-lld(1)) ) 
  endif else begin
      xsized= ( long(urd(0)-lld(0)) ) * win_px_cm / !d.x_px_cm ; normalize
      ysized= ( long(urd(1)-lld(1)) ) * win_px_cm / !d.y_px_cm ; resolution
  endelse

; check device size
  dd1= (urd-lld)
  if dd1(0) le 0 or dd1(1) le 0 then begin
      message, 'Insufficient room for plot.  Try reducing charsize,', /cont
      message, '!p.multi, or margins.'
  endif
  
; find average array target size
  if !d.name eq 'Z' then begin
      avg_xsize= long( ( long(urd(0)-lld(0)) ) * target_px_cm / !d.x_px_cm )
      avg_ysize= long( ( long(urd(1)-lld(1)) ) * target_px_cm / !d.y_px_cm )
  endif else begin
      avg_xsize= long( ( long(urd(0)-lld(0)) ) * target_px_cm / !d.x_px_cm )
      avg_ysize= long( ( long(urd(1)-lld(1)) ) * target_px_cm / !d.y_px_cm )
  endelse

; average down the array if there aren't enough pixels to show each 
; measurement.
  if (avg_xsize lt nx) then begin
      if (sample) then begin
          print, '% hydra_plot_clr: sampling data'
          rrr= long( findgen( avg_xsize ) / avg_xsize * nx )
          zimage= zimage( rrr, * )
          if diff then zimage_err= zimage_err( rrr, * )
      endif else begin
          print, '% hydra_plot_clr: x-averaging '+strtrim(nx,2)+$
            ' time intervals down to '+strtrim(avg_xsize,2)+'.'    
          zimage= hpc_average( zimage, avg_xsize, /allow, nodata=nodata )
          if diff then begin    ; propogate errors though average
              if r(0) ne -1 then zimage_err(r)=nodata
              zimage_err= hpc_average( zimage_err, avg_xsize, $
                                       /allow, naverage, /noavg, $ 
                                       nodata=nodata )
              r= where( zimage_err ne nodata )
              if r(0) ne -1 then zimage_err(r)= $
                sqrt(zimage_err(r)) / naverage(r)
          endif
      endelse
      nx= avg_xsize
  endif

  if diff then begin
      r= where( zimage_err ne nodata )
      if r(0) ne -1 then zimage(r)= zimage(r) / sqrt( zimage_err(r) )
      rnv= where( zimage_err eq nodata )
      if rnv(0) ne -1 then zimage(rnv)= nodata

;   transform the colorbar into linear bipolar
      r_neg = where(zimage lt 0.0)
      if (r_neg(0) ne -1) then zimage(r_neg) = -zimage(r_neg)
      r_nodata = where(zimage eq nodata)
      zimage= zimage/zmin
      zmax= zmax/zmin
      zmin= zmin/zmin           ; aka 1
;      take log
      r_small = where(zimage lt 1.0)
      if (r_small(0) ne -1) then zimage(r_small) = 1.0
      zimage= alog10(zimage)
;      add sign
      if (r_neg(0) ne -1) then zimage(r_neg) = -zimage(r_neg)
      if (r_nodata(0) ne -1) then zimage(r_nodata) = nodata

      zmin= -alog10( zmax )
      zmax= alog10( zmax )
      zlog=0
  endif

; transform data to color
  if (zlog) then begin
      zsize= alog10( zmax/zmin )
      zcolor= make_array( nx, ny, /byte, value=colorbase )
      rok= where( zimage ge zmin )
      if rok(0) ne -1 then zcolor(rok)= $
        byte( ( alog10( zimage(rok)/zmin ) / $
                ( zsize / ncolors ) + colorbase + 0.5 ) < colortop > colorbase )
  endif else begin
      zsize= zmax-zmin
      zcolor= make_array( nx, ny, /byte, value=colorbase )
      rcol= where( zimage ge zmin )
      if rcol(0) ne -1 then zcolor(rcol)= $
        byte( ( ( zimage(rcol) - zmin ) / $
                ( zsize / ncolors ) + colorbase + 0.5 ) < colortop > colorbase )
      r= check_math()
      if r ne 0 then message, 'check_math()='+strtrim(r,2), /cont
  endelse

; fill color nodata points grey
  rnd=where( zimage eq nodata )
  if rnd(0) ne -1 then zcolor( rnd ) = backcolor

; fill special color points with special colors
  if n_elements(special) gt 0 then begin
      for i=0L,n_elements(special(0,*))-1 do begin
          value= special(0,i)
          color= byte(special(1,i))
          rval= where( zimage eq value )
          if rval(0) ne -1 then zcolor(rval)=color
      endfor
  endif

  if diff then begin
      r= where( zimage eq 0. )
      if r(0) ne -1 then zcolor(r)= black
  endif

  if not keyword_set( overplot ) then begin
      polyfill, (xcrange10)[[0,1,1,0,0]], (ycrange10)[[1,1,0,0,1]], color=backcolor
  endif
  
  if (nplotx gt 0) then begin
      if !d.name eq 'PS' then begin
          lln= convert_coord( lld, /device, /to_normal )
          urn= convert_coord( urd, /device, /to_normal )
          xsizen= urn(0)-lln(0)
          ysizen= urn(1)-lln(1)
          hyd_tv, zcolor, lln(0), lln(1), xsize=xsizen, ysize=ysizen, /normal
      endif else begin
          hyd_tv, congrid(zcolor, xsized, ysized), lld(0), lld(1)+1 ; +1 is a kludge
      endelse
  endif

  q=n_elements(xarr_in)-1 & r=n_elements(yarr_in)-1

  if not keyword_set(overplot) then begin
      plot, xarr_in([0,q]), yarr_in([0,r]), $
        /noerase, /nodata, _extra=e, ylog=ylog, xlog=xlog
      !p.multi(0)= ( !p.multi(0)-1 ) mod ((!p.multi(1)*!p.multi(2))>1)
  endif
  
  if n_elements( cbar_charsize ) eq 0 then cbar_charsize=1.0
  if n_elements( cbar_ticklen ) eq 0 then cbar_ticklen=0.03

  if keyword_set( colorbar ) then begin
      hydra_colorbar, zrange=[zmin,zmax], zlog=zlog, $
        xpos= cbar_xpos, zword= zword, ztitle= ztitle, $
        charsize= cbar_charsize, ticklen=cbar_ticklen, $
        ctable= [colorbase, colortop ]    
  endif

  if n_elements( cbar_inset ) then begin
      hydra_colorbar, zrange=[zmin,zmax], zlog=zlog, $
        inset= cbar_inset, zword= zword, ztitle= ztitle, $
        ctable=[colorbase,colortop], $
        charsize=cbar_charsize, ticklen=cbar_ticklen
  endif

  if zlog then begin
      !z.crange= alog10( [ zmin, zmax ] )
  endif else begin
      !z.crange=[ zmin, zmax ]
  endelse

  return

end
