function hydra_tricolor_get_area, x, y, T
  mag_cross=  ( x(T(1,*))-x(T(0,*)) ) * ( y(T(2,*))-y(T(0,*)) )  -  $
    ( y(T(1,*))-y(T(0,*)) ) * ( x(T(2,*))-x(T(0,*)) ) 
  return, mag_cross/2.
end

pro hydra_tricolor, x_in, y_in, z_in, $
                    xlog=xlog, ylog=ylog, $
                    zrange=zrange_in, zlog=zlog, _extra=e, $
                    plot_triangles=plot_triangles, $
                    ctable=ctable, $
                    overplot=overplot, $
                    favor_vertical=fvert, $
                    favor_horizontal=fhorz, $
                    triangles=T, $
                    out_triangles=T_out, $
                    node_reject=node_reject, $
                    reject_large_triangles=reject_large_triangles, $
                    draw_colorbar=draw_colorbar, $
                    inset_colorbar= cbar_inset, $
                    cbar_xpos=cbar_xpos, $ 
                    cbar_charsize=cbar_charsize, $
                    cbar_ticklen=cbar_ticklen, $
                    cbar_title=zword, ztitle=ztitle, $
                    min_value=min_value, max_value=max_value, $
                    fill=rfill, $
                    backcolor=backcolor

;+
; NAME: hydra_tricolor
;
;
;
; PURPOSE: draw colorplot from buckshot data.  
;
;
;
; CATEGORY: colorplotting
;
;
;
; CALLING SEQUENCE: hydra_tricolor, X, Y, Z
;
;
; 
; INPUTS: X, Y, Z(X,Y)  each 1D arrays
;
;
;
; OPTIONAL INPUTS:
;
;
;	
; KEYWORD PARAMETERS:
;    node_reject=fltarr(2,n)    reject triangles that connect to this
;                               node.  Useful for removing concave regions.
;    triangles=lonarr(3,n)      use these triangles.  Otherwise
;                               triangulate is used.
;    out_triangles=lonarr(3,n)  (output) returns the triangles used.
;    /plot_triangles            overplot triangles
;    favor_horizontal=fh        favor horizontal triangles 
;    favor_vertical=fv          favor vertical triangles 
;    fill=f                     treat these points as invalid.
;    <hydra_colorbar keywords>
;    <IDL plot keywords>
;
; OUTPUTS:
;    To device.
;
;
; PROCEDURE:
;    Call plot to set plot coordinates.  Triangulate points within
; plot boundries.  Reject triangles based on node_reject.  Trigrid over
; plot boundries.  Transform to color image.  TV the color image.
;
; EXAMPLE:
;   x= randomn(s,1000)
;   y= randomn(s,1000)
;   z= atan(y,x)
;   ex_x= findgen(60)/59 * (-6)
;   ex_y= ex_x * 0.
;   hydra_tricolor, x, y, z, zrange=[-!pi,!pi], /inset_colorbar, $
;     node_reject= [ transpose(ex_x),transpose(ex_y) ]
;  
; MODIFICATION HISTORY:
;
;-

; pop out to the calling code should an error occur.
on_error, 2

; set the windows device resolution
case strlowcase(!version.os_family) of
    'unix': win_px_cm=40.
    'windows': win_px_cm=37.9259
    else: win_px_cm= 40.
endcase


  x= x_in
  y= y_in
  z= z_in

  if n_elements( rfill ) eq 0 then rfill=-1e31

  r= where( finite(z) eq 0 ) 
  if r(0) ne -1 then begin
      z(r)=rfill
  endif

  if n_elements( node_reject ) gt 0 then begin
      n= n_elements(node_reject(0,*))
      index_reject= lindgen(n) + n_elements(x)
      x= [ x, reform( node_reject(0,*) ) ]
      y= [ y, reform( node_reject(1,*) ) ]
      z= [ z, fltarr(n) ]
  endif

  overplot= keyword_set( overplot )

  if n_elements( backcolor ) eq 0 then backcolor=get_color_index('grey')

  if n_elements( fhorz ) gt 0 then fvert= 1/fhorz
  if n_elements( fvert ) eq 0 then fvert= 1.
  
;  set the zrange
  if n_elements( zrange_in ) eq 0 then begin
      if n_elements( min_value ) ne 0 then r= where( z ge min_value ) else $
        r= lindgen( n_elements(z) )
      if n_elements( max_value ) ne 0 then begin
          r1= where( z(r) le max_value ) 
          if r1(0) eq -1 then r=-1 else r= r(r1)
      endif
      r1= where( z(r) ne rfill )
      if r1(0) eq -1 then r=-1 else r=r(r1)
      if r(0) ne -1 then zrange_in=[min(z(r),max=ma),ma] else zrange_in=[0,1]
  endif
  zrange= zrange_in
  
  zlog= keyword_set(zlog)
  
  if keyword_set(zlog) then begin
      z= alog10( z>zrange(0) )
      zrange= alog10( zrange )
  endif
  
  if keyword_set(ylog) then $
    y_tri= alog10( y ) else $
    y_tri= y

  if keyword_set(xlog) then $
    x_tri= alog10( x ) else $
    x_tri= x

  if n_elements(min_value) ne 0 then begin
      r= where( z ge min_value ) 
      if r(0) ne -1 then begin
          x_tri=x_tri(r)
          y_tri=y_tri(r)
          z= z(r)
      endif else begin
          message, 'min_value filters all points', /cont
      endelse
  endif
  
  if n_elements(max_value) ne 0 then begin
      r= where( z le max_value ) 
      if r(0) ne -1 then begin
          x_tri=x_tri(r)
          y_tri=y_tri(r)
          z= z(r)
      endif else begin
          message, 'min_value filters all points', /cont          
      endelse
  endif

  if n_elements(T) eq 0 then triangulate, x_tri, y_tri/fvert, T

  for i=0,n_elements(index_reject)-1 do begin
      keep= intarr(n_elements(T(0,*)))+1
      r= ( where( T eq index_reject(i) ) / 3 )
      keep(r)=0
      r= where(keep)
      if r(0) ne -1 then begin
          T=T(*,r)
      endif else begin
          message, 'node reject filters all triangles', /cont
          return
      endelse
  endfor

  r= where( z(T(0,*)) ne rfill and $
            z(T(1,*)) ne rfill and $
            z(T(2,*)) ne rfill )
  if r(0) ne -1 then begin
      T=T(*,r)
  endif else begin
      message, 'fill data filters out all triangles'
  endelse

  if n_elements( reject_large_triangles ) gt 0 then begin
      area= hydra_tricolor_get_area( x_tri, y_tri/fvert, T )
      r= where( area lt reject_large_triangles )
      if r(0) ne -1 then begin
          T=T(*,r)
      endif else begin
          message, 'reject_large_triangles filters all triangles', /cont
          return
      endelse
  endif

  T_out= T

  if not overplot then begin
      plot, x, y, /nodata, _extra=e, $
        xlog= keyword_set(xlog), ylog=keyword_set(ylog)
      !p.multi= !p.multi+[1,0,0,0,0]
  endif
  
  limits= ([!x.crange,!y.crange])([0,2,1,3])

; get the size of a pixel
  dx=  1. / ( !d.x_size * win_px_cm / !d.x_px_cm * !x.s(1) )
  dy=  1. / ( !d.y_size * win_px_cm / !d.y_px_cm * !y.s(1) )
  
  gs= [ dx,dy ]
  
  zimage= trigrid( $
                   x_tri, $
                   y_tri, z, $
                   T, gs, limits, $
                   missing=1e31 )
  rnv= where( zimage eq 1e31 )
  
  if n_elements( ctable ) eq 0 then begin
      hydra_setcolors, cs
      colortable= cs.colortable
  endif else begin
      colortable= ctable
  endelse
  ncol= colortable(1)-colortable(0)+1

  image= bytscl( zimage, $
                 top=ncol-1, $
                 min=zrange(0), max=zrange(1) ) + colortable(0)
  if rnv(0) ne -1 then image(rnv)= backcolor
  
  xsize= (!x.crange(1)-!x.crange(0))*!x.s(1)
  ysize= (!y.crange(1)-!y.crange(0))*!y.s(1)

  
  if overplot and !d.name eq 'ps' then begin
      message, 'overplot not implemented for postscript device.', /cont
      plot, [0,1,0,1],[0,0,1,1]
      return
  endif
  
  if overplot then begin
      d= tvrd()
      lln= [ !x.crange(0)*!x.s(1)+!x.s(0), !y.crange(0)*!y.s(1)+!y.s(0) ]
      urn= [ !x.crange(1)*!x.s(1)+!x.s(0), !y.crange(1)*!y.s(1)+!y.s(0) ]
      ll= convert_coord( lln, /norm, /to_device )
      ur= convert_coord( urn, /norm, /to_device )      
      oldtv= d(ll(0):(ur(0)),ll(1):(ur(1)))
      r= where( zimage eq 1e31 )
      if r(0) ne -1 then image=oldtv(image)
  endif

  hyd_tv, image, !x.crange(0)*!x.s(1)+!x.s(0), $
    !y.crange(0)*!y.s(1)+!y.s(0), xsize=xsize, ysize=ysize, /norm
  
  if keyword_set( plot_triangles ) then begin
      plot_triangles, x, y, T, /overplot
  endif

  
  pold=!p
  plot, x, y, xlog=xlog, ylog=ylog, _extra=e, $
    /noerase, /nodata
  !p=pold  
  !p.multi=!p.multi-[1,0,0,0,0]

  !z.crange= zrange
  
  if zlog then zrange=10^zrange

  if n_elements( cbar_charsize ) eq 0 then cbar_charsize=1.0
  if n_elements( cbar_ticklen ) eq 0 then cbar_ticklen=0.03
  
  if keyword_set( draw_colorbar ) then begin
      hydra_colorbar, zrange=zrange, zlog=zlog, $
        xpos= cbar_xpos, zword= zword, ztitle= ztitle, $
        charsize= cbar_charsize, ticklen=cbar_ticklen, $
        ctable=colortable
  endif
  if n_elements( cbar_inset ) then begin
      hydra_colorbar, zrange=zrange, zlog=zlog, $
        inset= cbar_inset, zword= zword, ztitle= ztitle, $
        charsize=cbar_charsize, ticklen=cbar_ticklen, $
        ctable=colortable
  endif
end

